/* Copyright (c) 2018 VMware, Inc. All rights reserved. */

module platform {

   import ILogService = angular.ILogService;

   /**
    * This service provides APIs for plugin themes.
    */
   export class H5SdkThemeService {
      $inject = [
            "themeService",
            "$log"
      ];

      private readonly DEFAULT_CLIENT_THEME_NAME: string;
      private readonly CLIENT_THEME_NAME_TO_PLUGIN_THEME_NAME: { [ clientThemeName: string ]: PluginThemeName };
      private themeChangedCallbacks: ((PluginTheme) => void)[] = [];

      constructor(private themeService: any,
            private $log: ILogService) {

         this.DEFAULT_CLIENT_THEME_NAME = themeService.getKeys().light;
         this.CLIENT_THEME_NAME_TO_PLUGIN_THEME_NAME = { };
         this.CLIENT_THEME_NAME_TO_PLUGIN_THEME_NAME[themeService.getKeys().light] = PluginThemeName.LIGHT;
         this.CLIENT_THEME_NAME_TO_PLUGIN_THEME_NAME[themeService.getKeys().dark] = PluginThemeName.DARK;

         themeService.subscribeToThemeChanged(() => {
            let pluginTheme: PluginTheme = this.getPluginTheme();

            for (let i: number = 0; i < this.themeChangedCallbacks.length; i++) {
               try {
                  this.themeChangedCallbacks[i](pluginTheme);
               } catch (e) {
                  $log.error(`Error while invoking themeChangedCallback with index ${i}!`, e);
               }
            }
         });
      }

      /**
       * Returns information about the theme plugins should use.
       * @returns a PluginTheme object.
       */
      public getPluginTheme(): PluginTheme {
         let currentClientThemeName: string = this.themeService.getCurrentTheme();
         if (!this.CLIENT_THEME_NAME_TO_PLUGIN_THEME_NAME[currentClientThemeName]) {
            currentClientThemeName = this.DEFAULT_CLIENT_THEME_NAME;
         }

         let pluginThemeName: PluginThemeName =
               this.CLIENT_THEME_NAME_TO_PLUGIN_THEME_NAME[currentClientThemeName];

         return {
            name: pluginThemeName
         };
      }

      /**
       * Subscribe for notifications when the theme plugins should use changes.
       *
       * @param callback to invoke when the theme changes.
       * The callback has the following signature:
       * <code>function(theme: PluginTheme): void</code>
       */
      public subscribeToThemeChanged(callback: (theme: PluginTheme) => void): void {
         if (this.themeChangedCallbacks.indexOf(callback) !== -1) {
            return;
         }

         this.themeChangedCallbacks.push(callback);
      }

      /**
       * Unsubscribe for notifications when the theme plugins should use changes.
       *
       * @param callback that was used when subscribing (through the
       * <code>subscribeToThemeChanged</code> method) for theme changes.
       */
      public unsubscribeFromThemeChanged(callback: (theme: PluginTheme) => void): void {
         let callbackIndex = this.themeChangedCallbacks.indexOf(callback);
         if (callbackIndex === -1) {
            return;
         }

         this.themeChangedCallbacks.splice(callbackIndex, 1);
      }
   }

   /**
    * Describes a theme that can be used by a plugin
    */
   export interface PluginTheme {
      /**
       * The name of the theme. Possible values are "light" and "dark"
       */
      name: PluginThemeName;
   }

   /**
    * Enumerates all possible plugin theme names.
    * Current values are "light" and "dark".
    */
   //amarinov: https://stackoverflow.com/questions/15490560/create-an-enum-with-string-values-in-typescript
   export enum PluginThemeName {
      LIGHT = <any> 'light',
      DARK = <any> 'dark'
   }

   angular
         .module("com.vmware.platform.ui")
         .service("h5SdkThemeService", H5SdkThemeService);
}