/* Copyright (c) 2017 VMware, Inc. All rights reserved. -- VMware Confidential */
module platform {

   import IPromise = angular.IPromise;
   import IQService = angular.IQService;
   import IHttpService = angular.IHttpService;

   /**
    * This service class is used to create objects of type H5SdkApplicationApi.
    * The service also takes care of the needed dependencies.
    */
   export class H5SdkApplicationService {

      private _standardExtensionIdToSharedExtensionIdMapPromise: IPromise | undefined;

      public static $inject = [
         "navigation",
         "i18nService",
         "pluginUrlService",
         "iframeNavigationDataCache",
         "vxZoneService",
         "h5SdkThemeService",
         "$q",
         "$http"
      ];

      constructor(private navigation: Navigation,
            private i18nService: any,
            private pluginUrlService: PluginUrlService,
            private iframeNavigationDataCache: IframeNavigationDataCache,
            private vxZoneService: any,
            private h5SdkThemeService: H5SdkThemeService,
            private $q: IQService,
            private $http: IHttpService) {
      }

      public createH5SdkApplicationApi(pluginIframeScope: PluginIframeScope,
            locale: string): H5SdkApplicationApi {
         return new H5SdkApplicationApi(
               pluginIframeScope,
               this,
               this.i18nService,
               this.pluginUrlService,
               this.iframeNavigationDataCache,
               this.h5SdkThemeService,
               locale
         );
      }

      public navigateTo(options: NavigationOptions,
            remotePluginExtensionContext?: RemotePluginSingleInstanceExtensionContext): void {
         if (options.targetViewId && remotePluginExtensionContext) {
            options.targetViewId =
                  remotePluginExtensionContext.pluginNavigableExtensionIdsPrefix +
                  options.targetViewId;
         }

         if (options.customData && options.targetViewId &&
               // exclude the summary (portlets) view
               options.targetViewId.search(
                     "^vsphere\.core\..*\.summary$") === -1) {
            this.iframeNavigationDataCache.setNavigationData({
               viewId: options.targetViewId,
               customData: options.customData
            });
         }

         if (typeof options.objectId === "string" && options.objectId.trim().length === 0) {
            throw new Error(
                  "Navigation failed: objectId cannot be an empty string");
         }

         this.vxZoneService.runInsideAngular(() => {
            let navigationContext: any = {
               /*
                * If we are executing navigation to an already visible
                * plugin view we need to reload it to ensure that if the
                * navigation data has changed the view will re-read it.
                * To do this we put the "sdkReloadPluginExtensions" parameter to
                * cause the route to change which will trigger the
                * "viewDidUpdate" for  the active pluginIframe view if any.
                * We are using this long descriptive parameter name to avoid
                * collisions with other parameters that are used for refreshing
                * as this might have undesirable side effects. An example of
                * such parameter is "sdkUpdateRpxcSelector", part of
                * RemotePluginExtensionContextSelectorController.
                *
                * Bug:
                * See https://jira.eng.vmware.com/browse/VUSP-2156 or
                * https://bugzilla.eng.vmware.com/show_bug.cgi?id=2273357
                */
               sdkReloadPluginExtensions: Math.random()
            };

            if (remotePluginExtensionContext && !options.objectId) {
               navigationContext.rpxcId =
                     remotePluginExtensionContext.contextId;
            }

            this.getActualTargetViewId(options.targetViewId, remotePluginExtensionContext)
                  .then((actualTargetViewId: string | undefined) => {
                     if (actualTargetViewId && options.objectId) {
                        this.navigation.navigateToViewAndObject(
                              actualTargetViewId, options.objectId,
                              navigationContext
                        );
                     } else if (actualTargetViewId && !options.objectId) {
                        this.navigation.navigateToView(
                              actualTargetViewId, navigationContext
                        );
                     } else if (!actualTargetViewId && options.objectId) {
                        this.navigation.navigateToObject(
                              options.objectId, navigationContext
                        );
                     } else {
                        throw new Error("Navigation failed: At least one of" +
                              " targetViewId and objectId parameters must be specified"
                        );
                     }
                  });
         });
      }

      private getActualTargetViewId(
            targetViewId?: string,
            remotePluginExtensionContext?: RemotePluginSingleInstanceExtensionContext
      ): IPromise<string | undefined> {

         if (!targetViewId || !remotePluginExtensionContext) {
            return this.$q.when(targetViewId);
         }

         return this.getStandardExtensionIdToSharedExtensionIdMap()
               .then((map: { [key: string]: string }) => {
                  if (map[targetViewId]) {
                     return map[targetViewId];
                  }

                  return targetViewId;
               });
      }

      private getStandardExtensionIdToSharedExtensionIdMap(): IPromise<{ [key: string]: string }> {
         if (!this._standardExtensionIdToSharedExtensionIdMapPromise) {
            this._standardExtensionIdToSharedExtensionIdMapPromise = this.$http
                  .get("/ui/plugins/standardExtensionIdToSharedExtensionIdMap")
                  .then((response: any) => {
                     return response.data;
                  }, (error: any) => {
                     this._standardExtensionIdToSharedExtensionIdMapPromise = null;
                     return {};
                  });
         }

         return this._standardExtensionIdToSharedExtensionIdMapPromise;
      }
   }

   /**
    * The class defines APIs related to the H5C application.
    */
   export class H5SdkApplicationApi {

      constructor(pluginIframeScope: PluginIframeScope,
            h5SdkApplicationService: H5SdkApplicationService,
            i18nService: any,
            pluginUrlService: PluginUrlService,
            iframeNavigationDataCache: IframeNavigationDataCache,
            h5SdkThemeService: H5SdkThemeService,
            locale: string) {

         this.navigateTo = function (options: NavigationOptions): void {
            h5SdkApplicationService.navigateTo(
                  options, pluginIframeScope.remotePluginExtensionContext);
         };

         this.getClientInfo = function (): ClientInfo {
            let version = i18nService.getString("CommonUi", "client.version");
            version = version.split('-')[0];

            let clientInfo: ClientInfo = {
               type: "html",
               version: version
            };

            return clientInfo;
         };

         this.getContextObjects = function (): any[] {
            if (pluginIframeScope.isModal) {
               if (!pluginIframeScope.context) {
                  return [];
               }

               return pluginIframeScope.context.contextObjects ?
                     pluginIframeScope.context.contextObjects : [];
            }

            let contextObjectId: string | null = pluginUrlService
                  .getUrlParameterValue(window.location.href, "objectId");

            if (contextObjectId) {
               return [{
                  id: contextObjectId
               }];
            }

            return [];
         };

         this.getClientLocale = function (): string {
            return locale;
         };

         this.getNavigationData = function (): any {
            return iframeNavigationDataCache.getReturnedNavigationData();
         };

         this.getTheme = function (): PluginTheme {
            return h5SdkThemeService.getPluginTheme();
         };
      }

      /**
       * Navigate to the given view
       */
      public navigateTo: (options: NavigationOptions) => void;

      /**
       * Retrieve the passed navigation data
       */
      public getNavigationData: () => any;

      /**
       * Returns specific information about the client
       * @returns {{type: string, version: string}}
       */
      public getClientInfo: () => ClientInfo;

      /**
       * Returns current context objects:
       * <ul>
       *    <li>
       *       In global views the result is empty
       *    </li>
       *
       *    <li>
       *        In views that are associated with a particular vSphere object
       *        the result contains a single plain JavaScript object with
       *        a field named 'id' - the ID of the selected vSphere object
       *    </li>
       *    <li>
       *       In modals opened through a SDK call to htmlSdk.modal.open(configObj)
       *       the result is the value of configObj.contextObjects or an empty array
       *       if configObj.contextObjects is undefined
       *    </li>
       *    <li>
       *       In modals opened through actions defined in the plugin.xml file
       *       the result is an array of context objects representing the targets
       *       of the invoked action. Each context object is a plain JavaScript object
       *       that contains a field named 'id' which is the id of the corresponding
       *       target object.
       *       <i>Note:<i> the result will be an empty array if there are no target objects.
       *    </li>
       * </ul>
       *
       * @returns an array of context objects
       */
      public getContextObjects: () => any[] | undefined;

      /** Returns the current vSphere Client locale
       * @returns string the client locale
       */
      public getClientLocale: () => string;

      /**
       * Returns information about the theme the plugin should use.
       * @returns a PluginTheme object.
       */
      public getTheme: () => PluginTheme;
   }

   export interface NavigationOptions {
      /**
       * The extension id of the view
       */
      targetViewId?: string;

      /**
       * The id of the object
       */
      objectId?: string;

      /**
       * An optional object to store custom data passed to the target plugin view.
       */
      customData?: any;
   }

   export interface ClientInfo {
      /**
       * The type of the client
       */
      type: string;

      /**
       * The version of the client
       */
      version: string;
   }

   angular
         .module("com.vmware.platform.ui")
         .service("h5SdkApplicationService", H5SdkApplicationService);
}
