/* Copyright (c) 2018 VMware, Inc. All rights reserved. */
package com.vmware.samples.htmlsample.services;


import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.vmware.samples.htmlsample.model.Host;
import com.vmware.samples.htmlsample.vim25.services.VimObjectService;
import com.vmware.vise.usersession.ServerInfo;
import com.vmware.vise.usersession.UserSessionService;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Service used to retrieve information about vSphere objects
 */
public class VcInfoServiceImpl implements VcInfoService {
   private final UserSessionService _userSessionService;
   private final VimObjectService _vimObjectService;

   private static final Log _logger = LogFactory.getLog(VcInfoServiceImpl.class);

   public VcInfoServiceImpl(UserSessionService userSessionService, VimObjectService vimObjectService) {
      _userSessionService = userSessionService;
      _vimObjectService = vimObjectService;
   }

   /**
    * Retrieves all connected hosts
    */
   @Override
   public List<Host> getRelatedHosts() {
      final List<Host> connectedHosts = new ArrayList<>();
      final String expectedState = "connected";

      if (_userSessionService.getUserSession() == null) {
         _logger.warn("The UserSession is null.");
         return connectedHosts;
      }

      for (final ServerInfo serverInfoObject : _userSessionService.getUserSession().serversInfo) {
         final List<Host> retrievedHosts = _vimObjectService.retrieveHosts(serverInfoObject);

         for (final Host host : retrievedHosts) {
            if (expectedState.equals(host.state)) {
               // capitalize the first letter of the host state.
               host.state = Character.toUpperCase(host.state.charAt(0)) + host.state.substring(1);
               connectedHosts.add(host);
            }
         }
      }

      return connectedHosts;
   }

   @Override
   public Map<String, String> getServerInfos() {
      final Map<String, String> serverInfos = new HashMap<>();

      if (_userSessionService.getUserSession() == null) {
         _logger.warn("The user session is null.");
         return serverInfos;
      }

      for (final ServerInfo serverInfo : _userSessionService.getUserSession().serversInfo) {
         serverInfos.put(serverInfo.name, serverInfo.serviceGuid);
      }
      return serverInfos;
   }
}
