# Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
# documentation files (the “Software”), to deal in the Software without restriction, including without limitation
# the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and
# to permit persons to whom the Software is furnished to do so, subject to the following conditions:
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
# INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
# PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
# COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
# AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#
# Except as explicitly provided herein, no express or implied licenses, under any VMware patents, copyrights,
# trademarks, or any other intellectual property rights, are granted or waived by implication, exhaustion,
# estoppel, or otherwise, on modified versions of the Software.

import argparse
import datetime
import urllib.request
import inspect
import itertools
import logging
import queue
import sys
import threading

class ReportRow(object):
    def __init__(self, line):
        self.line = line
        self.cols = line.split("\t")

class CustomerReportRow(ReportRow):
    def key(self):
        return tuple(self.cols[0:6])

class LicenseReportRow(ReportRow):
    def key(self):
        return tuple(self.cols[0:3])

class Report(object):
    def __init__(self, um, lines):
        self.um = um
        self.lines = lines
        self.rows = [self.row_class(line) for line in self.lines if line[0] != '#']
        self.keys = set((row.key() for row in self.rows))

    def heading(self):
        return [line for line in self.lines if line[0] == '#' and not line.startswith('#Message Authentication Code')
            and not line.startswith(self.exclude_heading)]

    def keys(self):
        return self.keys

    def billing_memory_for_key(self, key):
        return str(sum((int(row.cols[self.mem_col_idx]) for row in self.rows if key == row.key())))

class CustomerReport(Report):
    combined_heading = '#Customer\tCountry\tPostal\tLicense Type\tCategory\tBillable\tUsage Meter\tCapped Billed vRAM (GB-Hours)\tTotal'
    um_report_num = 3
    title = 'Customer Summary Report'

    def __init__(self, um, lines):
        self.row_class = CustomerReportRow
        self.mem_col_idx = 7
        self.exclude_heading = '#Customer'
        super().__init__(um, lines)

class LicenseReport(Report):
    combined_heading = '#Billable\tCategory\tLicense Type\tUsage Meter\tCapped Billed vRAM (GB-Hours)\tTotal'
    um_report_num = 4
    title = 'License Summary Report'

    def __init__(self, um, lines):
        self.row_class = LicenseReportRow
        self.mem_col_idx = 4
        self.exclude_heading = '#Billable'
        super().__init__(um, lines)

class FailureReport(object):
    pass

class ReportFetcher(threading.Thread):
    def __init__(self, um, fm, to, report_type, results_queue):
        self.um = um
        self.fm = fm
        self.to = to
        self.report_type = report_type
        self.results_queue = results_queue
        super().__init__()

    def run(self):
        um = self.um
        logging.info('Fetching report from ' + um.addr)
        try:
            req = urllib.request.Request('%s://%s:%d/%s/api/report/%d?dateFrom=%s&dateTo=%s' %
                (um.proto, um.addr, um.port, um.context, self.report_type.um_report_num, self.fm, self.to))
            req.add_header("x-usagemeter-authorization", um.token)
            with urllib.request.urlopen(req) as f:
                lines = [l.decode('utf-8').rstrip() for l in f.readlines()]
            report = self.report_type(self.um, lines)
            logging.info('Report from %s fetched' % um.addr)
        except Exception as e:
            logging.error('%s: %s' % (um.addr, e))
            report = FailureReport()
        self.results_queue.put(report)

class ReportDetail(object):
    num_columns = 2
    def __init__(self, um_addr, billing_memory):
        self.um_addr = um_addr
        self.billing_memory = billing_memory

class Um(object):
    def __init__(self, addr, token, proto='https', port=8443, context='um/'):
        self.addr = addr
        self.token = token
        self.proto = proto
        self.port = port
        self.context = context

class ReportMaker(object):
    def __init__(self, ums, start, end):
        self.ums = ums
        self.start = start
        self.end = end

    def run(self, report_type):
        lines = []
        results_queue = queue.Queue()
        for um in self.ums:
            rf = ReportFetcher(um, self.start, self.end, report_type, results_queue)
            rf.start()
        all_reports = [results_queue.get() for um in self.ums]
        reports = [r for r in all_reports if r.__class__ != FailureReport]

        if reports:
            if len(reports) < len(all_reports):
                logging.error("The combined report will be made from only %d of the %d Usage Meters" %
                    (len(reports), len(all_reports)))

            all_keys = set(itertools.chain.from_iterable([report.keys for report in reports]))

            lines.append('\n' + '\n'.join(reports[0].heading()))
            lines.append('#Usage Meters: ' + ", ".join(sorted([report.um.addr for report in reports])))
            lines.append(report_type.combined_heading)

            for key in sorted(all_keys):
                details = [ReportDetail(report.um.addr, report.billing_memory_for_key(key))
                    for report in reports if key in report.keys]
                billing_sum = sum((int(item.billing_memory) for item in details))
                lines.append('\t'.join(key) + ('\t' * (ReportDetail.num_columns + 1)) + str(billing_sum))
                empty_key_cols = '\t' * (len(key) - 1)
                for item in details:
                    lines.append('\t'.join((empty_key_cols, item.um_addr, item.billing_memory)))
        return lines

class ConfigReader(object):
    def read_ums(self, filename, devum=False):
        try:
            with open(filename) as f:
                for line_and_cr in f.readlines():
                    line = line_and_cr.strip()
                    if not line.startswith('#'):
                        parts = line.split()
                        if len(parts) == 2:
                            yield Um(parts[0], parts[1], proto='http', port=8080, context='') \
                                if devum and parts[0] == 'localhost' else Um(parts[0], parts[1])
                        else:
                            logging.error('Line "%s" does not contain two strings separated by whitespace' % line)
                            sys.exit(1)
        except IOError as err:
            logging.error('Unable to open ' + filename)
            sys.exit

if __name__ == '__main__':
    def _discover_reports():
        report_classes = [x[1] for x in inspect.getmembers(sys.modules[__name__], inspect.isclass) \
            if issubclass(x[1], Report) and not x[1] == Report]
        return dict([(x.um_report_num, x) for x in report_classes])

    def _parse_args():
        parser = argparse.ArgumentParser()
        parser.add_argument('umsfile', help='a file containing, on each line, separated by whitespace, a Usage Meter IP address or hostname and an API token')
        def rn(num):
            keys = report_types.keys()
            if num in [str(k) for k in keys]:
                return int(num)
            else:
                raise argparse.ArgumentTypeError('Must be one of ' + str(sorted(keys)))
        sorted_reports = [report_types[rtkey] for rtkey in sorted(report_types)]
        report_choices = ', '.join(['%d: %s' % (d.um_report_num, d.title) for d in sorted_reports])
        parser.add_argument('reportnum', type=rn, help='the number of the report to produce. ' + report_choices)
        parser.add_argument('-o', '--outputfile', help='the name of the file in which to store the report')
        parser.add_argument('-v', '--verbose', action='count', default=0)

        def default_range():
            today = datetime.date.today()
            this_month_first = datetime.date(day=1, month=today.month, year=today.year)
            last_month = this_month_first - datetime.timedelta(days=1)
            last_month_first = datetime.date(day=1, month=last_month.month, year=last_month.year)
            return last_month_first.strftime("%Y%m%d" + '00'), this_month_first.strftime("%Y%m%d" + '00')

        from_default, to_default = default_range()
        parser.add_argument('--fromdate', default=from_default, help='the year, month, day and hour of the beginning of the time range for which to report, as a ten-digit number. This defaults to the beginning of the previous month.')
        parser.add_argument('--todate',   default=to_default, help='the year, month, day and hour of the end of the time range for which to report, as a ten-digit number. This defaults to the beginning of the current month.')
        parser.add_argument('--devum', action='store_true', help='Not for general use')
        return parser.parse_args()

    report_types = _discover_reports()

    args = _parse_args()
    loglev = logging.WARN if not args.verbose else logging.DEBUG if args.verbose > 1 else logging.INFO
    logging.basicConfig(format='%(asctime)s %(threadName)s %(levelname)s %(message)s', level=loglev)

    ums = tuple(ConfigReader().read_ums(args.umsfile, args.devum))

    rm = ReportMaker(ums, args.fromdate, args.todate)
    lines = rm.run(report_types[args.reportnum])
    report_string = '\n'.join(lines)

    if args.outputfile:
        with open(args.outputfile, 'w') as f:
            f.write(report_string)
    else:
        print(report_string)
