"""
Wsgi Server
"""

__author__ = 'VMware, Inc.'
__copyright__ = 'Copyright (c) 2015 VMware, Inc.  All rights reserved.'


import logging
import six
import werkzeug

from werkzeug.exceptions import HTTPException
from werkzeug.routing import (Rule, Map)
from werkzeug.wrappers import Request, Response

from vmware.vapi.settings import sections
from vmware.vapi.lib.constants import (
    JSON_CONTENT_TYPE, JSONRPC)
from vmware.vapi.server.rest_handler import RESTHandler
from vmware.vapi.server.server_interface import ServerInterface
from vmware.vapi.settings import config

logger = logging.getLogger(__name__)


class WsgiApplication(object):
    """
    Python WSGI application. For more details about WSGI
    specification, see PEP 333.
    """
    def __init__(self, msg_handler_map):
        """
        Initialize WsgiApplication

        :type  msg_handler_map: :class:`dict` of :class:`str` and
            :class:`vmware.vapi.protocol.server.api_handler.ApiHandler`
        :param msg_handler_map: Map of content type to the message
            handler for that content type
        """
        self._msg_handler_map = msg_handler_map

        if config.cfg.has_section(sections.JSONRPC):
            jsonrpc_prefix = config.cfg.get(sections.JSONRPC, 'prefix')
            routing_rules = [Rule(jsonrpc_prefix, endpoint=JSONRPC)]
        else:
            # No prefix specified, use path converter rule, that matches
            # any prefix.
            # This is used for backcompat as previously, specifying
            # jsonrpc prefix was not required for wsgi applications.
            routing_rules = [Rule('/<path:path>', endpoint=JSONRPC)]

        if config.cfg.has_section(sections.REST):
            json_msg_handler = self._msg_handler_map.get(JSON_CONTENT_TYPE)
            self._rest_handler = RESTHandler(
                json_msg_handler.provider)
            self._rest_prefix = config.cfg.get(sections.REST, 'prefix')
            routing_rules += self._rest_handler.rest_rules

        self.rule_map = Map(routing_rules)
        logger.info(self.rule_map)

    def _handle_jsonrpc_call(self, request):
        """
        Handle a JSONRPC protocol request

        :type  request: :class:`werkzeug.wrappers.Request`
        :param request: Request object
        :rtype: :class:`str`
        :return: output string
        """
        content_type, _ = werkzeug.http.parse_options_header(
            request.content_type)
        handler = self._msg_handler_map.get(content_type)
        if handler is None:
            raise werkzeug.exceptions.BadRequest(
                'Content-Type %s is not supported' % (content_type))
        else:
            try:
                result = handler.handle_request(request.stream.read())
            except Exception as e:
                logger.exception(e)
                raise werkzeug.exceptions.InternalServerError(
                    'Unexpected error. See server logs for more details.')
        return result

    def _handle_rest_call(self, request, endpoint, args):
        """
        Handle HTTP REST call

        :type  request: :class:`werkzeug.wrappers.Request`
        :param request: Request object
        :type  endpoint: :class:`str`
        :param endpoint: Identifier of the service to be invoked
        :type  args: :class:`dict` of :class:`str` and :class:`object`
        :param args: Arguments parsed from the HTTP URL
        :rtype: :class:`tuple` of :class:`str` and :class:`str`
        :return: HTTP status string and output
        """
        # Accept only json content type
        if request.content_length:
            content_type, _ = werkzeug.http.parse_options_header(
                request.content_type)
            if content_type != JSON_CONTENT_TYPE and request.method in [
                    'POST', 'PATCH', 'PUT']:
                raise werkzeug.exceptions.UnsupportedMediaType(
                    '%s content type is not supported' % content_type)
        try:
            output = self._rest_handler.invoke(request, endpoint, args)
        except werkzeug.exceptions.BadRequest as e:
            raise e
        except Exception as e:
            logger.exception(e)
            raise werkzeug.exceptions.InternalServerError(
                'Unexpected error. See server logs for more details.')
        return output

    @Request.application
    def __call__(self, request):
        """
        The implementation of WsgiApplication

        :type  request: :class:`werkzeug.wrappers.Request`
        :param request: Request object
        :rtype: :class:`werkzeug.wrappers.Response`
        :return: Response object
        """
        try:
            urls = self.rule_map.bind_to_environ(request.environ)
            endpoint, args = urls.match()
            if endpoint == JSONRPC:
                response = Response(self._handle_jsonrpc_call(request))
            else:
                status, result, cookies = self._handle_rest_call(
                    request, endpoint, args)
                response = Response(result)
                response.status_code = status
                if cookies:
                    path = self._rest_prefix
                    for k, v in six.iteritems(cookies):
                        response.set_cookie(k, v, path=path)
            response.content_type = JSON_CONTENT_TYPE
            return response
        except HTTPException as e:
            return e


class WsgiServer(ServerInterface):
    """
    Server wrapper class for Wsgi application.
    """

    SUPPORTED_SCHEMES = ('http', 'https')
    HTTP_CONTENT_MAPPING = {'json': 'application/json',
                            'xml': 'text/xml',
                            '': ''}

    def __init__(self):
        """
        Initialize WsgiServer
        """
        self._protocol_handler_map = {}
        ServerInterface.__init__(self)

    def register_handler(self, addr, msg_type, protocol_handler, ssl_args=None):
        """
        Register protocol handler

        :type  addr: :class:`str`
        :param addr: addr url
        :type  msg_type: :class:`str`
        :param msg_type: protocol message type
        :type  protocol_handler: :class:`vmware.vapi.protocol.server.transport.async_protocol_handler.AsyncProtocolHandler`
        :param protocol_handler: protocol handler for this addr
        :type  ssl_args: :class:`dict`
        :param ssl_args: ssl arguments
        """
        assert(protocol_handler)
        content_type = self.HTTP_CONTENT_MAPPING.get(msg_type)
        if content_type is None:
            logger.error('Unsupported msg type: %s', msg_type)
            return
        self._protocol_handler_map[content_type] = protocol_handler

    def get_wsgi_application(self):
        """
        Returns the WSGI application.

        :rtype: :class:`vmware.vapi.server.wsgi_server.WsgiApplication`
        :return: WSGI application.
        """
        return WsgiApplication(self._protocol_handler_map)

    def serve_forever(self):
        pass

    def shutdown(self):
        pass


def get_server(cfg):  # pylint: disable=W0613
    """
    Get wsgi server

    :type  cfg: :class:`ConfigParser.SafeConfigParser`
    :param cfg: Config parser
    :rtype: :class:`vmware.vapi.server.server_interface.ServerInterface`
    :return: subclass of ServerInterface
    """
    return WsgiServer()
