"""
Local connector
"""

__author__ = 'VMware, Inc.'
__copyright__ = 'Copyright (c) 2015 VMware, Inc.  All rights reserved.'


from vmware.vapi.protocol.client.connector import Connector
from vmware.vapi.core import ExecutionContext, SecurityContext
from vmware.vapi.lib.context import create_default_application_context


class LocalConnector(Connector):
    """
    Protocol connection class to get direct access to ApiProvider
    instead of going over the wire
    """
    def __init__(self, api_provider):
        """
        Initialize LocalConnector

        :type  api_provider: :class:`vmware.vapi.core.ApiProvider`
        :param api_provider: ApiProvider instance to be used
        """
        self._api_provider = api_provider
        self._application_ctx = None
        self._security_ctx = None
        Connector.__init__(self)

    def connect(self):
        """
        Create a connection. No-op for LocalConnector
        """
        pass

    def disconnect(self):
        """
        Disconnect from a connection. No-op for LocalConnector
        """
        pass

    def set_application_context(self, ctx):
        """
        Set the application context

        All the subsequent calls made using this
        connector will use this as the application
        context in the ExecutionContext

        :type  ctx: :class:`vmware.vapi.core.ApplicationContext`
        :param ctx: New application context
        """
        self._application_ctx = ctx

    def set_security_context(self, ctx):
        """
        Set the security context

        All the subsequent calls made using this
        connector will use this as the security
        context in the ExecutionContext

        :type  ctx: :class:`vmware.vapi.core.SecurityContext`
        :param ctx: New security context
        """
        self._security_ctx = ctx

    def new_context(self):
        """
        create new execution context object

        :rtype:  :class:`vmware.vapi.core.ExecutionContext`
        :return: execution context
        """
        app_ctx = self._application_ctx
        # Create a default application context only if
        # the user has not provided anything
        if app_ctx is None:
            app_ctx = create_default_application_context()
        sec_ctx = self._security_ctx
        if sec_ctx is None:
            sec_ctx = SecurityContext()
        return ExecutionContext(app_ctx, sec_ctx)

    def get_api_provider(self):
        """
        Returns the ApiProvider instance backing this connection

        :rtype: :class:`vmware.vapi.core.ApiProvider`
        :return: ApiProvider instance
        """
        return self._api_provider


def get_local_connector(api_provider):
    """
    Creates and returns a local connection for the input ApiProvider

    :type  api_provider: :class:`vmware.vapi.core.ApiProvider`
    :param api_provider: ApiProvider instance
    :rtype: :class:`vmware.vapi.protocol.client.local_connector.LocalConnector`
    :return: Newly created protocol connection for the given ApiProvider
    """
    return LocalConnector(api_provider)
