"""
vAPI Connection factory
"""

__author__ = 'VMware, Inc.'
__copyright__ = 'Copyright (c) 2015 VMware, Inc.  All rights reserved.'

import os
import warnings

from six.moves import configparser
from vmware.vapi.lib.ssl import UnverifiedClientContextFactory
from vmware.vapi.protocol.client.rpc.provider_factory import RpcProviderFactory
from vmware.vapi.protocol.client.msg.connector_factory import ProtocolConnectorFactory
from vmware.vapi.settings import config

REQUESTS_RPC_PROVIDER_NAME = 'requests'


def check_file_exists(filename):
    """
    Check if name is a file and exists

    :type  :class:`str`
    :param file name
    """
    if not os.path.isfile(filename):
        raise OSError(2, 'No such file', filename)


def get_connector(rpc_provider_name, msg_protocol, ssl_context=None, **kwargs):
    """
    Create a connection to the vAPI Provider using the specified arguments

    :type  rpc_provider_name: :class:`str`
    :param rpc_provider_name: The RPC provider to be used for the connection.
                              Valid values are 'http', 'https' and 'requests'
    :type  msg_protocol: :class:`str`
    :param msg_protocol: The message protocol to be used for the connection.
                         Valid values are 'json'.
    :type  ssl_context: :class:`OpenSSL.SSL.Context`
    :param ssl_context: SSL context to use for SSL connections
    :param kwargs: Additional arguments for the RPC provider

    :rtype:  :class:`vmware.vapi.protocol.client.connector.Connector`
    :return: Connection to the vAPI provider
    """
    warnings.warn("This function is deprecated. Please use \
                  get_requests_connector instead.",
                  DeprecationWarning)
    return _get_connector(rpc_provider_name, msg_protocol, ssl_context, **kwargs)


def _get_connector(rpc_provider_name, msg_protocol, ssl_context=None, **kwargs):
    """
    Create a connection to the vAPI Provider using the specified arguments

    :type  rpc_provider_name: :class:`str`
    :param rpc_provider_name: The RPC provider to be used for the connection.
                              Valid values are 'http', 'https' and 'requests'
    :type  msg_protocol: :class:`str`
    :param msg_protocol: The message protocol to be used for the connection.
                         Valid values are 'json'.
    :type  ssl_context: :class:`OpenSSL.SSL.Context`
    :param ssl_context: SSL context to use for SSL connections
    :param kwargs: Additional arguments for the RPC provider

    :rtype:  :class:`vmware.vapi.protocol.client.connector.Connector`
    :return: Connection to the vAPI provider
    """
    #RPC provider
    rpc_provider_args = {}

    if rpc_provider_name in ('http', 'https'):
        # provider url
        rpc_provider_args['url'] = kwargs.get('url')
        if rpc_provider_name == 'http' and not ssl_context:
            # In case of http, use an unverified context. This is a temporary
            # hack to not break clients who pass an https url for an http
            # connector since we now verify server cert by default. Once all the
            # clients move off to using requests, this block can be deleted.
            ssl_context = UnverifiedClientContextFactory().get_context()
        rpc_provider_args['ssl_args'] = {'ssl_context': ssl_context}
    elif rpc_provider_name in ('requests'):
        for arg_name in ('url', 'session', 'timeout', 'pool_size'):
            rpc_provider_args[arg_name] = kwargs.get(arg_name)
    else:
        raise Exception('Invalid RPC Provider specified %s' % rpc_provider_name)

    # Accept additional configuration using a client config file
    if config.cfg is None:
        config.cfg = configparser.SafeConfigParser()
        config_file = kwargs.get('config')
        if config_file:
            check_file_exists(config_file)
            config.cfg.read(config_file)

    # Using SAML based authentication
    # If the configuration does not have SSO signer, add it
    use_saml_token = kwargs.get('use_saml_token', False)
    if (use_saml_token and not config.cfg.has_section(
            'vmware.vapi.protocol.client.msg.json_connector')):
        config.cfg.add_section('vmware.vapi.protocol.client.msg.json_connector')
        config.cfg.set('vmware.vapi.protocol.client.msg.json_connector',
                       'processors',
                       'vmware.vapi.security.sso.JSONSSOSigner')

    rpc_factory = RpcProviderFactory()
    rpc_provider = rpc_factory.get_rpc_provider(rpc_provider_name,
                                                **rpc_provider_args)

    # Msg protocol provider
    connector_factory = ProtocolConnectorFactory()
    connector = connector_factory.get_connector(msg_protocol, rpc_provider)

    return connector


def get_saml_hok_connector(rpc_provider_name, msg_protocol='json', ssl_context=None, **kwargs):
    """
    Create a connection that uses SAML Hok based authentication
    to connect to a vAPI Provider

    :type  rpc_provider_name: :class:`str`
    :param rpc_provider_name: The RPC provider to be used for the connection.
                              Valid values are 'http', 'https' or 'requests'
    :type  msg_protocol: :class:`str`
    :param msg_protocol: The message protocol to be used for the connection.
                         Valid values are 'json'.
    :type  ssl_context: :class:`OpenSSL.SSL.Context`
    :param ssl_context: SSL context to use for SSL connections
    :type  kwargs: :class:`dict` of :class:`str` and :class:`object`
    :param kwargs: Additional arguments for the RPC provider

    :rtype:  :class:`vmware.vapi.protocol.client.connector.Connector`
    :return: Connection to the vAPI provider
    """
    warnings.warn("This function is deprecated. Please use \
                  get_requests_hok_connector instead.",
                  DeprecationWarning)
    return _get_connector(rpc_provider_name, msg_protocol, ssl_context,
                          use_saml_token=True, **kwargs)


def get_requests_connector(session, msg_protocol='json', url=None,
                           timeout=None, pool_size=8):
    """
    Create a connection that uses 'requests' library for http(s) connections to
    a vAPI Provider.

    :type  session: :class:`requests.Session`
    :param session: Session object
    :type  msg_protocol: :class:`str`
    :param msg_protocol: Message protocol to be used for the connection. Valid
        values are 'json'.
    :type  url: :class:`str`
    :param url: HTTP(S) URL to be used
    :type  timeout: :class:`int`
    :param timeout: Request timeout
    :type  pool_size: :class:`int`
    :param pool_size: Connection pool size to be used
    :rtype: :class:`vmware.vapi.protocol.client.connector.Connector`
    :return: Connection to the vAPI provider
    """
    return _get_connector(REQUESTS_RPC_PROVIDER_NAME, msg_protocol,
                          url=url, session=session,
                          timeout=timeout, pool_size=pool_size)


def get_requests_hok_connector(session, msg_protocol='json', url=None,
                               timeout=None, pool_size=8):
    """
    Create a connection that uses SAML Hok based authentication using 'requests'
    library to connect to a vAPI Provider.

    :type  session: :class:`requests.Session`
    :param session: Session object
    :type  msg_protocol: :class:`str`
    :param msg_protocol: Message protocol to be used for the connection. Valid
        values are 'json'.
    :type  url: :class:`str`
    :param url: HTTP(S) URL to be used
    :type  timeout: :class:`int`
    :param timeout: Request timeout
    :type  pool_size: :class:`int`
    :param pool_size: Connection pool size to be used
    :rtype: :class:`vmware.vapi.protocol.client.connector.Connector`
    :return: Connection to the vAPI provider
    """
    return _get_connector(REQUESTS_RPC_PROVIDER_NAME, msg_protocol, url=url,
                          session=session, timeout=timeout, pool_size=pool_size,
                          use_saml_token=True)
