/*
 * ******************************************************
 * Copyright VMware, Inc. 2016.  All Rights Reserved.
 * ******************************************************
 *
 * DISCLAIMER. THIS PROGRAM IS PROVIDED TO YOU "AS IS" WITHOUT
 * WARRANTIES OR CONDITIONS # OF ANY KIND, WHETHER ORAL OR WRITTEN,
 * EXPRESS OR IMPLIED. THE AUTHOR SPECIFICALLY # DISCLAIMS ANY IMPLIED
 * WARRANTIES OR CONDITIONS OF MERCHANTABILITY, SATISFACTORY # QUALITY,
 * NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
 */

package com.vmware.spbm.samples;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.vmware.common.annotations.Action;
import com.vmware.common.annotations.Option;
import com.vmware.common.annotations.Sample;
import com.vmware.pbm.InvalidArgumentFaultMsg;
import com.vmware.pbm.PbmCapabilityProfile;
import com.vmware.pbm.PbmProfileId;
import com.vmware.spbm.connection.ConnectedServiceBase;
import com.vmware.spbm.connection.helpers.PbmUtil;
import com.vmware.spbm.connection.helpers.VCUtil;
import com.vmware.vim25.ArrayOfVirtualDevice;
import com.vmware.vim25.BaseConfigInfoBackingInfo;
import com.vmware.vim25.BaseConfigInfoDiskFileBackingInfo;
import com.vmware.vim25.BaseConfigInfoFileBackingInfo;
import com.vmware.vim25.ConcurrentAccessFaultMsg;
import com.vmware.vim25.DuplicateNameFaultMsg;
import com.vmware.vim25.DynamicProperty;
import com.vmware.vim25.FileFaultFaultMsg;
import com.vmware.vim25.ID;
import com.vmware.vim25.InsufficientResourcesFaultFaultMsg;
import com.vmware.vim25.InvalidCollectorVersionFaultMsg;
import com.vmware.vim25.InvalidControllerFaultMsg;
import com.vmware.vim25.InvalidDatastoreFaultMsg;
import com.vmware.vim25.InvalidNameFaultMsg;
import com.vmware.vim25.InvalidPropertyFaultMsg;
import com.vmware.vim25.InvalidStateFaultMsg;
import com.vmware.vim25.LocalizedMethodFault;
import com.vmware.vim25.ManagedObjectReference;
import com.vmware.vim25.MissingControllerFaultMsg;
import com.vmware.vim25.NotFoundFaultMsg;
import com.vmware.vim25.ObjectContent;
import com.vmware.vim25.ObjectSpec;
import com.vmware.vim25.PropertyFilterSpec;
import com.vmware.vim25.PropertySpec;
import com.vmware.vim25.RetrieveOptions;
import com.vmware.vim25.RetrieveResult;
import com.vmware.vim25.RuntimeFaultFaultMsg;
import com.vmware.vim25.TaskInProgressFaultMsg;
import com.vmware.vim25.TaskInfoState;
import com.vmware.vim25.TraversalSpec;
import com.vmware.vim25.VStorageObject;
import com.vmware.vim25.VirtualDevice;
import com.vmware.vim25.VirtualDeviceConfigSpec;
import com.vmware.vim25.VirtualDeviceConfigSpecOperation;
import com.vmware.vim25.VirtualDeviceFileBackingInfo;
import com.vmware.vim25.VirtualDisk;
import com.vmware.vim25.VirtualMachineConfigSpec;
import com.vmware.vim25.VirtualMachineDefinedProfileSpec;
import com.vmware.vim25.VirtualSCSIController;
import com.vmware.vim25.VmConfigFaultFaultMsg;

/**
 * <pre>
 * FcdAssociateProfile
 *
 * This sample attaches a virtual storage object to given virtual machine and
 * associates a given storage profile to the virtual storage object.
 *
 * <b>Parameters:</b>
 * url                    [required] : url of the web service
 * username               [required] : username for the authentication
 * password               [required] : password for the authentication
 * vstorageobjectid       [required] : Uuid of the disk
 * vmpath                 [required] : Inventory path of the virtual machine
 * datastorename          [required] : Name of the datastore which contains
 *                                     the virtual storage object
 * profilename            [required] : Name of the storage profile
 *
 * <b>Command Line:</b>
 * Attach a given virtual storage object to the given virtual machine.
 * run.bat com.vmware.spbm.samples.FcdAssociateProfile --url [webserviceurl]
 * --username [username] --password [password]
 * --vstorageobjectid [vstorageobjectid] --vmpath [vmpath]
 * --datastorename [datastorename] --profilename [profilename]
 *
 * Ex: vmpath ::
 * &lt;DatacenterName&gt;/&lt;HiddenFolder - "vm"&gt;/&lt;VmName&gt;
 * ( testDC/vm/VM_Test1 )
 * </pre>
 */
@Sample(name = "fcd-associateprofile", description = "This sample attaches a"
        + " given virtual storage object(FCD) to the given virtual machine.")
public class FcdAssociateProfile extends ConnectedServiceBase {

    private String vStorageObjectId;
    private String vmPath;
    private String datastoreName;
    private String profileName;

    /**
     * @param vStorageObjectId the vStorageObjectId to set
     */
    @Option(name = "vstorageobjectid",
            required = true,
            description = "Uuid of the disk.")
    public void setVStorageObjectId(String vStorageObjectId) {
        this.vStorageObjectId = vStorageObjectId;
    }

    /**
     * @param vmPath the vmPath to set
     */
    @Option(name = "vmpath",
            required = true,
            description = "Path of virtual machine.")
    public void setVmPath(String vmPath) {
        this.vmPath = vmPath;
    }

    /**
     * @param datastoreName the datastoreName to set
     */
    @Option(name = "datastorename",
            required = true,
            description = "Name of datastore containing the  vstorageobject.")
    public void setDatastoreName(String datastoreName) {
        this.datastoreName = datastoreName;
    }

    /**
     * @param profileName the profileName to set
     */
    @Option(name = "profilename",
            required = true,
            description = "Name of the storage profile")
    public void setProfileName(String profileName) {
        this.profileName = profileName;
    }

    /**
     * This method returns a boolean value specifying whether the Task has
     * succeeded or failed.
     *
     * @param task ManagedObjectReference representing the Task.
     * @return boolean value representing the Task result.
     * @throws InvalidCollectorVersionFaultMsg
     * @throws RuntimeFaultFaultMsg
     * @throws InvalidPropertyFaultMsg
     */
    boolean getTaskResultAfterDone(ManagedObjectReference task)
            throws InvalidPropertyFaultMsg, RuntimeFaultFaultMsg,
            InvalidCollectorVersionFaultMsg {

        boolean retVal = false;

        // info has a property "state" - for state of the task
        Object[] result = VCUtil.waitForTask(connection, task, new String[] {
                "info.state", "info.error" }, new String[] { "state" },
                new Object[][] { new Object[] { TaskInfoState.SUCCESS,
                        TaskInfoState.ERROR } });

        if (result[0].equals(TaskInfoState.SUCCESS)) {
            retVal = true;
        }
        if (result[1] instanceof LocalizedMethodFault) {
            throw new RuntimeException(
                    ((LocalizedMethodFault) result[1]).getLocalizedMessage());
        }
        return retVal;
    }

    /**
     * Creates the virtual storage object.
     *
     * @throws InvalidPropertyFaultMsg
     * @throws com.vmware.pbm.RuntimeFaultFaultMsg
     * @throws InvalidArgumentFaultMsg
     * @throws NotFoundFaultMsg
     * @throws InvalidDatastoreFaultMsg
     * @throws FileFaultFaultMsg
     * @throws VmConfigFaultFaultMsg
     * @throws MissingControllerFaultMsg
     * @throws InvalidStateFaultMsg
     * @throws InvalidControllerFaultMsg
     * @throws InvalidCollectorVersionFaultMsg
     * @throws TaskInProgressFaultMsg
     * @throws InvalidNameFaultMsg
     * @throws InsufficientResourcesFaultFaultMsg
     * @throws DuplicateNameFaultMsg
     * @throws ConcurrentAccessFaultMsg
     */
    void associateProfileWithFcdByAttachingToVm()
            throws InvalidPropertyFaultMsg, RuntimeFaultFaultMsg,
            InvalidArgumentFaultMsg, com.vmware.pbm.RuntimeFaultFaultMsg,
            FileFaultFaultMsg, InvalidDatastoreFaultMsg, NotFoundFaultMsg,
            InvalidControllerFaultMsg, InvalidStateFaultMsg,
            MissingControllerFaultMsg, VmConfigFaultFaultMsg,
            InvalidCollectorVersionFaultMsg, ConcurrentAccessFaultMsg,
            DuplicateNameFaultMsg, InsufficientResourcesFaultFaultMsg,
            InvalidNameFaultMsg, TaskInProgressFaultMsg {
        // Get vStorageObjectManager Mor.
        ManagedObjectReference morVStrObjManager = connection
                .getVimServiceContent().getVStorageObjectManager();

        System.out.println("Info: vmPath is :: " + vmPath);
        // Get Virtual Machine Mor.
        ManagedObjectReference morVm = connection.getVimPort()
                .findByInventoryPath(
                        connection.getVimServiceContent().getSearchIndex(),
                        vmPath);

        if (morVm == null) {
            throw new RuntimeException("Error: morVm Obtained is null");
        }
        // Get all the input Mor's required.
        ManagedObjectReference morDatastore = getMOREFsInContainerByType(
                connection.getVimServiceContent().getRootFolder(), "Datastore")
                .get(datastoreName);

        PbmCapabilityProfile profile = PbmUtil.getPbmProfile(connection,
                profileName);
        PbmProfileId profileId = profile.getProfileId();

        // Retrieve a vStorageObject based on the given vStorageObjectId.
        System.out.println("Operation: Retrieve the vStorageObjects in"
                + " datastore.");
        VStorageObject retrievedVStrObj = connection.getVimPort()
                .retrieveVStorageObject(morVStrObjManager,
                        makeId(vStorageObjectId), morDatastore);
        if (retrievedVStrObj.getConfig().getId().getId()
                .equals(vStorageObjectId)) {
            System.out.printf(
                    "Success: Retrieved vStorageObject :: %n [ %s ]%n",
                    retrievedVStrObj.getConfig().getId().getId());
        } else {
            String msg = "Error: Given vStorageObject [ " + vStorageObjectId
                    + "] and retrieved VStorageObject are different.";
            throw new RuntimeException(msg);
        }

        if (reconfigVMToAddFcdAndUpdateProfile(morVm, retrievedVStrObj,
                profileId)) {
            System.out.printf("Success: Associated profile :: [ Name = %s ] to"
                    + " vStorageObject [ Name = %s , Uuid = %s ]%n",
                    profileName, retrievedVStrObj.getConfig().getName(),
                    retrievedVStrObj.getConfig().getId().getId());
        } else {
            String msg = "Error: Associating profile [ " + profileName
                    + " ] to the vStorageObject [ " + vStorageObjectId + " ].";
            throw new RuntimeException(msg);
        }

        // Retrieve a vStorageObject based on the given vStorageObjectId and
        // verify
        // virtualMachine is reflected as a new consumer in the
        // retrievedVStorageObject
        // when a virtualMachine is reconfigured to ADD a FCD.
        System.out.println("Operation: Retrieve the vStorageObjects in"
                + " datastore.");
        VStorageObject retrievedVStrObjWithConsumer = connection.getVimPort()
                .retrieveVStorageObject(morVStrObjManager,
                        makeId(vStorageObjectId), morDatastore);
        if (retrievedVStrObjWithConsumer.getConfig().getId().getId()
                .equals(vStorageObjectId)) {
            if (retrievedVStrObjWithConsumer.
                    getConfig().getConsumerId().get(0) != null) {
                System.out.printf("Success: Retrieved vStorageObject :: %n"
                                + " [ Uuid = %s ] is associated with"
                                + " consumer [ ConsumerId = %s ]%n",
                                retrievedVStrObjWithConsumer.getConfig()
                                        .getId().getId(),
                                retrievedVStrObjWithConsumer.getConfig()
                                        .getConsumerId().get(0).getId());
            } else {
                String msg = "Error: Given vStorageObject [ "
                        + vStorageObjectId
                        + "] does not have a consumer attached to it.";
                throw new RuntimeException(msg);
            }
        } else {
            String msg = "Error: Given vStorageObject [ " + vStorageObjectId
                    + "] and retrieved VStorageObject are different.";
            throw new RuntimeException(msg);
        }
    }

    /**
     * Reconfigures a VM to add a FCD to the VM and update the disk profile. If
     * input param profileId is null, an empty profile is attempted to be set.
     *
     * @param morVm
     * @param vStorageObj
     * @param profileId
     *
     * @throws InvalidPropertyFaultMsg
     * @throws RuntimeFaultFaultMsg
     * @throws VmConfigFaultFaultMsg
     * @throws TaskInProgressFaultMsg
     * @throws InvalidStateFaultMsg
     * @throws InvalidNameFaultMsg
     * @throws InvalidDatastoreFaultMsg
     * @throws InsufficientResourcesFaultFaultMsg
     * @throws FileFaultFaultMsg
     * @throws DuplicateNameFaultMsg
     * @throws ConcurrentAccessFaultMsg
     * @throws InvalidCollectorVersionFaultMsg
     * @return isReconfigSuccessfull
     */
    protected boolean reconfigVMToAddFcdAndUpdateProfile(
            ManagedObjectReference morVm, VStorageObject vStorageObj,
            PbmProfileId profileId) throws RuntimeFaultFaultMsg,
            InvalidPropertyFaultMsg, ConcurrentAccessFaultMsg,
            DuplicateNameFaultMsg, FileFaultFaultMsg,
            InsufficientResourcesFaultFaultMsg, InvalidDatastoreFaultMsg,
            InvalidNameFaultMsg, InvalidStateFaultMsg, TaskInProgressFaultMsg,
            VmConfigFaultFaultMsg, InvalidCollectorVersionFaultMsg {
        System.out.printf("Info: Reconfig VM To Add Fcd [ %s ]%n", vStorageObj
                .getConfig().getId().getId());
        if (profileId != null) {
            System.out.printf(
                    "Info: Associating profile with Profile Id [ %s ] to"
                            + " vStorageObject [ Uuid = %s ].",
                    profileId.getUniqueId(),
                    vStorageObj.getConfig().getId().getId());
        }
        boolean isReconfigSuccessful = false;

        VirtualMachineConfigSpec vmConfigSpec = new VirtualMachineConfigSpec();
        VirtualDeviceConfigSpec vdiskSpec = getDiskDeviceConfigSpec(morVm,
                vStorageObj, profileId);
        List<VirtualDeviceConfigSpec> vdiskSpecArray =
                new ArrayList<VirtualDeviceConfigSpec>();
        vdiskSpecArray.add(vdiskSpec);
        vmConfigSpec.getDeviceChange().addAll(vdiskSpecArray);
        ManagedObjectReference reconfigVmTaskMor = connection.getVimPort()
                .reconfigVMTask(morVm, vmConfigSpec);
        isReconfigSuccessful = getTaskResultAfterDone(reconfigVmTaskMor);
        return isReconfigSuccessful;
    }

    /**
     * Get the VirtualDeviceConfigSpec of the new disk to be added to
     * the a VM.
     *
     * @param morVm
     * @param vStorageObj
     * @param profileId
     * @return VirtualDeviceConfigSpec of the disk
     * @throws RuntimeFaultFaultMsg
     * @throws InvalidPropertyFaultMsg
     */
    private VirtualDeviceConfigSpec getDiskDeviceConfigSpec(
            ManagedObjectReference morVm, VStorageObject vStorageObj,
            PbmProfileId profileId) throws InvalidPropertyFaultMsg,
            RuntimeFaultFaultMsg{

        VirtualDeviceConfigSpec virtualDiskDeviceConfigSpec =
                new VirtualDeviceConfigSpec();

        VirtualDisk virtualDisk = new VirtualDisk();
        String fileName = null;
        BaseConfigInfoBackingInfo vStorageObjBackingInfo = vStorageObj
                .getConfig().getBacking();
        if (vStorageObjBackingInfo instanceof BaseConfigInfoFileBackingInfo) {
            BaseConfigInfoFileBackingInfo vStorageObjFileBackingInfo =
                    (BaseConfigInfoDiskFileBackingInfo) vStorageObjBackingInfo;
            fileName = vStorageObjFileBackingInfo.getFilePath();
        }

        if (fileName == null) {
            String message = "Error: No FileName/FilePath obtained for the"
                    + " vStorageObject [ "
                    + vStorageObj.getConfig().getId().getId() + " ].";
            throw new RuntimeException(message);
        }

        // Set the fileName / backing for the VirtualDisk =
        // fileName/filePath/backing of vStorageObject.
        VirtualDeviceFileBackingInfo virtualDiskDeviceFilebackingInfo =
                new VirtualDeviceFileBackingInfo();
        virtualDiskDeviceFilebackingInfo.setFileName(fileName);
        virtualDisk.setBacking(virtualDiskDeviceFilebackingInfo);

        // Set the controller Key and unit number obtained from virtual
        // machine.
        int ckey = 0;
        int unitNumber = 0;
        List<Integer> getControllerKeyReturnArr = getControllerKey(morVm);
        if (!getControllerKeyReturnArr.isEmpty()) {
            ckey = getControllerKeyReturnArr.get(0);
            unitNumber = getControllerKeyReturnArr.get(1);
        }
        virtualDisk.setControllerKey(ckey);
        virtualDisk.setUnitNumber(unitNumber);

        // Set the size of the virtualDisk = size of vStorageObject.
        long virtualDiskSizeInKB = 1024 * vStorageObj.getConfig()
                .getCapacityInMB();
        virtualDisk.setCapacityInKB(virtualDiskSizeInKB);
        virtualDisk.setKey(-1);

        // Set the profile to be associated with the disk.
        VirtualMachineDefinedProfileSpec diskProfileSpec =
                new VirtualMachineDefinedProfileSpec();
        diskProfileSpec.setProfileId(profileId.getUniqueId());
        System.out.printf("Info: Associating disk with profileId : %s%n",
                profileId.getUniqueId());
        virtualDiskDeviceConfigSpec.getProfile().add(diskProfileSpec);

        // Set the operation , file operation and device details.
        virtualDiskDeviceConfigSpec
                .setOperation(VirtualDeviceConfigSpecOperation.ADD);
        virtualDiskDeviceConfigSpec.setFileOperation(null);
        virtualDiskDeviceConfigSpec.setDevice(virtualDisk);

        return virtualDiskDeviceConfigSpec;
    }

    /**
     * Gets the controller key and the next available free unit number on the
     * SCSI controller
     *
     * @param vmMor
     * @return a list containing controller key followed by next free unit
     *         number.
     * @throws InvalidPropertyFaultMsg
     * @throws RuntimeFaultFaultMsg
     */
    List<Integer> getControllerKey(ManagedObjectReference vmMor)
            throws InvalidPropertyFaultMsg, RuntimeFaultFaultMsg {
        List<Integer> retVal = new ArrayList<Integer>();

        List<VirtualDevice> listvd = ((ArrayOfVirtualDevice) VCUtil
                .getEntityProps(connection, vmMor,
                        new String[] { "config.hardware.device" }).get(
                        "config.hardware.device")).getVirtualDevice();

        Map<Integer, VirtualDevice> deviceMap =
                new HashMap<Integer, VirtualDevice>();
        for (VirtualDevice virtualDevice : listvd) {
            deviceMap.put(virtualDevice.getKey(), virtualDevice);
        }
        boolean found = false;
        for (VirtualDevice virtualDevice : listvd) {
            if (virtualDevice instanceof VirtualSCSIController) {
                VirtualSCSIController vscsic =
                        (VirtualSCSIController) virtualDevice;
                int[] slots = new int[16];
                slots[7] = 1;
                List<Integer> devicelist = vscsic.getDevice();
                for (Integer deviceKey : devicelist) {
                    if (deviceMap.get(deviceKey).getUnitNumber() != null) {
                        slots[deviceMap.get(deviceKey).getUnitNumber()] = 1;
                    }
                }
                for (int i = 0; i < slots.length; i++) {
                    if (slots[i] != 1) {
                        retVal.add(vscsic.getKey());
                        retVal.add(i);
                        found = true;
                        break;
                    }
                }
                if (found) {
                    break;
                }
            }
        }

        if (!found) {
            throw new RuntimeException(
                    "The SCSI controller on the vm has maxed out its "
                            + "capacity. Please add an additional SCSI"
                            + " controller");
        }
        return retVal;
    }

    /**
     * Returns all the MOREFs of the specified type that are present under the
     * container
     *
     * @param folder
     *            {@link ManagedObjectReference} of the container to begin the
     *            search from
     * @param morefType
     *            Type of the managed entity that needs to be searched
     * @return Map of name and MOREF of the managed objects present. If none
     *         exist then empty Map is returned
     * @throws InvalidPropertyFaultMsg
     * @throws RuntimeFaultFaultMsg
     */

    Map<String, ManagedObjectReference> getMOREFsInContainerByType(
            ManagedObjectReference folder, String morefType)
            throws InvalidPropertyFaultMsg, RuntimeFaultFaultMsg {
        String PROP_ME_NAME = "name";
        ManagedObjectReference viewManager = connection.getVimServiceContent()
                .getViewManager();
        ManagedObjectReference containerView = connection.getVimPort()
                .createContainerView(viewManager, folder,
                        Arrays.asList(morefType), true);

        Map<String, ManagedObjectReference> tgtMoref =
                new HashMap<String, ManagedObjectReference>();

        // Create Property Spec
        PropertySpec propertySpec = new PropertySpec();
        propertySpec.setAll(Boolean.FALSE);
        propertySpec.setType(morefType);
        propertySpec.getPathSet().add(PROP_ME_NAME);

        TraversalSpec ts = new TraversalSpec();
        ts.setName("view");
        ts.setPath("view");
        ts.setSkip(false);
        ts.setType("ContainerView");

        // Now create Object Spec
        ObjectSpec objectSpec = new ObjectSpec();
        objectSpec.setObj(containerView);
        objectSpec.setSkip(Boolean.TRUE);
        objectSpec.getSelectSet().add(ts);

        // Create PropertyFilterSpec using the PropertySpec and ObjectPec
        // created above.
        PropertyFilterSpec propertyFilterSpec = new PropertyFilterSpec();
        propertyFilterSpec.getPropSet().add(propertySpec);
        propertyFilterSpec.getObjectSet().add(objectSpec);

        List<PropertyFilterSpec> propertyFilterSpecs =
                new ArrayList<PropertyFilterSpec>();
        propertyFilterSpecs.add(propertyFilterSpec);

        RetrieveResult rslts = connection.getVimPort().retrievePropertiesEx(
                connection.getVimServiceContent().getPropertyCollector(),
                propertyFilterSpecs, new RetrieveOptions());
        List<ObjectContent> listobjcontent = new ArrayList<ObjectContent>();
        if (rslts != null && rslts.getObjects() != null
                && !rslts.getObjects().isEmpty()) {
            listobjcontent.addAll(rslts.getObjects());
        }
        String token = null;
        if (rslts != null && rslts.getToken() != null) {
            token = rslts.getToken();
        }
        while (token != null && !token.isEmpty()) {
            rslts = connection.getVimPort().continueRetrievePropertiesEx(
                    connection.getVimServiceContent().getPropertyCollector(),
                    token);
            token = null;
            if (rslts != null) {
                token = rslts.getToken();
                if (rslts.getObjects() != null &&
                        !rslts.getObjects().isEmpty()) {
                    listobjcontent.addAll(rslts.getObjects());
                }
            }
        }
        for (ObjectContent oc : listobjcontent) {
            ManagedObjectReference mr = oc.getObj();
            String entityNm = null;
            List<DynamicProperty> dps = oc.getPropSet();
            if (dps != null) {
                for (DynamicProperty dp : dps) {
                    entityNm = (String) dp.getVal();
                }
            }
            tgtMoref.put(entityNm, mr);
        }
        return tgtMoref;
    }

    /**
     * Utility method to wrap ID string in an ID object.
     *
     * @param idStr
     * @return id in ID format.
     */
    private static ID makeId(String idStr) {
        ID id = new ID();
        id.setId(idStr);
        return id;
    }

    @Action
    public void run() throws InvalidPropertyFaultMsg, RuntimeFaultFaultMsg,
            FileFaultFaultMsg, InvalidDatastoreFaultMsg, NotFoundFaultMsg,
            InvalidControllerFaultMsg, InvalidStateFaultMsg,
            MissingControllerFaultMsg, VmConfigFaultFaultMsg,
            InvalidCollectorVersionFaultMsg, InvalidArgumentFaultMsg,
            com.vmware.pbm.RuntimeFaultFaultMsg, ConcurrentAccessFaultMsg,
            DuplicateNameFaultMsg, InsufficientResourcesFaultFaultMsg,
            InvalidNameFaultMsg, TaskInProgressFaultMsg {
        associateProfileWithFcdByAttachingToVm();
    }
}
