﻿-- These are needed to prevent errors when handling Xml data.
SET ARITHABORT ON
SET QUOTED_IDENTIFIER ON
GO

-- This script creates the stored procedures needed by the Task Execution subsystem of VRM.

CREATE PROCEDURE [vrm_tasks_UpdateTaskExecutionAgentExpiration] (
	@TaskExecutionAgentId uniqueidentifier,
	@ExpirationDateTimeUtc datetime,
	@PreviousExpirationDateTimeUtc datetime = null output 
	)
AS
	SET NOCOUNT ON
	
	BEGIN TRANSACTION
	
	DECLARE @CurrentExpirationDateTimeUtc datetime
	
	SELECT
		@PreviousExpirationDateTimeUtc = ExpirationDateTimeUtc
	FROM
		vrm_tasks_TaskExecutionAgents WITH(UpdLock)
	WHERE
		TaskExecutionAgentId = @TaskExecutionAgentId
	
	IF @PreviousExpirationDateTimeUtc IS NULL
	BEGIN
		
		INSERT INTO vrm_tasks_TaskExecutionAgents (
			TaskExecutionAgentId,
			ExpirationDateTimeUtc
		) VALUES (
			@TaskExecutionAgentId,
			@ExpirationDateTimeUtc
		)
		
		SET @PreviousExpirationDateTimeUtc = @ExpirationDateTimeUtc
		
	END ELSE BEGIN
		
		UPDATE vrm_tasks_TaskExecutionAgents
		SET
			ExpirationDateTimeUtc = @ExpirationDateTimeUtc
		WHERE
			TaskExecutionAgentId = @TaskExecutionAgentId
			
	END
	
	COMMIT TRANSACTION
	
GO
GRANT EXECUTE ON [vrm_tasks_UpdateTaskExecutionAgentExpiration] TO Task_Manager
GO

CREATE PROCEDURE [vrm_tasks_ExpireTaskExecutionAgents] (
	@CurrentDateTimeUtc datetime
	)
AS
	SET NOCOUNT ON
	
	DELETE FROM vrm_tasks_TaskExecutionAgents
	OUTPUT (
		deleted.TaskExecutionAgentId
	)
	WHERE
		ExpirationDateTimeUtc < @CurrentDateTimeUtc
GO
GRANT EXECUTE ON [vrm_tasks_ExpireTaskExecutionAgents] TO Task_Manager
GO


CREATE PROCEDURE [vrm_tasks_AddTask] (
	@TaskId uniqueidentifier,
	@TaskData varbinary(max),
	@IdleTimeout datetime,
	@NonExistentResourcesTimeout datetime)
AS
	SET NOCOUNT ON
	
	INSERT INTO vrm_tasks_Tasks (
		TaskId,
		TaskData,
		IdleTimeout,
		NonExistentResourcesTimeout,
		TaskExecutionStatus
	) VALUES (
		@TaskId,
		@TaskData,
		@IdleTimeout,
		@NonExistentResourcesTimeout,
		0)
GO
GRANT EXECUTE ON [vrm_tasks_AddTask] TO Task_Manager
GO


CREATE PROCEDURE [vrm_tasks_GetTask] (
	@TaskId uniqueidentifier)
AS
	SET NOCOUNT ON

	SELECT 
		TaskData
	FROM
		vrm_tasks_Tasks WITH(UpdLock)
	WHERE
		TaskId = @TaskId
GO
GRANT EXECUTE ON [vrm_tasks_GetTask] TO Task_Manager
GO


CREATE PROCEDURE [vrm_tasks_RemoveTask] (
	@TaskId uniqueidentifier,
	@Removed bit = 0 output)
AS
	SET NOCOUNT ON

	DELETE FROM vrm_tasks_Tasks
	WHERE
		TaskId = @TaskId
	
	SET @Removed = @@ROWCOUNT
GO
GRANT EXECUTE ON [vrm_tasks_RemoveTask] TO Task_Manager
GO


CREATE PROCEDURE [vrm_tasks_UpdateTaskStatus] (
	@TaskId uniqueidentifier,
	@TaskExecutionStatus int)
AS
	SET NOCOUNT ON

	UPDATE 
		vrm_tasks_Tasks
	SET
		TaskExecutionStatus = @TaskExecutionStatus
	WHERE
		TaskId = @TaskId
GO
GRANT EXECUTE ON [vrm_tasks_UpdateTaskStatus] TO Task_Manager
GO


CREATE PROCEDURE [vrm_tasks_UpdateTaskExecutionTimeout] (
	@TaskId uniqueidentifier,
	@Timeout datetime)
AS
	SET NOCOUNT ON

	UPDATE 
		vrm_tasks_Tasks
	SET
		ExecutionTimeout = @Timeout
	WHERE
		TaskId = @TaskId
GO
GRANT EXECUTE ON [vrm_tasks_UpdateTaskExecutionTimeout] TO Task_Manager
GO


CREATE PROCEDURE [vrm_tasks_UpdateTaskStatusUpdateTimeout] (
	@TaskId uniqueidentifier,
	@Timeout datetime)
AS
	SET NOCOUNT ON

	UPDATE 
		vrm_tasks_Tasks
	SET
		StatusUpdateTimeout = @Timeout
	WHERE
		TaskId = @TaskId
GO
GRANT EXECUTE ON [vrm_tasks_UpdateTaskStatusUpdateTimeout] TO Task_Manager
GO


CREATE PROCEDURE [vrm_tasks_GetExpiredTasks] (
	@CurrentDateTimeUtc datetime)
AS
	SET NOCOUNT ON

	SELECT
		TaskId,
		CASE
			-- 2 == Idle timeout
			WHEN (
				TaskExecutionStatus = 0
				AND IdleTimeout <= @CurrentDateTimeUtc)
			THEN 2
			
			-- 3 == Nonexistent resources timeout
			WHEN (
				TaskExecutionStatus = 0 
				AND NonExistentResourcesTimeout <= @CurrentDateTimeUtc)
			THEN 3
			
			-- 4 == Executing timeout
			WHEN (
				TaskExecutionStatus = 1
				AND ExecutionTimeout <= @CurrentDateTimeUtc)
			THEN 4
			
			-- 5 == Status Update Timeout
			WHEN (
				TaskExecutionStatus = 1
				AND StatusUpdateTimeout <= @CurrentDateTimeUtc)
			THEN 5
		END AS TaskExecutionStatus
		
	FROM
		vrm_tasks_Tasks
		
	WHERE
		-- Get Idle tasks
		(
			-- 0 == Idle
			TaskExecutionStatus = 0 
			AND IdleTimeout <= @CurrentDateTimeUtc)
		
		-- Get Nonexistent Resources Timed Out tasks
		OR (
			-- 0 == Idle
			TaskExecutionStatus = 0 
			AND NonExistentResourcesTimeout <= @CurrentDateTimeUtc)

		-- Get Execution Timed Out tasks
		OR (
			-- 1 == Executing
			TaskExecutionStatus = 1
			AND ExecutionTimeout <= @CurrentDateTimeUtc)
		
		-- Get Status Update Timed Out tasks
		OR (
			-- 1 == Executing
			TaskExecutionStatus = 1
			AND StatusUpdateTimeout <= @CurrentDateTimeUtc)
GO
GRANT EXECUTE ON [vrm_tasks_GetExpiredTasks] TO Task_Manager
GO


CREATE PROCEDURE [vrm_tasks_RegisterRequestHandler] (
	@RequestHandlerId uniqueidentifier,
	@Resources xml)
AS
	SET NOCOUNT ON

	BEGIN TRANSACTION

	INSERT INTO vrm_tasks_RequestHandlers (
		RequestHandlerId
	) VALUES (
		@RequestHandlerId
	)
	
	INSERT INTO vrm_tasks_RequestHandlerResources (
		RequestHandlerId,
		ResourceName,
		ResourceValue
	) SELECT
		@RequestHandlerId,
		resources.Resource.value (N'Name[1]', N'nvarchar(200)'),
		resources.Resource.value (N'Value[1]', N'nvarchar(200)')
	FROM
		@Resources.nodes (N'/ArrayOfResource/Resource') resources(Resource)

	COMMIT TRANSACTION
GO
GRANT EXECUTE ON [vrm_tasks_RegisterRequestHandler] TO Task_Manager
GO

CREATE PROCEDURE [vrm_tasks_UnregisterRequestHandler] (
	@RequestHandlerId uniqueidentifier)
AS
	SET NOCOUNT ON

	-- This will cascade delete the request handler's resources.
	DELETE FROM vrm_tasks_RequestHandlers
	WHERE
		RequestHandlerId = @RequestHandlerId
GO
GRANT EXECUTE ON [vrm_tasks_UnregisterRequestHandler] TO Task_Manager
GO

CREATE PROCEDURE [vrm_tasks_AddRequest] (
	@RequestId uniqueidentifier,
	@Resources xml)
AS
	SET NOCOUNT ON

	BEGIN TRANSACTION

	INSERT INTO vrm_tasks_Requests (
		RequestId,
		Idle
	) VALUES (
		@RequestId,
		1
	)
	
	INSERT INTO vrm_tasks_RequestResources (
		RequestId,
		ResourceName,
		ResourceValue,
		Usage,
		AllowedUsage,
		Explicit
	) SELECT
		@RequestId,
		resourceRequests.ResourceRequest.value (N'Name[1]', N'nvarchar(200)'),
		resourceRequests.ResourceRequest.value (N'Value[1]', N'nvarchar(200)'),
		resourceRequests.ResourceRequest.value (N'Usage[1]', N'decimal(28,6)'),
		resourceRequests.ResourceRequest.value (N'AllowedUsage[1]', N'decimal(28,6)'),
		resourceRequests.ResourceRequest.value (N'Explicit[1]', N'bit')
	FROM
		@Resources.nodes (N'/ArrayOfResourceRequest/ResourceRequest') resourceRequests(ResourceRequest)

	COMMIT TRANSACTION
GO
GRANT EXECUTE ON [vrm_tasks_AddRequest] TO Task_Manager
GO

CREATE PROCEDURE [vrm_tasks_RemoveRequest] (
	@RequestId uniqueidentifier)
AS
	SET NOCOUNT ON

	-- This will cascade delete the request's resources.
	DELETE FROM vrm_tasks_Requests
	WHERE
		RequestId = @RequestId
GO
GRANT EXECUTE ON [vrm_tasks_RemoveRequest] TO Task_Manager
GO


CREATE PROCEDURE [vrm_tasks_SatisfyRequest] (
	@RequestHandlerId uniqueidentifier)
AS
	SET NOCOUNT ON;

	-- Create a Common table expression that collects
	-- the SUM(Usage) and MIN(AllowedUsage) for each
	-- ACTIVE resource.
	WITH CurrentResourceUsage (
		ResourceName,
		ResourceValue,
		TotalUsage,
		AllowedUsage
	) AS (SELECT
		rr.ResourceName,
		rr.ResourceValue,
		SUM (Usage),
		MIN (AllowedUsage)
		
		-- Get the resources
	FROM
		vrm_tasks_RequestResources rr 
		
		-- Limit to active requests
	INNER JOIN
		vrm_tasks_Requests r 
	ON
		r.RequestId = rr.RequestId
	AND r.Idle = 0
	
	GROUP BY
		rr.ResourceName,
		rr.ResourceValue)
		
	-- Mark exactly one request as not idle
	UPDATE TOP (1) vrm_tasks_Requests

	SET
		Idle = 0
	
	-- And return it to the caller
	OUTPUT inserted.RequestId
		
	WHERE
		-- The task is currently idle
		Idle = 1
		
		AND
		
		-- There isn't a resource that is required and not provided by this
		-- request handler or
		-- would violated the AllowedUsage restrictions.
		NOT EXISTS (
			SELECT * 
			
			FROM 
				vrm_tasks_RequestResources rr
			
			-- Find the resource provided by the request handler
			LEFT OUTER JOIN
				vrm_tasks_RequestHandlerResources rhr
			ON
				rhr.RequestHandlerId = @RequestHandlerId
			AND rhr.ResourceName = rr.ResourceName
			AND (rhr.ResourceValue = rr.ResourceValue
				OR rhr.ResourceValue IS NULL)
			
			-- Find the current resource usage.
			LEFT OUTER JOIN
				CurrentResourceUsage cru
			ON
				cru.ResourceName = rr.ResourceName
			AND cru.ResourceValue = rr.ResourceValue
			
			WHERE
				-- Limit to the resources provided by this request
				rr.RequestId = vrm_tasks_Requests.RequestId
			AND
			
				(
					-- If the request resource is "explicit" and the request handler
					-- doesn't have the resource, we can't satisfy this request.
					rr.Explicit = 1 
				AND rhr.RequestHandlerId IS NULL
				
					OR
					
					-- Validate the resource's usage restrictions.
					
					(
						(rr.Usage + cru.TotalUsage) > rr.AllowedUsage
						OR (rr.Usage + cru.TotalUsage) > cru.AllowedUsage
					)
				)
		)
		
		
GO
GRANT EXECUTE ON [vrm_tasks_SatisfyRequest] TO Task_Manager
GO

CREATE PROCEDURE [vrm_tasks_DoRequestResourcesExist] (
	@RequestId uniqueidentifier,
	@ResourcesExist bit = null output
) AS

	SET NOCOUNT ON


	SET @ResourcesExist = 0

	SELECT
	-- If the number of rows returned is greater than 0, this sets
	-- @ResourcesExist to true.
		@ResourcesExist = 1
		--rh.RequestHandlerId,COUNT(rr.Explicit), COUNT(rhr.RequestHandlerId)
		--*
	-- Start with the list of request handlers
	FROM
		vrm_tasks_RequestHandlers rh
	
	-- Make sure there is in fact a request.
	-- This whole select will return 0 rows if the request doesn't exist,
	-- so that it won't be possible to execute the request.
	INNER JOIN
		vrm_tasks_Requests r
	ON	r.RequestId = @RequestId

	-- Get the explicitly requested resources that are not provided.
	LEFT OUTER JOIN (
		-- Get the requested resources.
		vrm_tasks_RequestResources rr

	-- Find the request handler resource.
	-- If it doesn't exist, this will make count(rr.Explicit)
	-- get a value > 0 because of the WHERE clause down below.
	LEFT OUTER JOIN
		vrm_tasks_RequestHandlerResources rhr
	ON
		rr.Explicit = 1
	AND	rhr.ResourceName = rr.ResourceName
	AND (rhr.ResourceValue = rr.ResourceValue
		OR rhr.ResourceValue IS NULL)
	) ON
		-- Limit to those resources that are for this request, explicit and
		-- either not provided by any resource handler or are provided by this resource handler.
		rr.RequestId = @RequestId
	AND	rr.Explicit = 1
	AND	(rhr.RequestHandlerId = rh.RequestHandlerId
		OR	rhr.RequestHandlerId IS NULL)

	-- Group by the request handler so we can count
	-- the number of explicit request resources that don't have corresponding
	-- request handler resources.
	GROUP BY
		rh.RequestHandlerId
	
	HAVING
		-- This selects those request handlers that have 0 explicitly requested resources
		-- missing from the set of resources provided by the request handler
		COUNT(rr.Explicit) = COUNT(rhr.RequestHandlerId)
GO
GRANT EXECUTE ON [vrm_tasks_DoRequestResourcesExist] TO Task_Manager
GO
