from bag import Bag
from ipaddr import IpAddr
from operator import or_
from version import Version

class InterfaceFactory:
    @staticmethod
    def create(bag, offset):
        v = Version().get_version()
        
        if v[0] == 'RedHat':
            return RedHatInterface(bag, offset)
        elif v[0] == 'SuSE':
            return SuseInterface(bag, offset)

class Interface:
    def __init__(self, bag, offset):
        self.bag = bag
        self.offset = offset

    def address(self):
        return "VirtualMachine.Network%s.Address" % self.offset

    def is_address(self):
        return self.bag.has_key(self.address())

    def subnet_mask(self):
        return "VirtualMachine.Network%s.SubnetMask" % self.offset

    def is_subnet_mask(self):
        return self.bag.has_key(self.subnet_mask())

    def mac(self):
        return "VirtualMachine.Network%s.MacAddress" % self.offset

    def is_mac(self):
        return self.bag.has_key(self.mac())

#     def ignores(self):
#     def lines_dhcp(self):
#     def lines_static(self):

    def lines(self, ifcfg_eth_lines):
        lines = []
        for line in ifcfg_eth_lines:
            if not reduce(or_, map(lambda x: line.startswith(x), self.ignores())):
                lines.append(line.strip())

        if self.is_address():
            return self.lines_static(lines)

        return lines

class RedHatInterface(Interface):
    def lines_dhcp(self, lines):
        lines.append("BOOTPROTO=dhcp")
        lines.append("DEVICE=eth%s" % self.offset)
        lines.append("ONBOOT=yes")

        return lines

    def lines_static(self, lines):
        addr = self.bag[self.address()]
        mask = self.bag[self.subnet_mask()]
        ipv4 = IpAddr(addr, mask)

        lines.append("BOOTPROTO=static")
        lines.append("BROADCAST=%s" % ipv4.broadcast())
        lines.append("DEVICE=eth%s" % self.offset)
        lines.append("IPADDR=%s" % addr)
        lines.append("ONBOOT=yes")
        lines.append("NETMASK=%s" % mask)
        lines.append("NETWORK=%s" % ipv4.network())

        return lines

    def ignores(self):
        return ('BOOTPROTO=', 'BROADCAST=', 'DEVICE=', 'IPADDR=', 'ONBOOT=', 'NETMASK=', 'NETWORK')


class SuseInterface(Interface):
    def lines_dhcp(self, lines):
        lines.append('BOOTPROTO=dhcp')
        lines.append('STARTMODE=auto')
        
        return lines

    def lines_static(self, lines):
        addr = self.bag[self.address()]
        mask = self.bag[self.subnet_mask()]
        ipv4 = IpAddr(addr, mask)

        lines.append("BOOTPROTO=static")
        lines.append("BROADCAST=%s" % ipv4.broadcast())
        lines.append("IPADDR=%s" % addr)
        lines.append("NETMASK=%s" % mask)
        lines.append("STARTMODE=auto")

        return lines

    def ignores(self):
        return ('BOOTPROTO=', 'BROADCAST=', 'IPADDR=', 'STARTMODE=', 'NETMASK=')

# vim: sw=4 ts=4 et
