#!/bin/sh

###################################################
# Copyright 2015-2020 VMware, Inc.  All rights reserved.
###################################################

#
# Run uninstallation before installation
# if there is existing viewagent installed
#
uninstall_existing_package() {
   #
   # if current is managed installation and pre-installation is unmanaged,
   # delete the machine config.
   #
   if [ $MANAGED = "true" ] &&
      [ $preManagedFlag = "false" ] &&
      [ -f "$MACHINE_CFG_FILE" ]; then
      echo_log "Unmanaged agent upgrade to managed agent, removing old machine config."
      echo -n "Warning: You are upgrading un-managed VMware Horizon Agent to managed "
      echo -n "VMware Horizon Agent. Your existing desktop pool will not work after "
      echo -n "that, and you need create new desktop pool. Please refer to the product "
      echo -n "guide for detailed steps."
      rm -f $MACHINE_CFG_FILE
   fi

   if [ -f /usr/lib/vmware/viewagent/bin/uninstall_viewagent.sh ]; then
      /usr/lib/vmware/viewagent/bin/uninstall_viewagent.sh
   fi
}

configure_distribution() {
   #
   # Common
   #
   configure_distribution_common

   #
   # Command specific
   #
   case "$DISTRO_ID" in
   ${DISTRO_ID_CENTOS}|\
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_SUSE}|\
   ${DISTRO_ID_RHEL_WORKSTATION}|\
   ${DISTRO_ID_NEOKYLIN})
      INSTALLER_OPTS_UNREG="n:b:d:r:m:a:s:u:p:j:k:K:B:S:A:M:L:F:C:f:U:T:R"
      INSTALLER_OPTS_REG="A:r:m:a:M:S:L:F:C:T:R"
      ;;
   ${DISTRO_ID_KYLIN}|\
   ${DISTRO_ID_UBUNTU})
      INSTALLER_OPTS_UNREG="n:b:d:r:m:a:s:u:p:j:k:K:B:S:A:M:L:F:C:f:U:T:XR"
      INSTALLER_OPTS_REG="A:r:m:a:M:S:L:F:C:T:U:XR"
      ;;
   esac
}

isPlatformArchSupported() {
   local arch="`uname -i`"
   local binpath="`dirname $0`/VMwareBlastServer/VMwareBlastServer"
   local binarch="`file $binpath`"
   local ret=1

   case "$arch" in
   "x86_64")
      echo "$binarch" | grep "x86-64" >/dev/null 2>&1
      if [ "$?" = "0" ]; then
         ret=0
      fi
      ;;
   esac

   return $ret
}

#
# Note, this usage doesn't consider already registered agents.
# prompt the usage and exit if user doesn't input mandatory parameters
#
usage_exit() {
   name=`basename $0`
   buildtype=`sed -n '2p' Product.txt`

   echo "Usage: $name [OPTION]... [VALUE]..."
   echo ""
   echo "Optional parameters but required information"
   echo "-A        yes|no <Accept|Refuse EULA and FIPS statement>"
   echo ""
   echo "Optional parameters"
   echo "--multiple-session Install or Upgrade Linux Agent to Multiple-Session Mode. Default is Singleton Mode."
   echo "-M        yes|no Upgrade the Linux Agent to managed|un-managed agent. Default is yes."
   echo "-s        Self signed cert subject DN. By default, installer will use Blast for CN."
   echo "-j        JMS SSL keystore password. By default, installer will generate a random string."
   echo "-r        yes|no <Do|Not restart system after installation automatically>. Default is no."
   echo "-m        yes|no <Install|Bypass smartcard redirection support>. Default is no."
   echo "-F        yes|no <Install|Bypass Client Drive Redirection support>. Default is yes."
   echo "-f        yes|no <Install|Bypass FIPS mode>. Default is no.(Only support RedHat 7.x/8.x)"
   echo "-a        yes|no <Install|Bypass audioin support>. Default is no."
   echo "-U        yes|no <Install|Bypass USB Redirection support>. Default is no."
   echo "-C        yes|no <Install|Bypass Clipboard Redirection support>. Default is yes."
   echo "-S        yes|no <Install|Bypass SingleSignOn support>. Default is yes."
   echo "-T        yes|no <Install|Bypass TrueSSO support>. Default is no."
   #echo "-L        yes|no <enable|disable Composer Link Clone Agent for Automated Desktop Pool. Composer Link Clone Agent requires ntfs-3g package>. Default is yes."
   echo ""
   echo "Example:"
   echo "Fresh installation"
   echo "  sudo ./install_viewagent.sh -A yes"
   echo ""
   echo "Upgrade from un-managed VM deployment and keep un-managed VM style"
   echo "  sudo ./install_viewagent.sh -A yes -M no"
   echo ""
   echo "Upgrade from un-managed VM deployment and convert to managed VM style, which requires new desktop pool creation on broker:"
   echo "  sudo ./install_viewagent.sh -A yes"
   echo ""

   if [ "$buildtype" != "release" ]; then
      echo "Below parameters and examples are for un-managed VM, which is internal using only."

      echo "Mandatory parameters"
      echo "-b Horizon Connection Server hostname or IP address"
      echo "-d Broker administrator's domain name"
      echo "-u Horizon Connection Server administrator username"
      echo ""
      echo "Optional parameters but required information"
      echo "-p Horizon Connection Server administrator password"
      echo ""
      echo "Optional parameters"
      echo "-n Machine Name"
      echo "-k Broker administrator's Active Directory address. Only necessary for Kerberos authentication"
      echo "-K Broker host's Active Directiory address. Only necessary for Kerberos authentication when broker host and broker admin are from different domain"
      echo "-B Broker host's domain name. Only necessary for Kerberos authentication when broker host and broker admin are from different domain"
      echo ""
      echo "Example:"
      echo "Broker host and admin are from same domain"
      echo "    Digest-MD5: sudo install_viewagent.sh -M no -A yes -n mymachine -b mybroker.mydomain.com -d mydomain.com -u myadmin -p Pa\$ssword"
      echo "    Kerberos: sudo install_viewagent.sh -M no -A yes -n mymachine -b mybroker.mydomain.com -d mydomain.com -u myadmin -p Pa\$ssword -k mydomain-host.mydomain.com
Broker host and admin are from different domain"
      echo "    MD5-Digest: sudo install_viewagent.sh -M no -A yes -n mymachine -b mybroker.mydomain1.com -d mydomain2.com -u myadmin@mydomain2.com -p Pa\$ssword"
      echo "    Kerberos: sudo install_viewagent.sh -M no -A yes -n mymachine -b mybroker.mydomain1.com -B mydomain1.com -K mydomain1-host.mydomain1.com -d mydomain2.com -u myadmin -p Pa\$ssword -k mydomain2-host.mydomain2.com"
      echo ""
   fi

   exit
}

#
# Similar with usage_exit() function.
# Add tips about optional parameters.
# Note, this function is not used in this shell, just for future usage.
#
usage_exit_with_optional_params(){
   name=`basename $0`
   echo -n "Usage: $name -M"
   echo -n "[-n <vm name>] [-s <self signed cert subject DN>] "
   echo ""
   echo -n "Example: sudo install_viewagent.sh -M -n linux-centOS-x64-1 "
   echo ""
   echo "Some optional parameters"
   echo "     -A     yes|no <Accept|Refuse EULA>"
   exit
}

dependency_package_guide_message(){
   echo_term -n "For more details, please review the Horizon Setup Guide, "
   echo_term "with regard to Package Dependencies that may not be provided by this distribution."
}

set_name_and_DN() {
   if [ -z "$NAME" ]; then
      NAME="`hostname`"
      echo_log -n "Linux agent generates a default name: $NAME. "
      echo_log ""
   fi

   if [ -z "$DN" ]; then
      echo_log "Subject distinguished name for certificate not specified"
      echo_log "Generate default distinguished name:"
      # change the CN to uuid, since managed agent don't input domain name
      DN=${DNPREFIX}"/CN="Blast
      echo_log "${DN}"
      #
      # check the length of CN (common name) in Subject distinguished name for
      # openssl certificate. The limit of CN is up to 64 characters.
      #
      CN=Blast
      if [ ${#CN} -gt 64 ]; then
         echo_term "The length of CN ($CN) is too long, it should not be more than 64 characters."
         echo_term "Please re-input -n parameter."
         exit
      fi
   fi
}

verify_umanaged_agent_parameters() {
   if [ -z "$BROKER" ]; then
      echo_term "Horizon Connection Server not specified"
      usage_exit
   fi
   if [ -z "$KDC" ]; then
      echo_log -n "KDC was not specified, and cannot use Kerberos authentication to "
      echo_log -n "broker. Use DIGEST-MD5 authentication to register machine to broker."
   else
      echo_log "KDC was specified, and use Kerberos authentication to register machine to broker."
   fi

   if [ -z "$DOMAIN" ]; then
      echo_term "Domain not specified"
      usage_exit
   fi
   if [ -z "$LDAP_USERNAME" ]; then
      echo_term "LDAP username not specified"
      usage_exit
   fi
   if [ -z "$LDAP_PASSWORD" ]; then
      echo_term "LDAP password not specified"
      usage_exit
   fi
}

verify_vmtools() {
   #
   # Check open-vm-tools package on RHEL/CentOS 7.x, SUSE 12.x and Ubuntu16.04
   #
   local isOpenVmTool="false"
   local needCheckDeploypkg="true"
   case "$DISTRO_ID" in
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_RHEL_WORKSTATION}|\
   ${DISTRO_ID_CENTOS})
      if [ `echo "$DISTRO_RELEASE" | cut -f1 -d"."` = 7 ]; then
         isOpenVmTool="true"
      fi
      ;;
   ${DISTRO_ID_SUSE})
      if [ `echo "$DISTRO_RELEASE" | cut -f1 -d"."` = 12 ]; then
         isOpenVmTool="true"
      fi
      ;;
   ${DISTRO_ID_UBUNTU})
      if [ $OS_MAJOR = 16 ] || [ $OS_MAJOR = 18 ]; then
         isOpenVmTool="true"
         needCheckDeploypkg="false"
      fi
      ;;
   *)
      ;;
   esac

   if [ "$isOpenVmTool" = "true" ]; then
      $PKG_QUERY open-vm-tools > /dev/null 2>&1
      if [ "$?" != "0" ]; then
         echo_term -n "No Open-VM-Tools package found. Open-VM-Tools is recommended on this platform, VMware Tools can also work but it's deprecated. "
         echo_term "Please follow the product guide to install Open-VM-Tools."
         vmtoolsd -v >/dev/null 2>&1
         if [ "$?" != "0" ]; then
            echo_term "Please install open-vm-tools or VMware Tools."
            exit
         else
            # Check the vmtools running or not
            vmtoolsPid=$(pidof vmtoolsd)
            if [ "$vmtoolsPid" = "" ]; then
               echo_term "VMware Tools is not running."
               exit
            fi
         fi
      else
         if [ "$needCheckDeploypkg" = "true" ]; then
            OPEN_VM_TOOLS_VERSION=$(rpm -q --queryformat "%{VERSION}\n" open-vm-tools | awk 'BEGIN{FS="."} (($1<9) || ($1==9 && $2<10)) {print $0}')
            if [ -n "$OPEN_VM_TOOLS_VERSION" ]; then
               rpm -q open-vm-tools-deploypkg >/dev/null 2>&1
               if [ "$?" != "0" ]; then
                  echo_term "Please install package open-vm-tools-deploypkg. More details please refer to http://kb.vmware.com/kb/2075048"
                  exit
               fi
            fi
         fi
      fi
   else
      #
      # Check vmtools installed or not on other distros
      #
      vmtoolsd -v >/dev/null 2>&1
      if [ "$?" != "0" ]; then
         echo_term "Please install VMware Tools."
         exit
      else
         # Check the vmtools running or not
         vmtoolsPid=$(pidof vmtoolsd)
         if [ "$vmtoolsPid" = "" ]; then
            echo_term "VMware Tools is not running."
            exit
         fi
      fi
   fi
}

#
# Parameters check should be the first thing to do
# it doesn't make sense to install a lot of things
# then tell user against wrong parameters
#
verify_fill_input_parameters() {

   if [ "$freshInstall" = "true" ]; then
      # fresh install
      echo "The installation will install VMware Horizon Agent on your computer."
      JMS_KEYSTORE_PASSWORD=$(makepassword)
      if [ "$MANAGED" = "false" ]; then
         echo_log "fresh install unmanaged agent"
         if [ "$SINGLETON_MODE" = "true" ]; then
            echo "Warning: You are installing un-managed VMware Horizon Agent, which is not officially supported."
         fi
         verify_umanaged_agent_parameters
      fi

      set_name_and_DN

   else
      # upgrade
      echo "The installation will upgrade VMware Horizon Agent on your computer."
      if [ "$preManagedFlag" = "true" ]; then
         # pre-installation is managed.
         if [ "$MANAGED" = "false" ]; then
            # -M no
            echo "Warning: Don't support managed agent upgrade to unmanaged VM"
            usage_exit
         fi
      else
         # pre-installation is unmanaged
         if [ "$MANAGED" = "false" ]; then
            # unmanaged agent upgrade
            JMS_KEYSTORE_PASSWORD=$(makepassword)
            if [ "$isRegistered" = "false" ]; then
               verify_umanaged_agent_parameters
               set_name_and_DN
            fi
         fi
      fi
   fi
   echo "Installation Start ..."
}

#
# When the first parameter is less than the second one,
# return 0, otherwise, return 1
#
compare_version(){
   if [ "$#" != 2 ]; then
      return 1
   fi

   local v1="$1"
   local v2="$2"
   [ "$v1" = "$v2" ] && return 1

   local v=`printf "$v1\n$v2" | sort --version-sort | head -n 1`
   [ "$v" = "$v1" ] && return 0

   return 1
}

verify_environment() {
   case "$DISTRO_ID" in
   ${DISTRO_ID_CENTOS}|\
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_RHEL_WORKSTATION})
      #
      # Gdm version is right?
      #
      local need_upgrade_gdm="false"
      local gdm_ver=$(rpmquery gdm | cut -c5- |awk -F'.x86_64' '{print $1}')

      if [ -n "$gdm_ver" ]; then
         if [ ${DISTRO_RELEASE} = "6.8" ]; then
            if compare_version "$gdm_ver" "2.30.4-69.el6"; then
               need_upgrade_gdm="true"
            fi
         elif [ ${DISTRO_RELEASE} = "7.2" ]; then
            if compare_version "$gdm_ver" "3.14.2-19.el7"; then
               need_upgrade_gdm="true"
            fi
         elif [ ${DISTRO_RELEASE} = "7.4" ]; then
            if compare_version "$gdm_ver" "3.26.2.1-5.el7"; then
               need_upgrade_gdm="true"
            fi
         fi
      fi

      if [ "$need_upgrade_gdm" = "true" ]; then
         echo_term "Gdm version $gdm_ver is lower than expected, please follow KB https://kb.vmware.com/s/article/71312 to upgrade gdm."
         exit
      fi
      ;;
   ${DISTRO_ID_SUSE}|${DISTRO_ID_NEOKYLIN})
      ;;
   ${DISTRO_ID_KYLIN}|${DISTRO_ID_UBUNTU})
      #
      # LightDM present?
      #
       if [ ! -d "$LIGHTDM_PATH" ] && [ "$ISKUBUNTU" != "true" ] && \
            ! [ "${DISTRO_ID}" = ${DISTRO_ID_UBUNTU} -a "$OS_MAJOR" = "18" ]; then
         echo_term "$LIGHTDM_PATH not found"
         exit
      fi
      ;;
   esac

   #
   # For NeoKylin, we need to check the version of gnome-shell and mate-session-manager.
   # In the default gnome-shell and mate-session-manager, there are no options that can control suspend/hibernate/shutdown/reboot.
   # The CS2C company supplies two patches to enhance control options.
   # The gnome-shell-3.8.3-2.nk.17.i686/x86_64.rpm will append new options in the config file /usr/share/gnome-shell/js/misc/config.js.
   # The mate-session-manager-1.6.0-3.nk.12.i686/x86_64.rpm will create a new config file /etc/mate-session-dialog.conf.
   # If the installer set the options firstly, then install these two patches, these control options will be overridden.
   # For the above reason, the installer must check if the customer has installer gnome-shell-3.8.3-2.nk.17.i686/x86_64.rpm
   # and mate-session-manager-1.6.0-3.nk.12.i686/x86_64.rpm or not.
   #
   if [ "$DISTRO_ID" = "$DISTRO_ID_NEOKYLIN" ]; then
      #
      # Check the gnome-shell version is correct or not.
      # If the version is lower than the min version 3.8.3, the installer should print error message and exit;
      # If the version is equal to the min version 3.8.3, the release number must not be lower than 2.nk.17.
      #
      GNOME_SHELL_VERSION=$(rpm -q --queryformat "%{VERSION}\n" gnome-shell | awk 'BEGIN{FS="."} ($1<3) || ($1==3 && $2<8) || ($1==3 && $2==8 && $3<3) {print $0}')
      if [ -n "$GNOME_SHELL_VERSION" ]; then
         # The gnome-shell version is lower than the min version 3.8.3
         echo_term "gnome-shell-3.8.3-2.nk.17 or above version is required"
         exit
      else
         GNOME_SHELL_VERSION=$(rpm -q --queryformat "%{VERSION}\n" gnome-shell | awk 'BEGIN{FS="."} ($1==3 && $2==8 && $3==3) {print $0}')
         if [ -n "$GNOME_SHELL_VERSION" ]; then
            # Check the gnome-shell release number is correct or not
            GNOME_SHELL_RELEASE=$(rpm -q --queryformat "%{RELEASE}\n" gnome-shell | awk 'BEGIN{FS="."} (($1>2) || ($1==2 && $3>=17)) && ($2=="nk") {print $0}')
            if [ "$GNOME_SHELL_RELEASE" = "" ]; then
               echo_term "gnome-shell-3.8.3-2.nk.17 or above version is required"
               exit
            fi
         fi
      fi

      #
      # Check the mate-session-manager version is correct or not.
      # If the version is lower than the min version 1.6.0, the installer should print error message and exit;
      # If the version is equal to the min version 1.6.0, the release number must not be lower than 3.nk.12.
      #
      MATE_SESSION_VERSION=$(rpm -q --queryformat "%{VERSION}\n" mate-session-manager | awk 'BEGIN{FS="."} ($1<1) || ($1==1 && $2<6) || ($1==1 && $2==6 && $3<0) {print $0}')
      if [ -n "$MATE_SESSION_VERSION" ]; then
         echo_term "mate-session-manager-1.6.0-3.nk.12 or above version is required"
         exit
      else
         MATE_SESSION_VERSION=$(rpm -q --queryformat "%{VERSION}\n" mate-session-manager | awk 'BEGIN{FS="."} ($1==1 && $2==6 && $3==0) {print $0}')
         if [ -n "$MATE_SESSION_VERSION" ]; then
            # Check the mate-session-manager release is correct or not
            MATE_SESSION_RELEASE=$(rpm -q --queryformat "%{RELEASE}\n" mate-session-manager | awk 'BEGIN{FS="."} (($1>3) || ($1==3 && $3>=12)) && ($2=="nk") {print $0}')
            if [ "$MATE_SESSION_RELEASE" = "" ]; then
               echo_term "mate-session-manager-1.6.0-3.nk.12 or above version is required"
               exit
            fi
         fi
      fi
   fi

   #
   # Installation location
   #
   if [ ! -d $VIEWAGENT_PATH ]; then
      echo_term "$VIEWAGENT_PATH not found"
      exit
   fi

   if [ "$MANAGED" = "true" ]; then
      verify_vmtools
   fi

#
# Is openssl package present (required to generate certificate)?
#
   $PKG_QUERY openssl > /dev/null 2>&1
   if [ "$?" != "0" ]; then
      $PKG_INSTALL openssl
      if [ "$?" != "0" ]; then
         echo_term "Failed to install OpenSSL"
         exit
      fi
      echo_log "OpenSSL installed"
   fi

#check if meet the link clone requirement.
if [ "$LINKCLONE" = "true" ]; then
   which ntfs-3g > /dev/null 2>&1
   if [ "$?" != "0" ]; then
      echo_term -n "You are installing Horizon Agent with Composer Link Clone "
      echo_term -n "feature enabled, and ntfs-3g package is required. To "
      echo_term -n "continue the installation, you need:"
      echo_term ""
      echo_term -n "1. disable the Composer Link Clone feature with option "
      echo_term -n "\"-L no\" if you don't need it"
      echo_term ""
      echo_term -n "2. install the ntfs-3g package before installation"
      echo_term -n " if you need Composer Link Clone feature"
      echo_term ""
      exit
   fi
fi

#
# Is hostname resolvable?
#
host="`hostname`"
ping -c 1 "$host" > /dev/null 2>&1
if [ "$?" != "0" ]; then
   echo_term "Hostname (${host}) not resolvable, terminating installation"
   echo_term "Ensure host is registered with name service or listed in hosts file"
   exit
fi

#
# Configuration
#
   if [ -z "$OS_VERSION" ]; then
      echo_term "OS version not specified"
      exit
   fi
}

# If user doesn't provide -j with passoword, generate one.
# make sure this can run on various Linux distributions.
# Note,ubuntu using dash,which doesn't support array and supports
# only minimal feature set.
# Also, don't use third-party tools like uuid gen.
makeonechar() {

   # generate a symbol whose ASCII value is between 48 and 48+43
   # in dash, $RANDOM is not reliable.

   # -n limit only 2 bytes into hexdump
   # the string format followed -e means decode 2 bytes each time ,
   # %u will interpret the 2 bytes into unsigned integers in decimal
   random=`hexdump -n 2 -e '/2 "%u"' /dev/urandom`
   value1=`expr $random % 44 + 48`
   # convert integet into char
   # printf instead of awk maybe more portable
   # chara=`awk "BEGIN{printf \"%c\", $value1}"`
   tmp=$(printf '%03o' $value1)
   chara=$(printf \\$tmp)

   echo "$chara"
}

makepassword() {
   i=0
   while [ $i -lt 8 ]
   do
      i=`expr $i + 1`
      onechar=$(makeonechar)
      mergechar=$mergechar$onechar
   done
   echo "$mergechar"
}

backup_pam() {
   if [ "$SSO_ENABLE" = "yes" -o "$SCREDIR_ENABLE" = "yes" ]; then
      #
      # backup /etc/pam.d/lightdm or /etc/pam.d/gdm-password on ubuntu
      #
      if [ "$DISTRO_ID" = ${DISTRO_ID_UBUNTU} ]; then
         if [ ! -d ${VIEWAGENT_BACKUP_PATH} ]; then
            mkdir -p ${VIEWAGENT_BACKUP_PATH}
         fi

         if [ $OS_MAJOR = 14 -o ${OS_MAJOR} = "16" ]; then
            install /etc/pam.d/lightdm ${VIEWAGENT_BACKUP_PATH}
         else
            install /etc/pam.d/gdm-password ${VIEWAGENT_BACKUP_PATH}
         fi
      fi
   fi
}

create_blast_account() {
   #
   # Create the vmware blast server group
   #
   echo_log "Begin to create vmware blast server group $BLAST_GROUP"
   if ! getent group $BLAST_GROUP >/dev/null; then
      ${ADDGROUP} --system $BLAST_GROUP > /dev/null 2>&1
   else
      echo_log "Blast server group $BLAST_GROUP already exists"
   fi

   #
   # Create the vmware blast account
   #
   echo_log "Begin to create vmware blast server account $BLAST_ACCOUNT"
   if ! getent passwd $BLAST_ACCOUNT >/dev/null; then
      if [ "$DISTRO_ID" = "$DISTRO_ID_SUSE" ]; then
         useradd --system --shell /bin/false $BLAST_ACCOUNT > /dev/null 2>&1
      else
         adduser --system --no-create-home --shell /bin/false  -N $BLAST_ACCOUNT > /dev/null 2>&1
      fi
      usermod -g $BLAST_GROUP -L -c "VMware Blast Server Manager" $BLAST_ACCOUNT > /dev/null 2>&1
   else
      if PWSTAT=`passwd -S "$BLAST_ACCOUNT"` 2>/dev/null;then
         echo "$PWSTAT" | cut -f2 -d' ' | grep "L" >/dev/null 2>&1
         if [ "$?" != "0" ]; then
            echo_term "Blast server account $BLAST_ACCOUNT already exists and is not locked."
            registerRollback
            echo_term
            echo_term "All changes has been rollbacked."
            exit
         fi
      fi
      echo_log "Blast server account $BLAST_ACCOUNT already exists."
   fi
}

extract_viewagent() {
   tar xfChp $VIEWAGENT_TAR $VIEWAGENT_PATH
   if [ "$?" != "0" ]; then
      echo_term "Extraction of horizon agent failed"
      exit
   fi

   cp -p $VIEWAGENT_DESKTOPC_SCRIPT /etc/xdg/autostart/
   chmod 644 /etc/xdg/autostart/$VIEWAGENT_DESKTOPC_SCRIPT

   # Under Ubuntu1804 KDE, xdg/autostart program will be started by systemd,
   # then desktopController will not exit when logout.
   # Move desktopController start point from autostart to plasma-workspace/env.
   if [ "${DISTRO_ID}" = ${DISTRO_ID_UBUNTU} ] &&
      [ "${DISTRO_RELEASE}" = "18.04" ]; then
      echo "NotShowIn=KDE" >> /etc/xdg/autostart/$VIEWAGENT_DESKTOPC_SCRIPT

      mkdir -p /etc/xdg/plasma-workspace/env/
      cat > /etc/xdg/plasma-workspace/env/desktopc.sh <<-EOF
#!/bin/sh
/usr/lib/vmware/viewagent/desktop/desktopController >/dev/null 2>&1 &
EOF
   fi

   cp -p vmware-lang.sh $LOCALE_PROFILE_PATH
   chown root:root ${LOCALE_PROFILE_PATH}/vmware-lang.sh
   chmod 644 ${LOCALE_PROFILE_PATH}/vmware-lang.sh

   cp -p kdcAuthLogin.conf $VIEWAGENT_PATH

   cp -p libunixdomainsocket-linux-*.so $VIEWAGENT_PATH
   chown root:root ${VIEWAGENT_PATH}/libunixdomainsocket-linux-i386.so
   chmod 555 ${VIEWAGENT_PATH}/libunixdomainsocket-linux-i386.so
   chown root:root ${VIEWAGENT_PATH}/libunixdomainsocket-linux-x86_64.so
   chmod 555 ${VIEWAGENT_PATH}/libunixdomainsocket-linux-x86_64.so

   if [ "$FIPS_ENABLE" = "true" ]; then
      cp -p java.security.txt $VIEWAGENT_PATH
      rm -f $VIEWAGENT_PATH/bcprov-jdk15on.jar
      rm -f $VIEWAGENT_PATH/bcpkix-jdk15on.jar
   else
      rm -f $VIEWAGENT_PATH/bc-fips.jar
      rm -f $VIEWAGENT_PATH/bcpkix-fips.jar
   fi

   cp -p Product.txt $VIEWAGENT_PATH
   cp -p open_source_licenses.txt $VIEWAGENT_PATH
   cp -rp resources $VIEWAGENT_PATH
   chown -R root:root $VIEWAGENT_PATH
   chmod -R 755 $VIEWAGENT_PATH/resources
   cp -rp bin $VIEWAGENT_PATH
   chown -R root:root $VIEWAGENT_PATH/bin
   chmod -R 550 $VIEWAGENT_PATH/bin
   cp -rp desktop $VIEWAGENT_PATH
   chown -R root:root $VIEWAGENT_DESKTOP_PATH
   chmod -R 755 $VIEWAGENT_DESKTOP_PATH
   chown vmwblast $VIEWAGENT_DESKTOP_PATH/desktopController
   chmod u+s $VIEWAGENT_DESKTOP_PATH/desktopController

   cp -rp DesktopDaemon $VIEWAGENT_PATH
   chown -R root:root $VIEWAGENT_PATH/DesktopDaemon
   chmod -R 550 $VIEWAGENT_PATH/DesktopDaemon

   #
   # Create a hard link that point to the bin/commonlib.sh and change the permisssion,
   # as it will be consumed by scripts those are not run as root.
   #
   ln $VIEWAGENT_PATH/bin/commonlib.sh $VIEWAGENT_PATH/commonlib.sh
   chmod 555 $VIEWAGENT_PATH/commonlib.sh

   uninstall_path="${VIEWAGENT_PATH}/bin"
   sed -i"" "s:__INSTALL_LOC__:${uninstall_path}:g" \
      ${uninstall_path}/uninstall_viewagent.sh
   [ "$?" != "0" ] && echo_term "Failed to patch uninstaller"
   mkdir -p $VMW_LOG_PATH
   chmod 755 $VMW_LOG_PATH
   mkdir -p $JMS_KEYSTORE_DIR
   chmod 750 $JMS_KEYSTORE_DIR
   mkdir -p $VIEWAGENT_RUNTIME_DIR
   chmod 700 $VIEWAGENT_RUNTIME_DIR
}


extract_jre() {
   cp -rp jre $VIEWAGENT_PATH
   chown -R root:root $VIEWAGENT_PATH/jre
   chmod -R 550 $VIEWAGENT_PATH/jre
}

extract_sc_sso() {
   local archpath="lib64"

   cp -p sso/vmwsccred.so /usr/$archpath/gdm/simple-greeter/plugins/vmwsccred.so
   chown root:root /usr/$archpath/gdm/simple-greeter/plugins/vmwsccred.so
   chmod 555 /usr/$archpath/gdm/simple-greeter/plugins/vmwsccred.so

   mkdir -p /usr/share/gdm/simple-greeter/extensions/vmwsccred
   chown -R root:root /usr/share/gdm/simple-greeter/extensions/vmwsccred
   chmod 755 /usr/share/gdm/simple-greeter/extensions/vmwsccred

   cp -p sso/page.ui /usr/share/gdm/simple-greeter/extensions/vmwsccred/page.ui
   chown root:root /usr/share/gdm/simple-greeter/extensions/vmwsccred/page.ui
   chmod 555 /usr/share/gdm/simple-greeter/extensions/vmwsccred/page.ui

   cp -p sso/gdm-vmwcred16.png /usr/share/icons/hicolor/16x16/apps/gdm-vmwsccred.png
   chown root:root /usr/share/icons/hicolor/16x16/apps/gdm-vmwsccred.png
   chmod 555 /usr/share/icons/hicolor/16x16/apps/gdm-vmwsccred.png

   cp -p sso/gdm-vmwcred48.png /usr/share/icons/hicolor/48x48/apps/gdm-vmwsccred.png
   chown root:root /usr/share/icons/hicolor/48x48/apps/gdm-vmwsccred.png
   chmod 555 /usr/share/icons/hicolor/48x48/apps/gdm-vmwsccred.png

   cp -p sso/gdm-vmwsccred /etc/pam.d
   chown root:root /etc/pam.d/gdm-vmwcred
   chmod 644 /etc/pam.d/gdm-vmwcred
}


config_truesso() {
   if [ "$TRUESSO_ENABLE" = "yes" ]; then
      cp -p sso/libvmwpkcs11.so $VIEWAGENT_PATH/sso
      chown root:root $VIEWAGENT_PATH/sso/libvmwpkcs11.so
      chmod 555 $VIEWAGENT_PATH/sso/libvmwpkcs11.so
      sed -i '3iauth        sufficient pam_krb5.so use_first_pass no_subsequent_prompt preauth_options=X509_user_identity=PKCS11:module_name=/usr/lib/vmware/viewagent/sso/libvmwpkcs11.so:token=Virtual_VMware_Slot' /etc/pam.d/gdm-vmwcred
   fi
}

config_truesso_rhel() {
   if [ "$TRUESSO_ENABLE" = "yes" ]; then
      cp -p sso/libvmwpkcs11.so $VIEWAGENT_PATH/sso
      chown root:root $VIEWAGENT_PATH/sso/libvmwpkcs11.so
      chmod 555 $VIEWAGENT_PATH/sso/libvmwpkcs11.so
      if [ "$OS_MAJOR" = "8" ]; then
         cp -p sso/rhel8/libvmwpkcs11.module /usr/share/p11-kit/modules
         chown root:root /usr/share/p11-kit/modules/libvmwpkcs11.module
         chmod 644 /usr/share/p11-kit/modules/libvmwpkcs11.module
         sed -i '3iauth        sufficient pam_sss.so require_cert_auth' /etc/pam.d/gdm-vmwcred
      else
         sed -i '3iauth        sufficient pam_krb5.so use_first_pass no_subsequent_prompt preauth_options=X509_user_identity=PKCS11:module_name=/usr/lib/vmware/viewagent/sso/libvmwpkcs11.so:token=Virtual_VMware_Slot' /etc/pam.d/gdm-vmwcred
      fi
   fi
}

config_sso_rhel() {
   local archpath="lib64"

   if [ "$OS_MAJOR" = "7" -o "$OS_MAJOR" = "8" ]; then
      local gnomeShellVer
      local gdmVMWCredPath="sso/gdm-vmwcred"
      local ssoDBusNotifyPath="sso/SsoDBusNotify.py"

      which gnome-shell > /dev/null 2>&1
      if [ "$?" != "0" ]; then
         echo_term "This system miss gnome-shell, doesn't support SSO"
         return
      fi
      gnomeShellVer=`gnome-shell --version | cut -d" " -f 3`

      # Backup original gnome-shell package
      if [ ! -d ${SSO_BACKUP_PATH} ]; then
         mkdir -p ${SSO_BACKUP_PATH}
      fi
      # Import RPM-GPG-KEY-redhat-release key
      rpm --import sso/RPM-GPG-KEY-redhat-release

      if [ "$OS_MAJOR" = "7" ]; then
         if [ "$gnomeShellVer" = "3.8.4" ]; then
            if [ ! -f "/usr/share/gnome-shell/js/gdm/authPrompt.js" ] ||
               [ ! -f "/usr/share/gnome-shell/js/gdm/util.js" ]; then
               echo_term "Failed to replace gnome-shell, disable sso"
               return
            fi
            cp -p sso/rhel7/authPrompt.js.orig ${SSO_BACKUP_PATH}
            cp -p sso/rhel7/util.js.orig ${SSO_BACKUP_PATH}
            cp -p sso/rhel7/authPrompt.js /usr/share/gnome-shell/js/gdm/authPrompt.js
            cp -p sso/rhel7/util.js /usr/share/gnome-shell/js/gdm/util.js
            cp -p sso/rhel7/vmCred.js /usr/share/gnome-shell/js/gdm/vmCred.js
         elif [ "$gnomeShellVer" = "3.14.4" ]; then
            if [ "$OS_MINOR" = "2" ]; then
               rpm -U --force sso/rhel7/gnome-shell-3.14.4-37.el7.centos.x86_64.rpm
               [ "$?" != "0" ] && echo_term "Failed to replace gnome-shell, disable sso" && return
               cp -p sso/rhel7/gnome-shell-3.14.4-37.el7.x86_64.orig.rpm ${SSO_BACKUP_PATH}
            else
               # rhel centos 7.3
               rpm -U --force sso/rhel7/gnome-shell-3.14.4-53.el7.centos.x86_64.rpm
               [ "$?" != "0" ] && echo_term "Failed to replace gnome-shell, disable sso" && return
               cp -p sso/rhel7/gnome-shell-3.14.4-53.el7.x86_64.orig.rpm ${SSO_BACKUP_PATH}
            fi
         elif [ "$gnomeShellVer" = "3.22.3" ]; then
            # rhel centos 7.4
            rpm -U --force sso/rhel7/gnome-shell-3.22.3-17.el7.redhat74.x86_64.rpm
            [ "$?" != "0" ] && echo_term "Failed to replace gnome-shell, disable sso" && return
            cp -p sso/rhel7/gnome-shell-3.22.3-17.el7.x86_64.orig.rpm ${SSO_BACKUP_PATH}
         elif [ "$gnomeShellVer" = "3.25.4" ] || [ "$gnomeShellVer" = "3.26.2" ]; then
            # rhel centos 7.5
            #
            # There is a bug for gnome-shell 3.26.2 package, 'gnome-shell --version'
            # output 3.25.4, not 3.26.2
            #
            rpm -U --force sso/rhel7/gnome-shell-3.26.2-5.el7.redhat75.x86_64.rpm
            [ "$?" != "0" ] && echo_term "Failed to replace gnome-shell, disable sso" && return
            cp -p sso/rhel7/gnome-shell-3.26.2-5.el7.x86_64.orig.rpm ${SSO_BACKUP_PATH}
         elif [ "$gnomeShellVer" = "3.28.3" ]; then
            # rhel centos 7.[6/7/8]
            rpm -U --force sso/rhel7/gnome-shell-3.28.3-6.el7.x86_64.rpm
            [ "$?" != "0" ] && echo_term "Failed to replace gnome-shell, disable sso" && return
            cp -p sso/rhel7/gnome-shell-3.28.3-6.el7.x86_64.orig.rpm ${SSO_BACKUP_PATH}
         else
            #
            # If there no any version match gnome-shell, we go back to check the
            # RHEL version and use the default version of gnome-shell.
            #
            echo_term "The version of gnome shell $gnomeShellVer isn't matched,\n
                       try to replace default version."
            if [ "$OS_MINOR" = "0" ] || [ $OS_MINOR = "1" ]; then
               if [ ! -f "/usr/share/gnome-shell/js/gdm/authPrompt.js" ] ||
                  [ ! -f "/usr/share/gnome-shell/js/gdm/util.js" ]; then
                  echo_term "Failed to replace gnome-shell, disable sso"
                  return
               fi
               cp -p sso/rhel7/authPrompt.js.orig ${SSO_BACKUP_PATH}
               cp -p sso/rhel7/util.js.orig ${SSO_BACKUP_PATH}
               cp -p sso/rhel7/authPrompt.js /usr/share/gnome-shell/js/gdm/authPrompt.js
               cp -p sso/rhel7/util.js /usr/share/gnome-shell/js/gdm/util.js
               cp -p sso/rhel7/vmCred.js /usr/share/gnome-shell/js/gdm/vmCred.js
            elif [ "$OS_MINOR" = "2" ]; then
               rpm -U --force sso/rhel7/gnome-shell-3.14.4-37.el7.centos.x86_64.rpm
               [ "$?" != "0" ] && echo_term "Failed to replace gnome-shell, disable sso" && return
               cp -p sso/rhel7/gnome-shell-3.14.4-37.el7.x86_64.orig.rpm ${SSO_BACKUP_PATH}
            elif [ "$OS_MINOR" = "3" ]; then
               rpm -U --force sso/rhel7/gnome-shell-3.14.4-53.el7.centos.x86_64.rpm
               [ "$?" != "0" ] && echo_term "Failed to replace gnome-shell, disable sso" && return
               cp -p sso/rhel7/gnome-shell-3.14.4-53.el7.x86_64.orig.rpm ${SSO_BACKUP_PATH}
            elif [ "$OS_MINOR" = "4" ]; then
               rpm -U --force sso/rhel7/gnome-shell-3.22.3-17.el7.redhat74.x86_64.rpm
               [ "$?" != "0" ] && echo_term "Failed to replace gnome-shell, disable sso" && return
               cp -p sso/rhel7/gnome-shell-3.22.3-17.el7.x86_64.orig.rpm ${SSO_BACKUP_PATH}
            elif [ "$OS_MINOR" = "5" ]; then
               rpm -U --force sso/rhel7/gnome-shell-3.26.2-5.el7.redhat75.x86_64.rpm
               [ "$?" != "0" ] && echo_term "Failed to replace gnome-shell, disable sso" && return
               cp -p sso/rhel7/gnome-shell-3.26.2-5.el7.x86_64.orig.rpm ${SSO_BACKUP_PATH}
            elif [ "$OS_MINOR" = "6" ]; then
               rpm -U --force sso/rhel7/gnome-shell-3.28.3-6.el7.x86_64.rpm
               [ "$?" != "0" ] && echo_term "Failed to replace gnome-shell, disable sso" && return
               cp -p sso/rhel7/gnome-shell-3.28.3-6.el7.x86_64.orig.rpm ${SSO_BACKUP_PATH}
            else
               echo_term "Don't support $DISTRO_RELEASE, disable sso"
               return
            fi
         fi
      else
         # latest RHEL 8.x version
         local GNOME_SHELL_SSO=''
         local GNOME_SHELL_ORIG=''
         if [ "$gnomeShellVer" = "3.32.2" ]; then
            # REHL8.1
            GNOME_SHELL_SSO='sso/rhel8/gnome-shell-3.32.2-9.el8.x86_64.rpm'
            GNOME_SHELL_ORIG='sso/rhel8/gnome-shell-3.32.2-9.el8.x86_64.orig.rpm'
         elif [ "$gnomeShellVer" = "3.28.3" ]; then
            # REHL8.0
            GNOME_SHELL_SSO='sso/rhel8/gnome-shell-3.28.3-10.el8.x86_64.rpm'
            GNOME_SHELL_ORIG='sso/rhel8/gnome-shell-3.28.3-10.el8.x86_64.orig.rpm'
         else
            echo_term "The version of gnome shell $gnomeShellVer isn't matched,\n
                       try to replace default version."
            case ${OS_MINOR} in
               0)
                  # RHEL 8.0
                  GNOME_SHELL_SSO='sso/rhel8/gnome-shell-3.28.3-10.el8.x86_64.rpm'
                  GNOME_SHELL_ORIG='sso/rhel8/gnome-shell-3.28.3-10.el8.x86_64.orig.rpm'
                  ;;
               1)
                  # RHEL 8.1
                  GNOME_SHELL_SSO='sso/rhel8/gnome-shell-3.32.2-9.el8.x86_64.rpm'
                  GNOME_SHELL_ORIG='sso/rhel8/gnome-shell-3.32.2-9.el8.x86_64.orig.rpm'
                  ;;
               *)
                  echo_term "Don't support $DISTRO_RELEASE, disable sso"
                  return;;
            esac
         fi
         rpm -U --force ${GNOME_SHELL_SSO}
         [ "$?" != "0" ] && echo_term "Failed to replace gnome-shell, disable sso" && return
         cp -p ${GNOME_SHELL_ORIG} ${SSO_BACKUP_PATH}

         gdmVMWCredPath="sso/rhel8/gdm-vmwcred"
         ssoDBusNotifyPath="sso/SsoDBusNotify3.py"

         if [ $SINGLETON_MODE = "false" ]; then
            cp -p sso/vmware-greeter-check.desktop /usr/share/gdm/greeter/autostart/
            chown root:root /usr/share/gdm/greeter/autostart/vmware-greeter-check.desktop
            chmod 644 /usr/share/gdm/greeter/autostart/vmware-greeter-check.desktop
         fi
      fi

      cp -p sso/pam_vmw_cred.so /$archpath/security
      chown root:root /$archpath/security/pam_vmw_cred.so
      chmod 555 /$archpath/security/pam_vmw_cred.so

      cp -p $gdmVMWCredPath /etc/pam.d
      chown root:root /etc/pam.d/gdm-vmwcred
      chmod 644 /etc/pam.d/gdm-vmwcred
      # Config TrueSSO
      config_truesso_rhel

      # Create the dbus conf file
      cp -p sso/org.vmware.viewagent.Credentials.conf $DBUS_CONF_PATH
      chown root:root $DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf
      chmod 644 $DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf

      # Create SSO notify file
      cp -p $ssoDBusNotifyPath $VIEWAGENT_PATH/bin/SsoDBusNotify.py
      chown root:root $SSO_DBUS_PY
      chmod 550 $SSO_DBUS_PY

      if [ ! -d $VIEWAGENT_TMP_ROOT ]; then
         mkdir -p -m 1777 $VIEWAGENT_TMP_ROOT
      fi
   fi

   if [ "$OS_MAJOR" = "6" ]; then
      gdm_binary_pid=`pidof gdm-binary` > /dev/null 2>&1
      if [ "$?" != "0" ]; then
         echo_term "SSO is supported only when using gdm-simple-greeter in GDM."
         return
      fi
      cp -p sso/vmwcred.so /usr/$archpath/gdm/simple-greeter/plugins/vmwcred.so
      chown root:root /usr/$archpath/gdm/simple-greeter/plugins/vmwcred.so
      chmod 555 /usr/$archpath/gdm/simple-greeter/plugins/vmwcred.so

      mkdir -p /usr/share/gdm/simple-greeter/extensions/vmwcred
      chown -R root:root /usr/share/gdm/simple-greeter/extensions/vmwcred
      chmod 755 /usr/share/gdm/simple-greeter/extensions/vmwcred

      cp -p sso/page.ui /usr/share/gdm/simple-greeter/extensions/vmwcred/page.ui
      chown root:root /usr/share/gdm/simple-greeter/extensions/vmwcred/page.ui
      chmod 555 /usr/share/gdm/simple-greeter/extensions/vmwcred/page.ui

      cp -p sso/gdm-vmwcred16.png /usr/share/icons/hicolor/16x16/apps/gdm-vmwcred.png
      chown root:root /usr/share/icons/hicolor/16x16/apps/gdm-vmwcred.png
      chmod 555 /usr/share/icons/hicolor/16x16/apps/gdm-vmwcred.png

      cp -p sso/gdm-vmwcred48.png /usr/share/icons/hicolor/48x48/apps/gdm-vmwcred.png
      chown root:root /usr/share/icons/hicolor/48x48/apps/gdm-vmwcred.png
      chmod 555 /usr/share/icons/hicolor/48x48/apps/gdm-vmwcred.png

      cp -p sso/pam_vmw_cred.so /$archpath/security
      chown root:root /$archpath/security/pam_vmw_cred.so
      chmod 555 /$archpath/security/pam_vmw_cred.so

      cp -p sso/gdm-vmwcred /etc/pam.d
      chown root:root /etc/pam.d/gdm-vmwcred
      chmod 644 /etc/pam.d/gdm-vmwcred

      # Create the dbus conf file
      cp -p sso/org.vmware.viewagent.Credentials.conf $DBUS_CONF_PATH
      chown root:root $DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf
      chmod 644 $DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf

      cp -p sso/SsoDBusNotify.py $VIEWAGENT_PATH/bin
      chown root:root $SSO_DBUS_PY
      chmod 550 $SSO_DBUS_PY

      if [ ! -d $VIEWAGENT_TMP_ROOT ]; then
         mkdir -p -m 1777 $VIEWAGENT_TMP_ROOT
      fi
   fi
}

config_sso_suse() {
   local archpath="lib64"
   local gdm_simple_greeter_status=0
   if [ "$DISTRO_RELEASE" = "11" ]; then
      gdm_simple_greeter_pid="`pidof gdm-simple-greeter`"
      if [ "$?" = "0" ]; then
         # If install view via ssh, need stop gdm first
         # Because we need replace gdm-simple-greeter
         gdm-stop
         gdm_simple_greeter_status=1
         while [ -n "`pidof gdm`" ]
         do
            sleep 1
         done
      fi
      # Backup gdm-simple-greeter
      if [ ! -d ${SSO_BACKUP_PATH} ]; then
         mkdir -p ${SSO_BACKUP_PATH}
      fi
      # Backup the original gdm-simple-greeter used by SLED
      # It will replace gdm-simple-greeter with SSO feature when uninstall View
      cp -p sso/sled11/gdm-simple-greeter.orig ${SSO_BACKUP_PATH}
      # Replace original gdm-simple-greeter with gdm-simple-greeter with SSO feature
      cp -p sso/sled11/gdm-simple-greeter /usr/lib/gdm
      chown root:root /usr/lib/gdm/gdm-simple-greeter
      chmod 755 /usr/lib/gdm/gdm-simple-greeter

      # If install view via ssh, start gdm process after replace gdm-simple-greeter
      if [ "${gdm_simple_greeter_status}" = "1" ]; then
         /usr/bin/gdm &
      fi

      # PAM module
      cp -p sso/sled11/pam_vmw_cred_sled11.so /$archpath/security/pam_vmw_cred.so
      chown root:root /$archpath/security/pam_vmw_cred.so
      chmod 555 /$archpath/security/pam_vmw_cred.so

      # Backup original gdm PAM config file
      # It will replace gdm PAM config file with SSO feature when uninstall View
      cp -p sso/sled11/gdm.orig ${SSO_BACKUP_PATH}
      # Repalce original gdm PAM config file with the one with SSO feature.
      cp -p sso/sled11/gdm-vmwcred /etc/pam.d/gdm
      chown root:root /etc/pam.d/gdm
      chmod 644 /etc/pam.d/gdm

      # Create the dbus conf file
      cp -p sso/org.vmware.viewagent.Credentials.conf $DBUS_CONF_PATH
      chown root:root $DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf
      chmod 644 $DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf

      # SsoDBusNotify
      cp -p sso/SsoDBusNotify.py $VIEWAGENT_PATH/bin
      chown root:root $SSO_DBUS_PY
      chmod 550 $SSO_DBUS_PY

      if [ ! -d $VIEWAGENT_TMP_ROOT ]; then
         mkdir -p -m 1777 $VIEWAGENT_TMP_ROOT
      fi
   fi

   if [ "$OS_MAJOR" = "12" ]; then
      if [ "$DISTRO_RELEASE" = "12.3" -o "$DISTRO_RELEASE" = "12.5" ]; then
         rpm -U --force sso/rhel7/gnome-shell-3.20.4-76.3.x86_64.SP3.rpm
      elif [ "$DISTRO_RELEASE" = "12.2" ]; then
         rpm -U --force sso/rhel7/gnome-shell-3.20.4-70.4.x86_64.SP2.rpm
      elif [ "$DISTRO_RELEASE" = "12.1" ]; then
         rpm -U --force sso/rhel7/gnome-shell-3.10.4-40.1.x86_64.SP1.rpm
      fi

      [ "$?" != "0" ] && echo_term "Failed to replace gnome-shell, disable sso" && return

      # Backup original gnome-shell package
      if [ ! -d ${SSO_BACKUP_PATH} ]; then
         mkdir -p ${SSO_BACKUP_PATH}
      fi

      if [ "$DISTRO_RELEASE" = "12.3" -o "$DISTRO_RELEASE" = "12.5" ]; then
         cp -p sso/rhel7/gnome-shell-3.20.4-76.3.x86_64.orig.rpm ${SSO_BACKUP_PATH}
      elif [ "$DISTRO_RELEASE" = "12.2" ]; then
         cp -p sso/rhel7/gnome-shell-3.20.4-70.4.x86_64.orig.rpm ${SSO_BACKUP_PATH}
      elif [ "$DISTRO_RELEASE" = "12.1" ]; then
         cp -p sso/rhel7/gnome-shell-3.10.4-40.1.x86_64.orig.rpm ${SSO_BACKUP_PATH}
      fi

      cp -p sso/pam_vmw_cred.so /$archpath/security
      chown root:root /$archpath/security/pam_vmw_cred.so
      chmod 555 /$archpath/security/pam_vmw_cred.so

      # Reuse Sled11 gdm-vmwcred
      cp -p sso/sled11/gdm-vmwcred /etc/pam.d
      chown root:root /etc/pam.d/gdm-vmwcred
      chmod 644 /etc/pam.d/gdm-vmwcred

      # Config TrueSSO
      config_truesso

      # Create the dbus conf file
      cp -p sso/org.vmware.viewagent.Credentials.conf $DBUS_CONF_PATH
      chown root:root $DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf
      chmod 644 $DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf

      # SsoDBusNotify
      cp -p sso/SsoDBusNotify.py $VIEWAGENT_PATH/bin
      chown root:root $SSO_DBUS_PY
      chmod 550 $SSO_DBUS_PY

      if [ ! -d $VIEWAGENT_TMP_ROOT ]; then
         mkdir -p -m 1777 $VIEWAGENT_TMP_ROOT
      fi
   fi
}

config_truesso_ubuntu() {
   if [ "$TRUESSO_ENABLE" = "yes" ]; then
      cp -p sso/libvmwpkcs11.so $VIEWAGENT_PATH/sso
      chown root:root $VIEWAGENT_PATH/sso/libvmwpkcs11.so
      chmod 555 $VIEWAGENT_PATH/sso/libvmwpkcs11.so
      if [ "$OS_MAJOR" = "18"  ]; then
         sed -i '3iauth        sufficient pam_pkcs11.so try_first_pass' /etc/pam.d/gdm-vmwcred
      else
         sed -i '3iauth        sufficient pam_pkcs11.so try_first_pass' /etc/pam.d/lightdm
      fi
   fi
}

config_sso_ubuntu() {
   local gdm_greeter_status=0
   local unity_greeter_status=0
   local archpath="lib/x86_64-linux-gnu"
   local count=0
   local ret=0

   if [ "$OS_MAJOR" = "14" ] || [ "$OS_MAJOR" = "16" ] || [ "$OS_MAJOR" = "18" ]; then
      if [ "$OS_MAJOR" = "18"  ]; then
         which gnome-shell > /dev/null 2>&1
         if [ "$?" != "0" ]; then
            echo_term "Gnome-shell is missing on this system, sso feature will not be supported."
            return
         fi
      else
         greeter_session=`lightdm --show-config 2>&1 | awk -F= '/greeter-session=/{print $2}'` > /dev/null 2>&1
         if [ "$?" != "0" ] || [ "$greeter_session" != "unity-greeter" ]; then
            echo_term "SSO is supported only when using unity-greeter in LightDM."
            return
         fi
      fi

      if [ "$OS_MAJOR" = "14" ]; then
         sso_config_path=sso/ubuntu/1404
      elif [ "$OS_MAJOR" = "16" ]; then
         sso_config_path=sso/ubuntu/1604
      fi

      if [ ! -d $VIEWAGENT_TMP_ROOT ]; then
         mkdir -p -m 1777 $VIEWAGENT_TMP_ROOT
      fi

      if [ ! -d ${SSO_BACKUP_PATH} ]; then
         mkdir -p ${SSO_BACKUP_PATH}
      fi

      if [ "$OS_MAJOR" = "18" ]; then
         local gnomeShellVer=
         local gnome_shell_pid="`pidof gnome-shell`"
         local gnome_shell_pid_nr=`echo $gnome_shell_pid|awk '{print NF}'`
         if [ "$gnome_shell_pid_nr" = "1" ]; then
            userid=`ps -o uid --no-headers -p $gnome_shell_pid`
            username=`getent passwd $userid|cut -d: -f1`
            [ -n "$username" ] && [ "$username" = "gdm" ] && gdm_greeter_status=1
         fi

         if [ "$gdm_greeter_status" = "1" ]; then
            # If install view via ssh, better to stop gdm first
            service gdm3 stop
            ret=$?
            while [ -n "`pidof gdm3`" ]
            do
               [ "$ret" != "0" ] && service gdm3 stop
               ret=$?

               sleep 1
            done
         fi

         gnomeShellVer=`gnome-shell --version | cut -d" " -f 3`
         if [ "$?" = "0" ]; then
            if [ "$gnomeShellVer" = "3.28.1" ] ||     \
               [ "$gnomeShellVer" = "3.28.3" ] ||     \
               [ "$gnomeShellVer" = "3.28.4" ]; then
               cp -p sso/ubuntu/1804/$gnomeShellVer/libgnome-shell.orig.so ${SSO_BACKUP_PATH}/libgnome-shell.orig.so
            else
               echo_term "Doesn't support libgnome-shell.so with V$gnomeShellVer for sso feature, replace libgnome-shell.so with V3.28.4 instead."
               gnomeShellVer="3.28.4"
               cp -p /usr/lib/gnome-shell/libgnome-shell.so ${SSO_BACKUP_PATH}/libgnome-shell.orig.so
            fi

            install sso/ubuntu/1804/$gnomeShellVer/libgnome-shell.so /usr/lib/gnome-shell/ >/dev/null 2>&1
            [ "$?" != "0" ] && echo_term "Failed to replace libgnome-shell.so, disable sso." && return
            chmod 644 /usr/lib/gnome-shell/libgnome-shell.so ${SSO_BACKUP_PATH}/libgnome-shell.orig.so
         else
            echo_term "Failed to get the version of gnome-shell, disable sso."
            return
         fi
      else
         unity_greeter_pid="`pidof unity-greeter`"
         if [ "$?" = "0" ]; then
            # If install view via ssh, need stop lightdm first
            # Because we need replace unity-greeter
            service lightdm stop
            unity_greeter_status=1
            while [ -n "`pidof lightdm`" ]
            do
               sleep 1
            done
         fi

         # Backup original unity-greeter
         cp -p ${sso_config_path}/unity-greeter.orig ${SSO_BACKUP_PATH}
         # Replace original unity-greeter with unity-greeter with SSO feature
         cp -p ${sso_config_path}/unity-greeter /usr/sbin/unity-greeter > /dev/null 2>&1
         ret=$?
         #
         # Sometimes "Text file busy" leads to unity-greeter installation failed, maybe
         # unity-greeter doesn't finish the stop process, need to retry.
         #
         while [ "$ret" != "0" -a "$count" -lt "5" ]
         do
            sleep 1
            echo_log "Installer is waiting unity-greeter to stop"
            cp -p ${sso_config_path}/unity-greeter /usr/sbin/unity-greeter > /dev/null 2>&1
            ret=$?
            count=`expr $count + 1`
         done

         if [ "$count" -ge "5" ]; then
            echo_term "Failed to install SSO supported unity-greeter, please stop lightdm manually and retry"
            exit
         fi
         chown root:root /usr/sbin/unity-greeter
         chmod 755 /usr/sbin/unity-greeter
      fi

      # SSO PAM module
      if [ "$OS_MAJOR" = "18"  ]; then
         cp -p sso/pam_vmw_cred.so /$archpath/security
      else
         cp -p sso/ubuntu/pam_vmw_cred_ubuntu.so /$archpath/security/pam_vmw_cred.so
      fi
      chown root:root /$archpath/security/pam_vmw_cred.so
      chmod 555 /$archpath/security/pam_vmw_cred.so

      # Null PAM module, this module will always return true
      cp -p sso/ubuntu/pam_vmw_null.so /$archpath/security/pam_vmw_null.so
      chown root:root /$archpath/security/pam_vmw_null.so
      chmod 555 /$archpath/security/pam_vmw_null.so

      if [ "$OS_MAJOR" = "18"  ]; then
         cp -p sso/ubuntu/gdm-vmwcred /etc/pam.d
         chown root:root /etc/pam.d/gdm-vmwcred
         chmod 644 /etc/pam.d/gdm-vmwcred

         # Add null PAM module to common-auth and update PAM authentication
         cp -p sso/ubuntu/pam-config-vmw-null /usr/share/pam-configs/vmw-null
         chown root:root /usr/share/pam-configs/vmw-null >/dev/null 2>&1
         chmod 644 /usr/share/pam-configs/vmw-null >/dev/null 2>&1
      else
         # Original lightdm has been backuped in backup_pam()
         # It will replace lightdm PAM config file with SSO feature when uninstall View

         # Repalce original lightdm PAM config file with the one with SSO feature.
         cp -p sso/ubuntu/lightdm-vmwcred /etc/pam.d/lightdm
         chown root:root /etc/pam.d/lightdm
         chmod 644 /etc/pam.d/lightdm

         # Backup and update pam config unix file to add try_first_pass, it will be used to
         # update common-auth to let both sso and smartcard work at Ubuntu1404 and 1604
         PAM_CONFIG_UNIX_FILE="/usr/share/pam-configs/unix"
         [ ! -d ${SSO_BACKUP_PATH} ] && mkdir -p ${SSO_BACKUP_PATH}
         cp -p ${PAM_CONFIG_UNIX_FILE} ${SSO_BACKUP_PATH}
         sed -i '/^Priority/c Priority: 512' ${PAM_CONFIG_UNIX_FILE}
         sed -i '/pam_unix.so nullok_secure$/s/nullok_secure/nullok_secure try_first_pass/' ${PAM_CONFIG_UNIX_FILE}
      fi

      # Configure trueSSO
      config_truesso_ubuntu

      # Add mkhomedir module to common-session and update PAM authentication
      cp -p sso/ubuntu/pam-config-vmw-mkhomedir /usr/share/pam-configs/vmw-mkhomedir
      chown root:root /usr/share/pam-configs/vmw-mkhomedir
      chmod 644 /usr/share/pam-configs/vmw-mkhomedir
      pam-auth-update --package --force >/dev/null 2>&1

      # Create the dbus conf file
      if [ "$OS_MAJOR" = "18"  ]; then
         cp -p sso/org.vmware.viewagent.Credentials.conf $DBUS_CONF_PATH
      else
         cp -p sso/ubuntu/org.vmware.viewagent.Credentials.conf $DBUS_CONF_PATH
      fi
      chown root:root $DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf
      chmod 644 $DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf

      # SsoDBusNotify
      cp -p sso/SsoDBusNotify.py $VIEWAGENT_PATH/bin
      chown root:root $SSO_DBUS_PY
      chmod 550 $SSO_DBUS_PY

      if [ "$OS_MAJOR" = "18"  ]; then
         if [ "$gdm_greeter_status" = "1" ]; then
            service gdm3 start
         fi
      else
         # If install view via ssh, start lightdm process after replace unity-greeter
         if [ "${unity_greeter_status}" = "1" ]; then
            service lightdm start
         fi
      fi
   fi
}


extract_sso() {
   local arch="`uname -i`"

   # Don't support 32 bit linux system
   [ "$arch" != "x86_64" ] && return

   umask 022

   case "$DISTRO_ID" in
   ${DISTRO_ID_CENTOS}|\
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_RHEL_WORKSTATION})
      config_sso_rhel
      ;;
   ${DISTRO_ID_SUSE})
      config_sso_suse
      ;;
   ${DISTRO_ID_UBUNTU})
      config_sso_ubuntu
      ;;
   *)
      return
   esac

   if [ $SINGLETON_MODE = "false" ]; then
      cp -p sso/vmware-greeter-check.sh ${VIEWAGENT_PATH}/sso
      chown root:root ${VIEWAGENT_PATH}/sso/vmware-greeter-check.sh
      chmod 555 ${VIEWAGENT_PATH}/sso/vmware-greeter-check.sh
   fi
}

extract_cdrserver() {
   if [ ! -d ${VCHAN_PLUGINS_PATH} ]; then
      mkdir -p ${VCHAN_PLUGINS_PATH}
   fi

   cp -p cdrserver/cdrserver ${VCHAN_PLUGINS_PATH}
   chown -R root:root ${VCHAN_PLUGINS_PATH}
   chmod -R 755 ${VCHAN_PLUGINS_PATH}
}

extract_usbredir() {
   if [ ! -d ${VCHAN_PLUGINS_PATH} ]; then
      mkdir -p ${VCHAN_PLUGINS_PATH}
   fi

   cp -pf usbRedirServer/* ${VCHAN_PLUGINS_PATH}
   chown -R root:root ${VCHAN_PLUGINS_PATH}
   chmod -R 755 ${VCHAN_PLUGINS_PATH}

   # config udev rules to support usb hid device redirection.
   if [ -d "$UDEV_RULES_PATH" ]; then
      local input_rules="$VIEWAGENT_PATH/resources/udev/99-vmw-usb-input.rules"
      if ! grep -q systemd /proc/1/comm; then
          input_rules="$VIEWAGENT_PATH/resources/udev/99-vmw-usb-input-nonseat.rules"
      fi
      cp -fp "$input_rules"  "$UDEV_RULES_PATH/99-vmw-usb-input.rules"
      chown root $UDEV_RULES_PATH/99-vmw-usb-input.rules
      chmod 644 $UDEV_RULES_PATH/99-vmw-usb-input.rules
   fi
}

config_scredir_ubuntu() {
   local archpath="lib/x86_64-linux-gnu"

   cp -p scredir/pam_vmw_setenv.so /$archpath/security
   chown root:root /$archpath/security/pam_vmw_setenv.so
   chmod 555 /$archpath/security/pam_vmw_setenv.so

   if [ ${OS_MAJOR} = "16" ]; then
      # Original lightdm has been backuped in backup_pam()

      grep "pam_vmw_cred" /etc/pam.d/lightdm >/dev/null 2>&1
      if [ "$?" = "0" ]; then
         sed -i '2iauth sufficient pam_pkcs11.so' /etc/pam.d/lightdm
      fi

      sed -i '1 iauth required pam_vmw_setenv.so' /etc/pam.d/unity
   elif [ ${OS_MAJOR} = "18" ]; then
      lineNum=`grep -n -m 1 pam_pkcs11.so /etc/pam.d/gdm-password | cut -d : -f 1`
      sed -i "${lineNum} iauth required pam_vmw_setenv.so" /etc/pam.d/gdm-password
   fi
}


config_scredir_redhat() {
   local archpath="/lib64/"

   if [ ${OS_MAJOR} -ge "7" ]; then
       archpath="/usr"${archpath}
   fi

   cp -p scredir/pam_vmw_setenv.so /$archpath/security
   chown root:root /$archpath/security/pam_vmw_setenv.so
   chmod 555 /$archpath/security/pam_vmw_setenv.so

   if [ ${OS_MAJOR} = "6" ]; then
      # Add pam_vmw_setenv.so at the begining of gdm-smartcard
      sed -i '1 iauth required pam_vmw_setenv.so' /etc/pam.d/gdm-smartcard

      # Insert pam_vmw_setenv before the first system-auth in /etc/pam.d/gnome-screensaver
      lineNum=`grep -n -m 1 system-auth /etc/pam.d/gnome-screensaver | cut -d : -f 1`
      sed -i "${lineNum} iauth required pam_vmw_setenv.so" /etc/pam.d/gnome-screensaver
   elif [ ${OS_MAJOR} = "7" ]; then
      # Add pam_vmw_setenv.so before pam_pkcs11.so in /etc/pam.d/smartcard-auth
      lineNum=`grep -n -m 1 pam_pkcs11.so /etc/pam.d/smartcard-auth | cut -d : -f 1`
      sed -i "${lineNum} iauth required pam_vmw_setenv.so" /etc/pam.d/smartcard-auth
   fi
}


config_scredir_suse() {
   local archpath="/lib64/"

   cp -p scredir/pam_vmw_setenv.so /$archpath/security
   chown root:root /$archpath/security/pam_vmw_setenv.so
   chmod 555 /$archpath/security/pam_vmw_setenv.so

   lineNum=`grep -n -m 1 common-auth-smartcard /etc/pam.d/gdm-password | cut -d : -f 1`
   sed -i "${lineNum} iauth required pam_vmw_setenv.so" /etc/pam.d/gdm-password
}


extract_scredir() {
   local count=0;
   local ret=0;
   local pcscd_bin="pcscd_188";
   local scrediragent_bin="libscrediragent_188.so";
   local meet_sc_sso="no";

   echo_log "Begin to install smartcard redirection module"

   if [ "${DISTRO_ID}" = "${DISTRO_ID_RHEL_WORKSTATION}" ] &&
      [ "${OS_MAJOR}" = "6" ]; then
      pcscd_bin="pcscd"
      scrediragent_bin="libscrediragent.so"
      meet_sc_sso="yes"
   fi

   #
   # Backup the pcsc daemon binary
   #
   mkdir -p ${SCREDIR_BACKUP_DIR}
   chown -R root:root ${SCREDIR_BACKUP_DIR}
   cp /usr/sbin/pcscd ${SCREDIR_BACKUP_DIR}


   if [ "$DISTRO_ID" = ${DISTRO_ID_UBUNTU} ]; then
      config_scredir_ubuntu
   elif [ "$DISTRO_ID" = ${DISTRO_ID_RHEL_WORKSTATION} ]; then
      config_scredir_redhat
   elif [ "$DISTRO_ID" = "${DISTRO_ID_SUSE}" ]; then
      config_scredir_suse
   fi

   #
   # Extract the pcscd binary
   #
   service pcscd status > /dev/null 2>&1
   if [ $? = "0" ]; then
      service pcscd stop > /dev/null 2>&1
   fi
   install -D scredir/${pcscd_bin} /usr/sbin/pcscd

   #
   # Extract the VVC plugin directory
   #
   install -D scredir/${scrediragent_bin} ${SCREDIR_VCHAN_PLUGINS_PATH}/libscrediragent.so
   chown -R root:root ${SCREDIR_VCHAN_PLUGINS_PATH}
   chmod -R 755 ${SCREDIR_VCHAN_PLUGINS_PATH}

   chkconfig pcscd on > /dev/null 2>&1
   service pcscd start > /dev/null 2>&1

   #
   # Install Smartcard SSO
   #
   if [ "${meet_sc_sso}" = "yes" ]; then
      extract_sc_sso
   fi
}

extract_audioin() {
   #
   # Extract the VVC plugin directory
   #
   if [ ! -d ${AUDIOIN_VCHAN_PLUGINS_PATH} ]; then
      mkdir -p ${AUDIOIN_VCHAN_PLUGINS_PATH}
   fi

   cp -p audioin/libviewMMDevRedir.so ${AUDIOIN_VCHAN_PLUGINS_PATH}
   chown -R root:root ${AUDIOIN_VCHAN_PLUGINS_PATH}
   chmod -R 755 ${AUDIOIN_VCHAN_PLUGINS_PATH}
}

extract_vhci() {
   local kernel_release="`uname -r`"
   local modules_path=
   local ret=0
   local vermagic=

   case "$DISTRO_ID" in
   ${DISTRO_ID_CENTOS}|\
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_RHEL_WORKSTATION})
      modules_path=resources/vhci/centos/${OS_MAJOR}${OS_MINOR}
      ;;
   ${DISTRO_ID_SUSE})
      modules_path=resources/vhci/suse/${OS_MAJOR}${OS_MINOR}
      ;;
   ${DISTRO_ID_UBUNTU})
      modules_path=resources/vhci/ubuntu/${OS_MAJOR}${OS_MINOR}
      ;;
   *)
      return 1
   esac

   vermagic="`modinfo -F"vermagic" $modules_path/usb-vhci-iocifc.ko | cut -d" " -f1`"

   #check if vhci module match the kernel
   if [ "$vermagic" != "$kernel_release" ]; then
      return 1
   fi

   mkdir -p /lib/modules/$kernel_release/kernel/drivers/usb/host && \
   cp -f $modules_path/usb-vhci-hcd.ko $modules_path/usb-vhci-iocifc.ko /lib/modules/$kernel_release/kernel/drivers/usb/host && \
   depmod -a $kernel_release  > /dev/null 2>&1
   if [ "$?" != "0" ]; then
      return 1
   fi

   echo_log "Install vhci kernel modules successfully"
   return 0
}

extract_vncserver() {
   local oldlib=""
   local oldhash=""
   local newlib=""
   local newhash=""

   echo_log "Begin to check the software x264 encode library"

   #
   # Only copy the encoder library on
   # 1) Fresh install;
   # 2) The new library has a differet hash value.
   #
   newlib=`ls VMwareBlastServer/libx264.so.*`
   oldlib="$VIEWAGENT_PATH/$newlib"

   # Fresh install
   if [ ! -e "$oldlib" ]; then
      cp -rp VMwareBlastServer $VIEWAGENT_PATH
   else
      # Upgrade
      oldhash=`md5sum "$oldlib" | cut -d " " -f 1`
      newhash=`md5sum "$newlib" | cut -d " " -f 1`

      # Compare hash
      if [ "$oldhash" != "$newhash" ]; then
         cp -rp VMwareBlastServer $VIEWAGENT_PATH
      else
         find VMwareBlastServer/* ! -name libx264* -exec cp -rp {} $VIEWAGENT_PATH/VMwareBlastServer/ \;
     fi
   fi

   chown -R root:root $VIEWAGENT_PATH/VMwareBlastServer
   chmod -R 755 $VIEWAGENT_PATH/VMwareBlastServer

   if [ ! -f "$CERT" ]; then
      #
      # Install SSL certificate
      #
      [ -f "$PUBLIC_KEY" ] && rm -f $PUBLIC_KEY

      mkdir -p $VMW_SSL_PATH
      chown -R root:root $VMW_SSL_PATH
      chmod 550 $VMW_SSL_PATH
      # Generate private key and certificate request, then self sign the
      # certificate.
      openssl req -nodes -newkey rsa:$PRIVATE_KEY_SIZE -keyout $PRIVATE_KEY \
                  -out $CERT_REQ -subj "$DN" > /dev/null 2>&1
      if [ "$?" != "0" ]; then
         echo_term "Generation of private key or certificate request failed"
         registerRollback
         echo_term
         echo_term "All changes has been rollbacked."
         exit
      fi
      openssl x509 -req -sha256 -days $CERT_VALIDITY_PERIOD -in $CERT_REQ  \
                   -signkey $PRIVATE_KEY -out $CERT > /dev/null 2>&1
      if [ "$?" != "0" ]; then
         echo_term "Generation of self signed certificate failed, the return value is $?"
         registerRollback
         echo
         echo_term "All changes has been rollbacked."
         exit
      fi
      rm -f $CERT_REQ
      chmod 440 $VMW_SSL_PATH/*
   fi

   if [ ! -f "$PUBLIC_KEY" ]; then
      #
      # Extract public key
      #
      mkdir -p $PUBLIC_KEY_PATH
      chown -R root:root $PUBLIC_KEY_PATH
      chmod 755 $PUBLIC_KEY_PATH

      openssl x509 -pubkey -noout -in $CERT > $PUBLIC_KEY
      if [ "$?" != "0" ]; then
         echo_term "Failed to extract public key"
         registerRollback
         echo_term "All changes has been rollbacked."
         exit
      fi
      chmod 444 $PUBLIC_KEY
   fi
}

extract_help_desk() {
   if [ ! -d ${HELPDESK_PATH} ]; then
      mkdir -p ${HELPDESK_PATH}
   fi

   cp -pf helpdesk/libhelpdesk.so ${HELPDESK_PATH}
   chown -R root:root ${HELPDESK_PATH}
   chmod -R 755 ${HELPDESK_PATH}
}

extract_session_collaboration() {
   cssFileName=""

   # Support Ubuntu1804, Redhat7.5/7.6/7.7/7.8 and Redhat8.0/8.1
   if [ "${DISTRO_ID}" = ${DISTRO_ID_UBUNTU} ]; then
      if [ ${DISTRO_RELEASE} != "18.04" ]; then
         echo_log "Bypass install Collaboration since currently only supports Ubuntu1804"
         return
      else
         cssFileName="collabui_ubuntu"
      fi
   elif [ "${DISTRO_ID}" = ${DISTRO_ID_RHEL_WORKSTATION} ]; then
      case ${DISTRO_RELEASE} in
         7.[5678] | 8.[012])
            cssFileName="collabui_redhat";;
         *)
            echo_log -n "Bypass install Collaboration since currently only supports Redhat7.5, Redhat7.6, Redhat7.7, "
            echo_log "Redhat7.8, Redhat8.0, Redhat8.1, Redhat8.2"
            return;;
      esac
   else
      echo_log "Bypass install Collaboration since currently only support Ubuntu and Redhat"
      return
   fi

   cssFileName=${cssFileName}${DISTRO_RELEASE}".css"
   if [ "${DISTRO_ID}" = ${DISTRO_ID_RHEL_WORKSTATION} ] &&
      [ "${DISTRO_RELEASE}" = "7.6" -o "${DISTRO_RELEASE}" = "7.7" -o \
        "${DISTRO_RELEASE}" = "7.8" -o \
        "${DISTRO_RELEASE}" = "8.0" -o "${DISTRO_RELEASE}" = "8.1" -o \
        "${DISTRO_RELEASE}" = "8.2" ]; then
      cssFileName="collabui_redhat7.5.css"
   fi

   COLLAB_ENABLE="yes"
   echo_log "Start install Session Collaboration"

   # Extract UI icons
   if [ ! -d ${AGENT_ICON_PATH} ]; then
      mkdir -p ${AGENT_ICON_PATH}
   fi
   chmod -R 655 ${AGENT_ICON_PATH}/

   if [ ! -d ${COLLABORATION_ICON_PATH} ]; then
      mkdir -p ${COLLABORATION_ICON_PATH}
   fi
   chmod -R 655 ${COLLABORATION_ICON_PATH}/

   cp -r icons/collabui/* ${COLLABORATION_ICON_PATH}

   # Update tray icon location
   mv ${COLLABORATION_ICON_PATH}/${COLLABORATION_TRAY_ICON} /usr/share/pixmaps/

   chmod 644 /usr/share/pixmaps/${COLLABORATION_TRAY_ICON}

   # Update CSS file base on distribution
   mv ${COLLABORATION_ICON_PATH}/${cssFileName} ${COLLABORATION_ICON_PATH}/collabui.css
   rm -rf ${COLLABORATION_ICON_PATH}/collabui_*.css

   #
   # Extract UI program
   #
   if [ ! -d ${COLLABUI_PATH} ]; then
      mkdir -p ${COLLABUI_PATH}
   fi

   cp -p collabui/collabui ${COLLABUI_PATH}
   chown -R root:root ${COLLABUI_PATH}
   chmod -R 755 ${COLLABUI_PATH}

   for supportLang in 'de' 'es' 'fr' 'ja' 'ko' 'zh_CN' 'zh_TW';
   do
      cp l10n/${supportLang}/collabui.mo /usr/share/locale/${supportLang}/LC_MESSAGES;
      chmod 644 /usr/share/locale/${supportLang}/LC_MESSAGES/collabui.mo;
   done

   echo_log "Session Collaboration install successfully"
}

extract_vchan_plugins() {
   # In Q4 release, we only support clipoard on 64-bit OS.
   # mksVChanserver and vvcProxyStub will not be put into
   # the 32-bit installer package.
   if [ ! -d "mksVChanServer" ] || [ ! -d "vvcProxy" ]; then
      return
   fi

   # Copy mksVChanServer files
   mkdir -p $VCHAN_PLUGINS_PATH
   if [ "${DISTRO_ID}" = "${DISTRO_ID_RHEL_WORKSTATION}" -a \
      "${OS_MAJOR}" -ge "7" ] || [ "${DISTRO_ID} = ${DISTRO_ID_UBUNTU}" -a \
      "${OS_MAJOR}" = "18" ]; then
      cp -rpf rde-sdk $VCHAN_PLUGINS_PATH/libs
      rm -f $VCHAN_PLUGINS_PATH/libs/libvdpservice.so $VCHAN_PLUGINS_PATH/libs/librdpvcbridge.so
      ln -fs $VCHAN_PLUGINS_PATH/libs/libvdpservice-gtk3.so $VCHAN_PLUGINS_PATH/libvdpservice.so
   else
      cp -pf rde-sdk/libvdpservice.so $VCHAN_PLUGINS_PATH
   fi
   cp -pf vvcProxy/* $VCHAN_PLUGINS_PATH
   chown -R root:root $VCHAN_PLUGINS_PATH
   chmod -R 755 $VCHAN_PLUGINS_PATH

   # Copy vvcProxyStub files
   mkdir -p $BLAST_VCHAN_PLUGINS_PATH
   chown -R root:root $BLAST_VCHAN_PLUGINS_PATH
   chmod -R 755 $BLAST_VCHAN_PLUGINS_PATH
}

extract_mks_vchan_server() {
   if [ ! -d ${VCHAN_PLUGINS_PATH} ]; then
      mkdir -p ${VCHAN_PLUGINS_PATH}
   fi

   cp -pf mksVChanServer/* $VCHAN_PLUGINS_PATH
   chown -R root:root $VCHAN_PLUGINS_PATH
   chmod -R 755 $VCHAN_PLUGINS_PATH
}

verify_rdeSvc_environment() {
   if [ "${DISTRO_ID}" = ${DISTRO_ID_RHEL_WORKSTATION} ]; then
      ldconfig -p | grep libcpptest.so >/dev/null 2>&1
      if [ "$?" != "0" ]; then
         echo_term "Required package \"libcpptest.so\" is not installed. Please install it first, then try again."
         dependency_package_guide_message
         exit
      fi
   fi

   ldconfig -p | grep liburiparser.so >/dev/null 2>&1
   if [ "$?" != "0" ]; then
      echo_term "Required package \"liburiparser.so\" is not installed. Please install it first, then try again."
      dependency_package_guide_message
      exit
   fi
}

extract_rdeserver() {
   if [ ! -d ${VCHAN_PLUGINS_PATH} ]; then
      mkdir -p ${VCHAN_PLUGINS_PATH}
   fi

   cp -pf rdeServer/rdeSvc $VCHAN_PLUGINS_PATH
   chown -R root:root $VCHAN_PLUGINS_PATH
   chmod -R 755 $VCHAN_PLUGINS_PATH

   cp -pf appScanner/appScanner $VCHAN_PLUGINS_PATH
   chown -R root:root $VCHAN_PLUGINS_PATH
   chmod -R 755 $VCHAN_PLUGINS_PATH

   if [ ! -d ${APP_ICON_DIR} ]; then
      mkdir -p ${APP_ICON_DIR}
   fi

   if [ ! -e ${APP_XML_FILE} ]; then
      touch ${APP_XML_FILE}
   fi

   #
   # vmtools dropped xdg-find-menus and vmware-xdg-detect-de in version 10,
   # need to restore them to support seamless window.
   #
   if [ ! -f /usr/lib/vmware-tools/bin/xdg-find-menus ]; then
      mkdir -p /usr/lib/vmware-tools/bin/
      ln -s ${VMTOOL_XDG_FOND_MENUS} /usr/lib/vmware-tools/bin/xdg-find-menus
   fi

   if [ ! -f /usr/bin/vmware-xdg-detect-de ]; then
      ln -s ${VMWARE_XDG_DETECT_DE} /usr/bin/vmware-xdg-detect-de
   fi
   if [ ! -d "${GNOME_SHELL_EXTENSIONS_DIR}/vmware-remote-app-helper@vmware.com" ]; then
      mkdir -p "${GNOME_SHELL_EXTENSIONS_DIR}/vmware-remote-app-helper@vmware.com"
   fi

   #
   # TODO: Workaround on for demo on POC branch. Will be removed once appScanner
   #       could be launched by DesktopDaemon in logoff state. Currently only
   #       firefox, termial and nautilus is published.
   #
   cp -rf icons/* ${APP_ICON_DIR}

   cp -rf themes/VMWLight ${APP_THEME_DIR}

   cp -rf extensions/vmware-remote-app-helper/* ${GNOME_SHELL_EXTENSIONS_DIR}/vmware-remote-app-helper@vmware.com
   cp -f extensions/vmware-remote-app-helper/shadowWindowExtension.js /usr/bin/
   chmod 755 /usr/bin/shadowWindowExtension.js

   cp -f extensions/gnome-shell-extension-tool-vmw /usr/bin/
   chmod 755 /usr/bin/gnome-shell-extension-tool-vmw

   if [ "${DISTRO_ID}" = ${DISTRO_ID_RHEL_WORKSTATION} ] &&
      [ "$OS_MAJOR" = "8" ]; then
      cp -f bin/ControlExtension3.py $VIEWAGENT_DESKTOP_PATH/ControlExtension.py
   else
      cp -f bin/ControlExtension.py $VIEWAGENT_DESKTOP_PATH
   fi
   chmod 755 $VIEWAGENT_DESKTOP_PATH/ControlExtension.py

   cp -f bin/ControlTheme.sh $VIEWAGENT_DESKTOP_PATH
   chmod 755 $VIEWAGENT_DESKTOP_PATH/ControlTheme.sh
}

write_env_file() {
   local cdrInstalled="false"
   local truessoInstalled="false"
   local usbInstalled="false"
   local collabInstalled="false"

   if [ ! -f "$AGENT_CFG_FILE" ]; then
      > $AGENT_CFG_FILE
      chmod 600 $AGENT_CFG_FILE
   fi

   # Check if CDR feature is enabled
   if [ "$CDRSERVER_ENABLE" = "yes" ]; then
      cdrInstalled="true"
   fi

   # Check if USB feature is enabled
   if [ "$USB_ENABLE" = "yes" ]; then
      usbInstalled="true"
   fi

   # Check if TrueSSO feature is enabled
   if [ "$TRUESSO_ENABLE" = "yes" ]; then
      truessoInstalled="true"
   fi

   # Check if Collaboration feature is enabled
   if [ "$COLLAB_ENABLE" = "yes" ]; then
      collabInstalled="true"
   fi

   while read cfg
   do
      var=`echo "$cfg" | cut -f1 -d" "`
      val=`echo "$cfg" | cut -f2 -d" "`
      grep "^$var" $AGENT_CFG_FILE >/dev/null 2>&1
      if [ "$?" != "0" ]; then
         echo "$var=$val" >> $AGENT_CFG_FILE
      elif [ $var = "MANAGED"  -o  $var = "LINKCLONE"  -o \
             $var = "CDRINST"  -o  $var = "USBINST"  -o \
             $var = "TRUESSOINST"  -o  $var = "FIPS_ENABLE" -o \
             $var = "OS_VERSION_MAJOR" -o $var = "OS_VERSION_MINOR"  -o \
             $var = "IPC_ENABLE"  -o $var = "SINGLETON_MODE" -o \
             $var = "COLLABINST" -o \
             $var = "CONTROL_EXTENSION_SCRIPT" -o \
             $var = "CONTROL_THEME_SCRIPT" ]; then
          sed -i '/^'$var'/d' $AGENT_CFG_FILE >/dev/null 2>&1
          echo "$var=$val" >> $AGENT_CFG_FILE
      fi
   done << !!!
NAME $NAME
HOSTNAME $HOSTNAME
MANAGED $MANAGED
SINGLETON_MODE $SINGLETON_MODE
LINKCLONE $LINKCLONE
BROKER $BROKER
KDC $KDC
DOMAIN $DOMAIN
BROKERKDC $BROKERKDC
BROKERDOMAIN $BROKERDOMAIN
JMS_KEYSTORE_DIR $JMS_KEYSTORE_DIR
JMS_KEYSTORE_PASSWORD $JMS_KEYSTORE_PASSWORD
OS_VERSION_LDAP $OS_VERSION
OS_VERSION_MAJOR $OS_MAJOR
OS_VERSION_MINOR $OS_MINOR
PLATFORM $DISTRO_PLATFORM_LABEL
PROTOCOL BLAST
PROTOCOL_PORT $BLAST_PORT
CERTIFICATE_PATH $CERT
BLAST_SERVER_STATUS_FILE $BLAST_STAT_FILE
MACHINE_CONFIG_FILE $MACHINE_CFG_FILE
CUSTOM_CONFIG_FILE $CUSTOM_CFG_FILE
CUSTOM_CONFIG_FILE_BLAST $CUSTOM_CFG_FILE_BLAST
BLAST_SERVER_PROGRAM $BLAST_SERVER_PATH
DESKTOP_DAEMON_PROGRAM $DESKTOP_DAEMON_PATH
CDR_SERVER_PATH $CDR_SERVER
COLLABORATION_UI_PATH $COLLABORATION_UI_PATH
START_CLIPBOARD_SCRIPT $START_CLIPBOARD_SH
START_CDR_SCRIPT $START_CDR_SH
START_RDE_SERVER_SCRIPT $START_RDE_SERVER_SH
STOP_RDE_SERVER_SCRIPT $STOP_RDE_SERVER_SH
LAUNCH_APP_SCRIPT $LAUNCH_APP_SH
CONTROL_EXTENSION_SCRIPT $CONTROL_EXTENSION_PY
CONTROL_THEME_SCRIPT $CONTROL_THEME_SH
START_USBREDIR_SCRIPT $START_USBREDIR_SH
STOP_USBREDIR_SCRIPT $STOP_USBREDIR_SH
SSO_NOTIFY_SCRIPT $SSO_DBUS_PY
SSO_CONFIG_DESKTOP_TYPE_SCRIPT $SSO_CONFIG_DESKTOP_TYPE_SH
CLEANLOG_SCRIPT $CLEANLOG_SH
MOUNT_FOLDER_SCRIPT $MOUNT_FOLDER_SH
UMOUNT_FOLDER_SCRIPT $UMOUNT_FOLDER_SH
START_XSERVER_SCRIPT $START_XSERVER_SH
START_BLASTWORKER_SCRIPT $START_BLASTWORKER_SH
RUN_APPSCANNER_SCRIPT $RUN_APPSCANNER_SH
POSTLOGIN_SCRIPT $POST_LOGIN_SH
APP_ICON_DIR $APP_ICON_DIR
APP_XML_FILE $APP_XML_FILE
RUNONCE $RUNONCE
CDRINST $cdrInstalled
USBINST $usbInstalled
COLLABINST $collabInstalled
TRUESSOINST $truessoInstalled
FIPS_ENABLE $FIPS_ENABLE
IPC_ENABLE $IPC_ENABLE
CUSTOMIZATION_STATE $CUSTOMIZATION_STATE
!!!
}

load_custom_configuration() {
   #
   # Copy custom config template
   # If template exists, overwrite.
   #
   cp template/config.template ${CUSTOM_CFG_FILE_BLAST}.template
   chown root:root ${CUSTOM_CFG_FILE_BLAST}.template
   chmod 644 ${CUSTOM_CFG_FILE_BLAST}.template

   cp template/viewagent-custom.conf.template ${CUSTOM_CFG_FILE}.template
   chown root:root ${CUSTOM_CFG_FILE}.template
   chmod 644 ${CUSTOM_CFG_FILE}.template

   #
   # Write template into config file
   # If the config file not exists, copy the template to it.
   #
   if [ ! -f "$CUSTOM_CFG_FILE_BLAST" ]; then
      cp template/config.template $CUSTOM_CFG_FILE_BLAST
      chown root:root $CUSTOM_CFG_FILE_BLAST
      chmod 644 $CUSTOM_CFG_FILE_BLAST
   fi

   if [ ! -f "$CUSTOM_CFG_FILE" ]; then
      cp template/viewagent-custom.conf.template $CUSTOM_CFG_FILE
      chown root:root $CUSTOM_CFG_FILE
      chmod 644 $CUSTOM_CFG_FILE
   fi
}


#
# Fix bug 1453710
# When we upgrade our viewagent, we may use the old configure file.
# So we need to upgrade the blast server port in /etc/vmware/viewagent-config.txt
#
upgrade_blast_server_port() {
   sed -i "s/PROTOCOL_PORT=[0-9]*/PROTOCOL_PORT=${BLAST_PORT}/g" $AGENT_CFG_FILE
}

write_bora_config() {
   if [ ! -f $BORA_CFG_FILE ]; then
      > $BORA_CFG_FILE
      chmod 644 $BORA_CFG_FILE
   else
      cp $BORA_CFG_FILE ${BORA_CFG_FILE}.orig
   fi

   while read cfg
   do
      var=`echo "$cfg" | cut -f1 -d" "`
      val=`echo "$cfg" | cut -f2 -d" "`
      grep "^$var" $BORA_CFG_FILE >/dev/null 2>&1
      if [ "$?" = "1" ]; then
         echo "$var = $val" >> $BORA_CFG_FILE
      fi
   done << !!!
!!!
}


register_viewagent() {
   if [ -f "$MACHINE_CFG_FILE" ]; then
      echo_log "Existing machine configuration file present. Not registering agent."
   else
      (cd "$VIEWAGENT_PATH" && \
         $JRE_JAVA -showversion -Xmx512m -XX:+UseConcMarkSweepGC -cp "*:" \
         com.vmware.vdi.agent.standalone.StandaloneAgent -r -k "$KDC" -d "$DOMAIN" \
         -b "$BROKER" -u "$LDAP_USERNAME" -p "$LDAP_PASSWORD" -n "$NAME" \
         -v "$OS_VERSION" -m "$MACHINE_CFG_FILE" -K "$BROKERKDC" \
         -B "$BROKERDOMAIN" > "$VIEWAGENT_REG_LOG" 2>&1)
      if [ "$?" = "1" ]; then
         echo_term "Agent registration failed and installation was terminated."
         echo_term ""
         echo_term "Please double check below things and retry:"
         echo_term -n "1) Make sure the broker address, domain name, user name, "
         echo_term -n "and password are correct."
         echo_term ""
         echo_term -n "2) Make sure the user provided has \"Agent Registration\" "
         echo_term -n "permission in Horizon Administrator."
         echo_term ""
         echo_term -n "3) If password contains shell special character, please "
         echo_term -n "ensure escape it."
         echo_term ""
         echo_term -n "4) If you are using DIGEST-MD5 authentication mechanism "
         echo_term -n "(you did not specify the -k option), please ensure the "
         echo_term -n "AD policy \"store password using reversible encryption\" "
         echo_term -n "is enabled for the user you specified."
         echo_term ""
         echo_term -n "5) If you are using keberos authentication mechanism "
         echo_term -n "(you specified the -k option), please ensure the AD address "
         echo_term -n "is correct for the -k option; Please ensure broker's "
         echo_term -n "\"Full computer name\" is specified for -b option, which "
         echo_term -n "includes the broker's hostname and domain name."
         echo_term ""
         registerRollback
         echo_term ""
         echo_term "All changes has been rollbacked."
         exit 1
      fi
      chmod 600 "$MACHINE_CFG_FILE"
   fi
}

setup_shutdown_script_service() {
   case "$SERVICE_MGR" in
   ${SERVICE_MGR_SYSV})
      if [ "$DISTRO_ID" = "$DISTRO_ID_SUSE" ]; then
         cp $SHUTDOWN_SYSV_SUSE $SYSV_CONF_PATH_SUSE/$SHUTDOWN_SYSV
         chmod 755 $SYSV_CONF_PATH_SUSE/$SHUTDOWN_SYSV
         # Create a symbolic link /sbin/rcviewagent_shutdown-->/etc/init.d/viewagent_shudown
         if [ ! -L /sbin/rcviewagent_shutdown ]; then
             ln -s $SYSV_CONF_PATH_SUSE/$SHUTDOWN_SYSV /sbin/rcviewagent_shutdown
         fi
      else
         cp $SHUTDOWN_SYSV $SYSV_CONF_PATH
         chmod 755 $SYSV_CONF_PATH/$SHUTDOWN_SYSV
      fi
      chkconfig --add viewagent_shutdown > /dev/null 2>&1
      ;;
   ${SERVICE_MGR_SYSTEMD})
      if [ "$DISTRO_ID" = "$DISTRO_ID_UBUNTU" ]; then
         cp $SHUTDOWN_SYSTEMD_UBUNTU $SYSTEMD_CONF_PATH_UBUNTU/$SHUTDOWN_SYSTEMD
         chmod 644 $SYSTEMD_CONF_PATH_UBUNTU/$SHUTDOWN_SYSTEMD
         if [ ! -f "$SHUTDOWN_ENV_FILE_UBUNTU" ]; then
            > $SHUTDOWN_ENV_FILE_UBUNTU
            chmod 600 $SHUTDOWN_ENV_FILE_UBUNTU
         fi
      else
         cp $SHUTDOWN_SYSTEMD $SYSTEMD_CONF_PATH
         chmod 644 $SYSTEMD_CONF_PATH/$SHUTDOWN_SYSTEMD
         #
         # systemd fails to start service unless the sysconfig file specified in
         # the unit file is present.
         #
         if [ ! -f "$SHUTDOWN_ENV_FILE" ]; then
            > $SHUTDOWN_ENV_FILE
            chmod 600 $SHUTDOWN_ENV_FILE
         fi
      fi
      systemctl enable viewagent_shutdown > /dev/null 2>&1
      ;;
   ${SERVICE_MGR_UPSTART})
      cp $SHUTDOWN_UPSTART $UPSTART_CONF_PATH
      chmod 644 $UPSTART_CONF_PATH/$SHUTDOWN_UPSTART
      ;;
   esac
}

setup_viewagent_service() {
   case "$SERVICE_MGR" in
   ${SERVICE_MGR_SYSV})
      if [ "$DISTRO_ID" = "$DISTRO_ID_SUSE" ]; then
         cp $VIEWAGENT_SYSV_SUSE $SYSV_CONF_PATH_SUSE/$VIEWAGENT_SYSV
         chmod 755 $SYSV_CONF_PATH_SUSE/$VIEWAGENT_SYSV
         # Create a symbolic link /sbin/rcviewagent-->/etc/init.d/viewagent
         if [ ! -L /sbin/rcviewagent ]; then
             ln -s $SYSV_CONF_PATH_SUSE/$VIEWAGENT_SYSV /sbin/rcviewagent
         fi
      else
         cp $VIEWAGENT_SYSV $SYSV_CONF_PATH
         chmod 755 $SYSV_CONF_PATH/$VIEWAGENT_SYSV
      fi
      chkconfig --add viewagent > /dev/null 2>&1
      ;;
   ${SERVICE_MGR_SYSTEMD})
      if [ "$DISTRO_ID" = "$DISTRO_ID_UBUNTU" ]; then
         cp $VIEWAGENT_SYSTEMD_UBUNTU $SYSTEMD_CONF_PATH_UBUNTU/$VIEWAGENT_SYSTEMD
         chmod 644 $SYSTEMD_CONF_PATH_UBUNTU/$VIEWAGENT_SYSTEMD
         if [ ! -f "$VIEWAGENT_ENV_FILE_UBUNTU" ]; then
            > $VIEWAGENT_ENV_FILE_UBUNTU
            chmod 600 $VIEWAGENT_ENV_FILE_UBUNTU
         fi
      else
         cp $VIEWAGENT_SYSTEMD $SYSTEMD_CONF_PATH
         chmod 644 $SYSTEMD_CONF_PATH/$VIEWAGENT_SYSTEMD
         #
         # systemd fails to start service unless the sysconfig file specified in
         # the unit file is present.
         #
         if [ ! -f "$VIEWAGENT_ENV_FILE" ]; then
            > $VIEWAGENT_ENV_FILE
            chmod 600 $VIEWAGENT_ENV_FILE
         fi
      fi
      systemctl enable viewagent > /dev/null 2>&1
      ;;
   ${SERVICE_MGR_UPSTART})
      cp $VIEWAGENT_UPSTART $UPSTART_CONF_PATH
      chmod 644 $UPSTART_CONF_PATH/$VIEWAGENT_UPSTART
      ;;
   esac
}


configure_firewall_iptables() {
   local ret=0
   local protocol="tcp"
   local port=${BLAST_PORT}

   if [ "$#" != "2" ]; then
      echo_term "configure_firewall_iptables: No command specified."
      return 1
   fi

   protocol="$1"
   prot="$2"

   #
   # If the iptables policy and configuration looks vanilla then configure,
   # otherwise inform.
   #
   last_input_rule="`iptables -nL INPUT --line-numbers | tail -1 | grep 'REJECT.*all'`"
   if [ "$?" = "0" ]; then
      line_number="`echo $last_input_rule | awk '{ print $1 }'`"

      #
      # Only save one time for iptables rules.
      #
      if [ ! -f $VIEWAGENT_PATH/iptables.pre ]; then
         iptables-save > $VIEWAGENT_PATH/iptables.pre
      fi
      ret=$?

      #
      # Allow client to initiate connection to agent
      #
      [ "$ret" != "0" ] || iptables -I INPUT $line_number -p ${protocol} \
         -m state --state NEW -m ${protocol} --dport ${port} -j ACCEPT || ret=1

      #
      # Make firewall change persistent across reboot (writeback to config file)
      #
      serviceSave iptables
      echo_log "Firewall for $protocol: configured."
   fi

   return $ret
}


configure_firewall_firewalld() {
   local ret=0
   local protocol="tcp"
   local port=${BLAST_PORT}

   if [ "$#" != "2" ]; then
      echo_term "configure_firewall_firewalld: No command specified."
      return 1
   fi

   protocol="$1"
   port="$2"

   firewall-cmd --add-port=${port}/${protocol} >/dev/null 2>&1 || ret=1
   [ "$ret" = "1" ] || firewall-cmd --add-port=${port}/${protocol} \
      --permanent >/dev/null 2>&1 || ret=1

   return $ret
}

configure_firewall_suse() {
   local ret=1
   local protocol="tcp"
   local port=${BLAST_PORT}
   local keyword

   if [ "$#" != "2" ]; then
      echo_term "configure_firewall_suse: No command specified."
      return 1
   fi

   protocol="$1"
   port="$2"

   if [ "$protocol" = "tcp" ]; then
      keyword="FW_SERVICES_EXT_TCP"
   else
      keyword="FW_SERVICES_EXT_UDP"
   fi

   SuSEfirewall="/etc/sysconfig/SuSEfirewall2"
   if [ -f $SuSEfirewall ]; then
      #
      # Backup existing firewall
      #
      if [ ! -f $VIEWAGENT_PATH/SuSEfirewall2.orig ]; then
         cp -f $SuSEfirewall $VIEWAGENT_PATH/SuSEfirewall2.orig
      fi

      num=`grep ${keyword}= $SuSEfirewall|wc -l`
      if [ $num -ge 1 ]; then
         str=`grep ${keyword}= $SuSEfirewall |grep -v "#"`
         pos=`expr index "$str" '='`
         length=`expr length "$str"`
         start=`expr $pos + 2`
         size=`expr $length - $pos - 2`
         content=`expr substr "$str" $start $size`
         rep="${keyword}=\"$content $port\""
         sed "/^${keyword}=*/c ${rep}" $SuSEfirewall > $SuSEfirewall.tmp
         mv -f $SuSEfirewall.tmp $SuSEfirewall
         # Restart the service if the original status of firewall is ON
         rcSuSEfirewall2 status | grep "unused" || rcSuSEfirewall2 status | grep "inactive" >/dev/null 2>&1
         if [ "$?" != "0" ]; then
            rcSuSEfirewall2 restart >/dev/null 2>&1
         fi
         ret=0
         echo_log "Firewall for $protocol: configured"
      fi
   fi
   return $ret
}


configure_firewall() {
   local ret

   if [ "$DISTRO_ID" = "$DISTRO_ID_SUSE" ]; then
      configure_firewall_suse tcp ${BLAST_PORT}
      configure_firewall_suse udp ${BLAST_PORT}
      ret=$?
   else
      case "${FIREWALL}" in
      ${FIREWALL_IPTABLES})
         configure_firewall_iptables tcp ${BLAST_PORT}
         configure_firewall_iptables udp ${BLAST_PORT}
         ret=$?
         ;;
      ${FIREWALL_FIREWALLD})
         configure_firewall_firewalld tcp ${BLAST_PORT}
         configure_firewall_firewalld udp ${BLAST_PORT}
         ret=$?
         ;;
      esac
   fi

   if [ "$ret" = "1" ]; then
      echo_term "Unable to automatically configure firewall."
      echo_term "Please configure firewall for inbound connections on port ${BLAST_PORT}/tcp and/or ${BLAST_PORT}/udp."
   fi
}


configure_selinux() {
   case "$DISTRO_ID" in
   ${DISTRO_ID_CENTOS}|\
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_RHEL_WORKSTATION})

      local tmpdir=$VIEWAGENT_PATH/selinux.tmp

      echo_log "Begin to configure selinux"

      which checkmodule >/dev/null 2>&1
      if [ "$?" != "0" ]; then
         echo_term "The required command \"checkmodule\" is currently not installed, configure selinux failed."
         return
      fi

      which semodule_package >/dev/null 2>&1
      if [ "$?" != "0" ]; then
         echo_term "The required command \"semodule_package\" is currently not installed, configure selinux failed."
         return
      fi

      which semodule >/dev/null 2>&1
      if [ "$?" != "0" ]; then
         echo_term "The required command \"semodule\" is currently not installed, configure selinux failed."
         return
      fi

      mkdir -p $tmpdir
      chmod -R 700 $tmpdir

      if [ "$SSO_ENABLE" = "yes" ]; then
         echo "module vmwsso 1.0;" > $tmpdir/vmwsso.te
         echo "require {" >> $tmpdir/vmwsso.te
         echo "type init_t;" >> $tmpdir/vmwsso.te
         echo "type initrc_t;" >> $tmpdir/vmwsso.te
         echo "type oddjob_t;" >> $tmpdir/vmwsso.te
         echo "type var_run_t;" >> $tmpdir/vmwsso.te
         if [ "$OS_MAJOR" = "8" -a "$TRUESSO_ENABLE" = "yes" ]; then
            echo "type xdm_var_run_t;" >> $tmpdir/vmwsso.te
            echo "type config_home_t;" >> $tmpdir/vmwsso.te
            echo "type sssd_t;" >> $tmpdir/vmwsso.te
         fi
         echo "type vmware_host_pid_t;" >> $tmpdir/vmwsso.te
         echo "type xdm_t;" >> $tmpdir/vmwsso.te
         echo "class dbus send_msg;" >> $tmpdir/vmwsso.te
         echo "class dir { search read getattr open };" >> $tmpdir/vmwsso.te
         if [ "$OS_MAJOR" = "8" -a "$TRUESSO_ENABLE" = "yes" ]; then
            echo "class sock_file write;" >> $tmpdir/vmwsso.te
         fi
         echo "class file { read open getattr lock };" >> $tmpdir/vmwsso.te
         echo "class unix_stream_socket connectto;" >> $tmpdir/vmwsso.te
         echo "}" >> $tmpdir/vmwsso.te
         echo "allow oddjob_t init_t:dbus send_msg;" >> $tmpdir/vmwsso.te
         echo "allow xdm_t initrc_t:unix_stream_socket connectto;" >> $tmpdir/vmwsso.te
         echo "allow xdm_t var_run_t:file { read open getattr lock };" >> $tmpdir/vmwsso.te
         echo "allow xdm_t vmware_host_pid_t:dir { search read getattr open };" >> $tmpdir/vmwsso.te
         if [ "$OS_MAJOR" = "8" -a "$TRUESSO_ENABLE" = "yes" ]; then
            echo "allow sssd_t config_home_t:dir search;" >> $tmpdir/vmwsso.te
            echo "allow sssd_t xdm_var_run_t:sock_file write;" >> $tmpdir/vmwsso.te
         fi

         checkmodule -M -m -o $tmpdir/vmwsso.mod $tmpdir/vmwsso.te > /dev/null 2>&1
         semodule_package -o $tmpdir/vmwsso.pp -m $tmpdir/vmwsso.mod > /dev/null 2>&1
         semodule -i $tmpdir/vmwsso.pp > /dev/null 2>&1
      fi

      # rhel8 & centos8
      if [ "$OS_MAJOR" = "8" ]; then
         # https://access.redhat.com/documentation/en-us/red_hat_enterprise_linux/8/html-single/using_selinux/index
         # https://access.redhat.com/documentation/en-us/red_hat_enterprise_linux/7/html/selinux_users_and_administrators_guide/index
         # https://access.redhat.com/solutions/3227751
         # https://access.redhat.com/documentation/en-us/red_hat_enterprise_linux/6/html/security-enhanced_linux/sect-security-enhanced_linux-selinux_contexts_labeling_files-persistent_changes_semanage_fcontext
         semanage fcontext -a -t bin_t "$VIEWAGENT_PATH(/.*)?" > /dev/null 2>&1
         restorecon -R -v $VIEWAGENT_PATH > /dev/null 2>&1
      fi

      rm -rf $tmpdir
      ;;
   ${DISTRO_ID_UBUNTU})
      ;;
   *)
      ;;
   esac
}

configure_ipc() {
   if [ ! -d ${SYSCTL_BACKUP_DIR} ]; then
       mkdir -p ${SYSCTL_BACKUP_DIR}
       chown -R root:root ${SYSCTL_BACKUP_DIR}
   fi

   cp /etc/sysctl.conf ${SYSCTL_BACKUP_DIR}/sysctl.conf.backup

   echo_log "Begin to configure network for IPC"

   #
   # Configure the network params to setup an efficient network environment
   # for nio ipc channels
   #
   sed -i 's^net.core.somaxconn^#net.core.somaxconn^g' /etc/sysctl.conf
   sed -i '$a net.core.somaxconn=8192' /etc/sysctl.conf

   sed -i 's^net.ipv4.tcp_max_syn_backlog^#net.ipv4.tcp_max_syn_backlog^g' /etc/sysctl.conf
   sed -i '$a net.ipv4.tcp_max_syn_backlog=8192' /etc/sysctl.conf

   sysctl --system > /dev/null 2>&1
}


read_password_secure()
{
  if [ "$isRegistered" = "false" ]; then

      # no need to input password if no user name is identified
      if [ -z "$LDAP_USERNAME" ]; then
        return 0
      fi

      # password need to be input only if no password input before
      if [  "$LDAP_PASSWORD" != "" ]; then
          return 0
      fi

      # input password in same line
      echo_term -n "Please enter the password for LDAP Username ${LDAP_USERNAME} :"

      # Disable echo
      stty -echo

      # set up trap for SIGINT
      trap "stty echo" 2

      # Read passowrd
      read LDAP_PASSWORD

      # enable echo
      stty echo

      # start new line
      echo_term
  fi
}

#
# Show EULA (End-user license agreement) to user
# If user accept the terms installaltion will continue
# or the installation will be terminated
#
verify_EULA_agreement()
{
    EULA_FILE=`dirname $0`/EULA.txt

    # Show EULA on terminal
    cat $EULA_FILE
    # Tell user to type [Yes/No]
    echo_term -n "Are you sure to install linux agent (y/n)?"
    read USER_INPUT
    echo_term
    # check user input
    case "$USER_INPUT" in
        Y|y)
            # User accept the license agreement, installation continue
            echo_term "Installation starting..."
            ;;
        N|n)
            # User reject the license agreement, installtion terminate
            echo_term "Installation stopped..."
            exit
            ;;
          *)
            # Invalid input
            echo_term "Invalid input and Installation stopped..."
            exit
            ;;
    esac
}

show_FIPS_statement() {
    echo_term -n "You have selected to install Horizon Agent for Linux, "
    echo_term -n "which implements the cryptographic modules designed for "
    echo_term -n "FIPS 140-2 compliance. These modules were validated in "
    echo_term -n "operational environments listed in CMVP certificate #2839 "
    echo_term -n "and #2866, and were ported to this platform. However, "
    echo_term -n "the CAVP and CMVP testing requirement to include the new "
    echo_term -n "operational environments in VMware's NIST CAVP and CMVP "
    echo_term -n "certificates remains to be completed on the product roadmap."
    echo_term ""
}

#
# Show FIPS statement to user
# If user accept the terms installaltion will continue
# or the installation will be terminated
#
verify_FIPS_statement()
{
    show_FIPS_statement

    # Tell user to type [Yes/No]
    echo_term -n " Do you want to continue? (y/n)"
    read USER_INPUT
    echo_term
    # check user input
    case "$USER_INPUT" in
        Y|y)
            # User accept the license agreement, installation continue
            ;;
        N|n)
            # User reject the license agreement, installtion terminate
            echo_term "Installation stopped..."
            exit
            ;;
          *)
            # Invalid input
            echo_term "Invalid input and Installation stopped..."
            exit
            ;;
    esac
}

#
# Compare the package version or release with the minimum version and release defined in the dedicated variable.
# The dedicated variable is defined in the fine commonlib.sh
# The format to call this function is "package_minimum_version_compare package_name minVersion minRelease"
#
package_minimum_version_compare()
{
   local PACKAGE_NAME=$1
   local minVERSION=$2.
   local minRELEASE=$3.
   local installVERSION=$(rpm -q --queryformat "%{VERSION}\n" $PACKAGE_NAME).
   local installRELEASE=$(rpm -q --queryformat "%{RELEASE}\n" $PACKAGE_NAME).
   local versionSatisfy="false"
   if [ $1 = "xorg-x11-server" ]; then
      versionRequired=$2.$3
   else
      versionRequired=$2-$3
   fi
   while [ "$minVERSION" != "" ]
   do
      VERSION_DEFINED=`echo "$minVERSION" | cut -f1 -d"."`
      VERSION_INSTALLED=`echo "$installVERSION" | cut -f1 -d"."`
      if [ "$VERSION_INSTALLED" -gt "$VERSION_DEFINED" ]; then
         versionSatisfy="true"
         break
      fi
      if [ "$VERSION_INSTALLED" -lt "$VERSION_DEFINED" ]; then
         echo_term "The version of $PACKAGE_NAME is lower than required. Please upgrade it to version no lower than \"$versionRequired\" ."
         dependency_package_guide_message
         exit
      fi

      minVERSION=${minVERSION#*.}
      installVERSION=${installVERSION#*.}
   done


   if [ "$versionSatisfy" = "false" ] && [ "$minVERSION" = "" ]; then
      while [ "$minRELEASE" != "" ]
      do
         RELEASE_DEFINED=`echo "$minRELEASE" | cut -f1 -d"."`
         RELEASE_INSTALLED=`echo "$installRELEASE" | cut -f1 -d"."`
         if [ "$RELEASE_INSTALLED" -gt "$RELEASE_DEFINED" ]; then
            break
         fi
         if [ "$RELEASE_INSTALLED" -lt "$RELEASE_DEFINED" ]; then
            echo_term "The version of $PACKAGE_NAME is lower than required. Please upgrade it to version no lower than \"$versionRequired\" ."
            dependency_package_guide_message
            exit
         fi
         minRELEASE=${minRELEASE#*.}
         installRELEASE=${installRELEASE#*.}
      done
   fi
}

verify_SLED_environment() {
   #
   # For SLED 11, we need to check the version of xorg-x11-server, if the version is lower than 7.4.27.111.1
   # the X server may hang forever thus cause VMwareBlastServer crash
   #
   # Check the xorg-x11-server version is correct or not
   # If the version is lower than the min version 7.4, the installer should print error message and exit;
   # If the version is equal to the min version 7.4, the release number must not be lower than 27.111.1.
   #
   if [ "$DISTRO_RELEASE" = "11" ]; then
      package_minimum_version_compare xorg-x11-server $MINVERSION_XORG_X11_SERVER $MINRELEASE_XORG_X11_SERVER
   fi

   #
   # For SLED, we need to check if "Change Hostname via DHCP" is disalbed
   # If not, installer should prompt user to manually disable it
   #
   if [ "$DISTRO_ID" = "$DISTRO_ID_SUSE" ]; then
      DHCLIENT_SET_HOSTNAME=$(grep -E "DHCLIENT_SET_HOSTNAME=\"yes\"" /etc/sysconfig/network/dhcp)
      if [ -n "$DHCLIENT_SET_HOSTNAME" ]; then
         echo_term "Please manually disable \"Change Hostname via DHCP\" and suggest you setting hostname you want, then try again."
         exit
      fi
   fi

   #
   # For SLED12, Check the xf86-video-vmware version is correct or not
   # If the version is lower than the min version 13.1.0, the installer should print error message and exit;
   # If the version is equal to the min version 13.1.0, the release number must not be lower than 5.2.
   #
   if [ "$OS_MAJOR" = "12" ]; then
      rpm -q python-gobject2 >/dev/null 2>&1
      if [ "$?" != "0" ]; then
         echo_term "Required package \"python-gobject2\" is not installed. Please install it first, then try again."
         dependency_package_guide_message
         exit
      fi
      package_minimum_version_compare xf86-video-vmware $MINVERSION_XF86_VIDEO_VMWARE $MINRELEASE_XF86_VIDEO_VMWARE
   fi
}

check_glibc_version() {
   local glibcVersion=`rpm -q --queryformat "%{VERSION}\n" glibc`
   local glibcRelease=`rpm -q --queryformat "%{RELEASE}\n" glibc`
   if [ "$glibcVersion" = "2.12" ] && [ "$glibcRelease" = "1.166.el6" ]; then
      echo_term "Warning: Your current system is using glibc-2.12-1.166.el6.x86_64, which has deadlock issue. Please upgrade to latest version."
      dependency_package_guide_message
   fi
}

check_appindicator3_package() {
   rpm -q "libappindicator-gtk3" >/dev/null 2>&1
   if [ "$?" != "0" ]; then
       echo_term "Required package \"libappindicator-gtk3\" is not installed. Please install it first, then try again."
       dependency_package_guide_message
       exit
   fi
}

check_xorg_x11_utils_package() {
   rpm -q "xorg-x11-utils" >/dev/null 2>&1
   if [ "$?" != "0" ]; then
       echo_term "Required package \"xorg-x11-utils\" is not installed. Please install it first, then try again."
       dependency_package_guide_message
       exit
   fi
}

check_xorg_x11_drv_vmware_package() {
   rpm -q "xorg-x11-drv-vmware" >/dev/null 2>&1
   if [ "$?" != "0" ]; then
       echo_term "Required package \"xorg-x11-drv-vmware\" is not installed. Please install it first, then try again."
       dependency_package_guide_message
       exit
   fi
}

verify_RHELOrCentOS_environment() {
   #
   # Check the version of glibc on RHEL/CentOS6.7 to avoid some deadlock issuses.
   #
   case "$DISTRO_ID" in
   ${DISTRO_ID_CENTOS}|\
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_RHEL_WORKSTATION})
      if [ ${OS_MAJOR} = "6" ] && [ ${OS_MINOR} = "7" ]; then
         check_glibc_version
      elif [ ${OS_MAJOR} = "7" ] && [ ${DISTRO_ID} = ${DISTRO_ID_RHEL_WORKSTATION} ] &&\
           [ ${OS_MINOR} = "5" -o ${OS_MINOR} = "6" -o ${OS_MINOR} = "7" -o \
             ${OS_MINOR} = "8" ]; then
         check_appindicator3_package
      elif [ ${OS_MAJOR} = "8" ]; then
         if [ ${DISTRO_ID} = ${DISTRO_ID_RHEL_WORKSTATION} ]; then
            check_appindicator3_package
         fi
         check_xorg_x11_drv_vmware_package
      fi
      ;;
   *)
      ;;
   esac
}

indicator_session_check()
{
   local system_version=""
   local indicator_session_version=""
   local sub_version=""
   system_version="`grep "RELEASE" /etc/lsb-release |cut -d= -f2`"
   if [ "$system_version" = "14.04" ] && [ "$ISKUBUNTU" != "true" ]; then
      indicator_session_version="`dpkg -s indicator-session | grep Version | cut -d" " -f2`"
      sub_version="`echo ${indicator_session_version}|cut -d- -f1|cut -d+ -f2`"
      a=`echo ${sub_version} |cut -d\. -f1`
      b=`echo ${sub_version} |cut -d\. -f2`
      c=`echo ${sub_version} |cut -d\. -f3`
      if [ ${a} -eq 14 -a ${b} -eq 10 -a ${c} -lt 20140718 ] ||
         [ ${a} -eq 15 -a ${b} -eq 04 -a ${c} -lt 20150327 ] ||
         [ ${a} -eq 14 -a ${b} -eq 04 ]; then
         echo_term "For better experience please download and install indicator-session package 12.10.5+15.04.20150327-0ubuntu1 or later."
         dependency_package_guide_message
         exit
      fi
   fi
}

verify_Ubuntu_environment() {
   #
   # Check whether system is kubuntu.
   #
   dpkg -s kubuntu-settings-desktop >/dev/null 2>&1
   if [ "$?" = "0" ]; then
      echo_term "The system is kubuntu."
      ISKUBUNTU="true"
   fi

   if [ "$ISKUBUNTU" != "true" ] && [ "$OS_MAJOR" = "18" ]; then
      # The default desktop environment ubuntu-desktop of Ubuntu1804 is supported, skip following checks
      return
   fi

   #
   # There is a bug in indicator-session in Ubuntu 14.04, which can not disable the user list.
   # To avoid this, check the version of indicator-session.
   #
   indicator_session_check

   #
   # Removing Guest Session at login stage
   #
   if [ -f /usr/share/lightdm/lightdm.conf.d/50-ubuntu.conf ]; then
      echo "allow-guest=false" >> /usr/share/lightdm/lightdm.conf.d/50-ubuntu.conf
   fi

   #
   # Check the package gnome-session-fallback in Ubuntu14.04 and check the gnome-session-flashback in Ubuntu16.04.
   # The package is required to disable compiz mode in Ubuntu.
   # Also, check what kind of desktop is installed to decide what message should be shown to user.
   #
   local gnome_session_package=""
   local hasGnomeSessionPackageInstalled="false"
   local hasKDEDesktopInstalled="false"
   local hasMateDesktopInstalled="false"
   if [ "${OS_MAJOR}" = "14" ] || [ "${OS_MAJOR}" = "16" ]; then
      if [ "${OS_MAJOR}" = "14" ]; then
         gnome_session_package="gnome-session-fallback"
         if [ -f "/usr/share/xsessions/kde-plasma.desktop" ]; then
            hasKDEDesktopInstalled="true"
         fi
      fi

      if [ "${OS_MAJOR}" = "16" ]; then
         gnome_session_package="gnome-session-flashback"

         if [ -f "/usr/share/xsessions/plasma.desktop" ]; then
            hasKDEDesktopInstalled="true"
         fi
      fi

      dpkg -s ${gnome_session_package} > /dev/null 2>&1
      if [ "$?" = "0" ]; then
         hasGnomeSessionPackageInstalled="true"
      fi

      if [ -f "/usr/share/xsessions/mate.desktop" ]; then
         hasMateDesktopInstalled="true"
      fi

      if [ "$hasGnomeSessionPackageInstalled" = "true" ] || [ "$hasKDEDesktopInstalled" = "true" ] || [ "$hasMateDesktopInstalled" = "true" ]; then
         echo_term -n "Warning: Your system alreday have a supported desktop environment. Please also select it when login. For more details,"
         echo_term " please review the Horizon Setup Guide."
      else
         echo_term -n "Warning: Your system does not have any supported desktop environment. Please install one and select it when login."
         echo_term " For more details, please review the Horizon Setup Guide."
      fi
   fi
}

#
# Check smartcard support environment
# 1. RedhatWorkstation 6.6 or later Or Ubuntu 18.04/16.4 Or SLED12SP3/SLES12SP3/SLED11SP4 supported
# 2. x86 64bit
# 3. SELinux is Disabled
# 4. pcsclite version is 1.7.4. for redhat 6.6 and later or 1.8.8 for redhat 7.X or Ubuntu 18.4/16.4 or SLED12SP3/SLES12SP3/SLED11SP4
#
verify_scredir_environment()
{
   local arch="`uname -i`"
   local secfg=""
   local pcscd_version=""

   local meet_all_requirements="yes"
   local meet_distro="yes"
   local meet_secfg="yes"
   local meet_pcscd="yes"

   local suse_major=""
   local suse_minor=""
   local suse_desc=""

   which pcscd > /dev/null 2>&1
   if [ $? = "0" ]; then
      pcscd_version="`pcscd -v | grep version | cut -d " " -f 3`"
   fi

   #
   # Smartcard redirection limitation
   #
   case "$DISTRO_ID" in
   ${DISTRO_ID_RHEL_WORKSTATION})

      which getenforce > /dev/null 2>&1
      if [ $? = "0" ]; then
         secfg="`getenforce`"
      fi
      if [ "${secfg}" = "Enforcing" ] ||
         [ "${secfg}" = "" ]; then
         meet_secfg="no"
      fi

      if [ ${arch} = "x86_64" ]; then
         if [ ${OS_MAJOR} != "6" -o ${OS_MINOR} -lt "6" ] &&
            [ ${OS_MAJOR} != "7" ] &&
            [ ${OS_MAJOR} != "8" ] ; then
            meet_distro="no"
         fi
      fi
      ;;

   ${DISTRO_ID_UBUNTU})

      if [ ${arch} = "x86_64" ]; then
         if [ ${OS_MAJOR} != "16" -a ${OS_MAJOR} != "18" ] ||
            [ ${OS_MINOR} != "04" ]; then
            meet_distro="no"
         fi
      fi
      ;;

   ${DISTRO_ID_SUSE})

      /sbin/SuSEfirewall2 status > /dev/null 2>&1
      if [ $? = "0" ]; then
         meet_secfg="no"
      fi

      if [ -f /etc/SuSE-release ]; then
         suse_desc="`cat /etc/SuSE-release`"
         suse_major=`echo "$suse_desc" | grep VERSION | cut -d " " -f 3`
         suse_minor=`echo "$suse_desc" | grep PATCHLEVEL | cut -d " " -f 3`
         if [ ${arch} = "x86_64" ]; then
            if [ ${suse_major} != "11" -o ${suse_minor} != "4" ] &&
               [ ${suse_major} != "12" -o ${suse_minor} != "3" ] &&
               [ ${suse_major} != "12" -o ${suse_minor} != "5"  ] ; then
               meet_distro="no"
            fi
         fi
      else
         echo_term "Can't find SUSE minor version"
         meet_distro="no"
      fi

      ;;

   *)
      ;;
   esac

   if [ ${OS_MAJOR} = "6" -a ${OS_MINOR} -ge "6" ] ; then
      if [ "${pcscd_version}" != "1.7.4." ] ||
         [ "${pcscd_version}" = "" ]; then
         meet_pcscd="no"
      fi
   else
      if [ "${pcscd_version}" != "1.8.8." ] ||
         [ "${pcscd_version}" = "" ]; then
         meet_pcscd="no"
      fi
   fi

   if [ "${meet_distro}" = "no" ] ||
      [ "${meet_secfg}" = "no" ] ||
      [ "${meet_pcscd}" = "no" ]; then
      meet_all_requirements="no"
   fi

   if [ "${meet_all_requirements}" = "no" ]; then
      echo_term -n "Failed to install smartcard redirection module, "
      echo_term "because the system doesn't meet the following requirement(s):"
      error_num=1

      if [ "${meet_distro}" = "no" ]; then
         echo_term "$error_num, The host system should be RedHatEnterpriseWorkstation 7.X/8.X or Ubuntu 18.4/16.4 Or SLED12SP3/SLES12SP3/SLES12SP5" && error_num=$(($error_num+1))
      fi

      if [ "${meet_secfg}" = "no" ]; then
         echo_term "$error_num, SELinux should be Disabled or Permissive or Firewall should be disabled." && error_num=$(($error_num+1))
      fi

      if [ "${meet_pcscd}" = "no" ]; then
         echo_term "$error_num, The pcsclite verison should be 1.8.8 for redhat 7.X/8.X or Ubuntu 18.4/16.4 or SLED12SP3/SLES12SP3/SLES12SP5"
      fi

      exit
   else
      echo_log "The host system meets the Smartcard requirements."
   fi
}

#
# Check audio in support environment
# 1. Redhat/CentOS 7.2/3/4, Ubuntu 1404/1604/1804, SLED 11/12 SLES 12 supported
# 2. X86 64bit
# 3. libudev.so.0 must be installed
#
verify_audioin_environment()
{

   local arch="`uname -i`"

   local meet_all_requirements="yes"
   local meet_distro="no"
   local meet_libudev="yes"

   #
   # AudioIn redirection limitation
   #
   case "$DISTRO_ID" in
   ${DISTRO_ID_CENTOS}|\
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_RHEL_WORKSTATION})

      if [ ${arch} = "x86_64" ]; then
         if [ ${OS_MAJOR} -eq "7" -a ${OS_MINOR} -ge 2 -o ${OS_MAJOR} -eq "8" ]; then
            meet_distro="yes"
            ldconfig -p |grep libudev.so >/dev/null 2>&1
            if [ "$?" != "0" ]; then
               meet_libudev="no"
            else
               [ ! -f /lib64/libudev.so.0 ] && cd /lib64 && \
                  ln -sf `ldconfig -p |grep libudev.so |awk 'NR==1{print $1}'` libudev.so.0 && cd -
            fi
         fi
      fi
      ;;

   ${DISTRO_ID_UBUNTU})

      if [ ${OS_MINOR} -eq "04" ] && [ ${arch} = "x86_64" ]; then
         if [ ${OS_MAJOR} -eq "14" ] || [ ${OS_MAJOR} -eq "16" ] || [ ${OS_MAJOR} -eq "18" ]; then
            meet_distro="yes"
            ldconfig -p |grep libudev.so >/dev/null 2>&1
            if [ "$?" != "0" ]; then
               meet_libudev="no"
            else
               [ ! -f /lib/x86_64-linux-gnu/libudev.so.0 ] && cd /lib/x86_64-linux-gnu && \
                  ln -sf `ldconfig -p |grep libudev.so |awk 'NR==1{print $1}'` libudev.so.0 && cd -
            fi
         fi
      fi
      ;;
   ${DISTRO_ID_SUSE})

      if [ ${arch} = "x86_64" ]; then
         if [ ${OS_MAJOR} -eq "11" ] || [ ${OS_MAJOR} -eq "12" ]; then
            meet_distro="yes"
            ldconfig -p |grep libudev.so >/dev/null 2>&1
            if [ "$?" != "0" ]; then
               meet_libudev="no"
            else
               local libudev=`ldconfig -p |grep libudev.so |awk 'NR==1{print $NF}'`
               libudev_path=${libudev%/*}
               [ ! -f "${libudev_path}/libudev.so.0" ] && cd "$libudev_path" && \
                 ln -sf "$libudev" libudev.so.0 && cd -
            fi
         fi
      fi
      ;;
   *)
      ;;
   esac


   if [ "${meet_distro}" = "no" ] ||
      [ "${meet_libudev}" = "no" ]; then
      meet_all_requirements="no"
   fi

   if [ "${meet_all_requirements}" = "no" ]; then
      echo_term -n "Failed to install audio in redirection module, "
      echo_term "because the system doesn't meet the following requirement(s):"
      error_num=1

      if [ "${meet_distro}" = "no" ]; then
         echo_term "$error_num, The audio in feature has not been supported on the host system ($DISTRO_ID $OS_MAJOR.$OS_MINOR) currently." && error_num=$(($error_num+1))
      fi

      if [ "${meet_libudev}" = "no" ]; then
         echo_term "$error_num, libudev.so.0 should be installed." && error_num=$(($error_num+1))
      fi

      exit
   else
      echo_log "The host system meets the Audio In requirements."
   fi
}

#
# Check USB support environment
# 1. X86 64bit
# 2. VHCI driver must be installed
# 3. libudev.so.0 must be installed
#
verify_usb_environment()
{
   local arch="`uname -i`"
   local meet_all_requirements="yes"
   local meet_arch="no"
   local meet_libudev="no"
   local meet_vhci="yes"
   local version=

   if [ "${arch}" = "x86_64" ]; then
      meet_arch="yes"
      verify_udev_environment
      if [ "$?" = "0" ];then
         meet_libudev="yes"
      fi
   fi

   if [ "${meet_libudev}" = "yes" ]; then
      stat /lib/modules/`uname -r`/kernel/drivers/usb/host/usb-vhci-hcd.ko > /dev/null 2>&1 && \
      stat /lib/modules/`uname -r`/kernel/drivers/usb/host/usb-vhci-iocifc.ko > /dev/null 2>&1
      if [ "$?" = "0" ]; then
         version="`modinfo -F"version" usb-vhci-iocifc`"
         if [ "$version" !=  "1.15.0" ]; then
            echo_term "Error: VHCI modules installed in the system maybe not the correct version. Please refer to product guide to install VHCI"
            exit
         fi
      else
         extract_vhci
         if [ "$?" = "0" ]; then
            echo_term "Warning: USB Redirection depends on the VHCI module, and Horizon Agent contains the compiled VHCI module for cuurent Linux kernel `uname -r`."
            echo_term "         Installing VHCI driver ..."
            echo_term "         You need compile your own VHCI module and install it to Linux kernel if you upgrade or downgrade Linux kernel after Horizon Agent installation."
            echo_term "         Please refer to product guide for detailed information."
         else
            meet_vhci="no"
         fi
      fi
   fi

   if [ "${meet_arch}" = "no" ] ||
      [ "${meet_vhci}" = "no" ] ||
      [ "${meet_libudev}" = "no" ]; then
      meet_all_requirements="no"
   fi

   if [ "${meet_all_requirements}" = "no" ]; then
      echo_term -n "Failed to install USB redirection module, "
      echo_term "because the system doesn't meet the following requirement(s):"
      error_num=1

      if [ "${meet_arch}" = "no" ]; then
         echo_term "The USB feature has only been supported on the 64-bit system currently."
      fi

      if [ "${meet_vhci}" = "no" ]; then
         echo_term "VHCI driver should be installed."
         echo_term "USB Redirection depends on the VHCI module, and Horizon Agent does not contain the compiled VHCI module for cuurent Linux kernel `uname -r`."
         echo_term "Please build your own VHCI module and install it to Linux kernel."
         echo_term "Please refer to product guide for detailed information."
      fi

      if [ "${meet_libudev}" = "no" ]; then
         echo_term "libudev.so.0 should be installed."
      fi

      exit
   else
      echo_log "The host system meets the USB requirements."
   fi
}

verify_truesso_environment_for_ubuntu()
{
   local arch="`uname -i`"

   local meet_all_requirements="yes"
   local meet_distro="no"
   local meet_pam_pkcs11="no"

   if [ ${OS_MAJOR} = "16" -o ${OS_MAJOR} = "18" ] && [ ${OS_MINOR} = "04" ]; then
      [ "${arch}" = "x86_64" ] && meet_distro="yes"
      dpkg -s libpam-pkcs11 > /dev/null 2>&1
      if [ "$?" = "0" ]; then
         meet_pam_pkcs11="yes"
      fi
   fi

   if [ "${meet_distro}" = "no" ] ||
      [ "${meet_pam_pkcs11}" = "no" ]; then
      meet_all_requirements="no"
   fi

   if [ "${meet_all_requirements}" = "no" ]; then
      echo_term "Failed to install TrueSSO module, "
      echo_term "because the system doesn't meet the following requirement(s):"

      if [ "${meet_distro}" = "no" ]; then
         echo_term "The host system should be Ubuntu16.04/18.04 64bit."
         exit
      fi

      if [ "$meet_pam_pkcs11" = "no" ]; then
         echo_term "The libpam_pkcs11 should be installed."
      fi

      exit
   else
      echo_log "The host system meets the TrueSSO requirements."
   fi
}

#
# Check truesso support environment
# 1. RHEL/CentOS 7/8
# 2. x86 64bit
# 3. pam_krb5 installed
# 4. krb5-pkinit installed
#
verify_truesso_environment_for_rhel()
{
   local arch="`uname -i`"

   if [ "$OS_MAJOR" = "7" ]; then
      local meet_pam_krb5="no"
      local meet_krb5_pkinit="no"

      rpm -q pam_krb5 > /dev/null 2>&1
      [ "$?" = "0" ] && meet_pam_krb5="yes"

      rpm -q krb5-pkinit > /dev/null 2>&1
      [ "$?" = "0" ] && meet_krb5_pkinit="yes"

      if [ "${meet_pam_krb5}" = "no" ] ||
         [ "${meet_krb5_pkinit}" = "no" ]; then
         echo_term "Failed to install TrueSSO module, "
         echo_term "because the system doesn't meet the following requirement(s):"
         error_num=1

         if [ "$meet_pam_krb5" = "no" ]; then
            echo_term "$error_num, The pam_krb5 should be installed." && error_num=$(($error_num+1))
         fi

         if [ "$meet_krb5_pkinit" = "no" ]; then
            echo_term "$error_num, The krb5-pkinit should be installed."
         fi

         exit
      fi
   fi

   if [ "$OS_MAJOR" != "7" ] && [ "$OS_MAJOR" != "8" ]; then
      echo_term "Failed to install TrueSSO module, "
      echo_term "because the system doesn't meet the following requirement(s):"
      echo_term "The host system should be RHEL/CentOS 7/8 64bit."
      exit
   fi

   echo_log "The host system meets the TrueSSO requirements."
}

#
# Check truesso support environment
# 1. SUSE 12
# 2. x86 64bit
# 3. pam_krb5 installed
# 4. krb5-plugin-preauth-pkinit installed
#
verify_truesso_environment_for_suse()
{
   local arch="`uname -i`"

   local meet_all_requirements="yes"
   local meet_distro="no"
   local meet_pam_krb5="no"
   local meet_krb5_pkinit="no"

   if [ "$OS_MAJOR" = "12" ]; then
      [ "${arch}" = "x86_64" ] && meet_distro="yes"

      rpm -q pam_krb5 > /dev/null 2>&1
      [ "$?" = "0" ] && meet_pam_krb5="yes"

      rpm -q krb5-plugin-preauth-pkinit > /dev/null 2>&1
      [ "$?" = "0" ] && meet_krb5_pkinit="yes"
   fi

   if [ "${meet_distro}" = "no" ] ||
      [ "${meet_pam_krb5}" = "no" ] ||
      [ "${meet_krb5_pkinit}" = "no" ]; then
      meet_all_requirements="no"
   fi

   if [ "${meet_all_requirements}" = "no" ]; then
      echo_term "Failed to install TrueSSO module, "
      echo_term "because the system doesn't meet the following requirement(s):"
      error_num=1

      if [ "${meet_distro}" = "no" ]; then
         echo_term "The host system should be SUSE12."
         exit
      fi

      if [ "$meet_pam_krb5" = "no" ]; then
         echo_term "$error_num, The pam_krb5 should be installed." && error_num=$(($error_num+1))
      fi

      if [ "$meet_krb5_pkinit" = "no" ]; then
         echo_term "$error_num, The krb5-plugin-preauth-pkinit should be installed."
      fi

      exit
   else
      echo_log "The host system meets the TrueSSO requirements."
   fi
}

verify_truesso_environment()
{
   case "$DISTRO_ID" in
   ${DISTRO_ID_CENTOS}|\
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_RHEL_WORKSTATION})
      verify_truesso_environment_for_rhel
      ;;
   ${DISTRO_ID_UBUNTU})
      verify_truesso_environment_for_ubuntu
      ;;
   ${DISTRO_ID_SUSE})
      verify_truesso_environment_for_suse
      ;;
   *)
      echo_term "TrueSSO is only supported on RHEL/CentOS 7/8, Ubuntu16.04/18.04, SLSE/SLSD 12"
      ;;
   esac
}

start_service() {
   serviceStart viewagent

   # start shutdown service
   serviceStart viewagent_shutdown

   # Show user to check "Recommended Video Memory (vRAM) Settings" for vSGA
   # TODO: vSGA supported for multiple session
   if [ $SINGLETON_MODE = "true" ]; then
      show_vRAM_Settings
   fi

   # add the vmware community link to show where customers can post questions
   echo_term " "
   echo_term -n "If you have any questions or issues regarding Linux VDI, please start a discussion at "
   echo_term -n "https://communities.vmware.com/community/vmtn/horizon/vmware-horizon-for-linux, "
   echo_term -n "and we will respond to you as soon as possible."
   echo_term " "
   #
   # Reminder user to do manual reboot so that
   # the changing xorg.conf and SingleSignOn
   # can take effect
   #
   echo_term "You must restart your system for the configuration changes made to the VMware Horizon Agent to take effect."
   echo_term " "
   echo_term "Installation done"
   echo_term " "
}

restart_system() {
   type reboot >/dev/null 2>&1
   if [ "$?" = "0" ] && [ "$AUTO_RESTART" = "yes" ]; then
      echo_term "restarting..."
      reboot
   fi
}

#
# Check the commands used in the script
# Give a kindly reminder when some required commands is not installed
#
command_check()
{
   COMMAND_LIST=$1
   while [ "$COMMAND_LIST" != "" ]
   do
      COMMAND_NAME=`echo "$COMMAND_LIST" | cut -f1 -d":"`
      type "$COMMAND_NAME" >/dev/null 2>&1
      if [ "$?" != "0" ]; then
         if [ "$COMMAND_NAME" != "vmware-rpctool" ]; then
            echo "The required command "$COMMAND_NAME" is currently not installed. Please ask your administrator to install the package first."
            exit
         else
            if [ "$MANAGED" = "true" ]; then
               echo "Fails to find vmware-rpctool under PATH , please follow the product guide to install VMTools."
               exit
            fi
         fi
      fi
      COMMAND_LIST=${COMMAND_LIST#*:}
   done
}

#
# Check the commands in all kinds of systems
# Please remember to add your new commands into the command list
# The command list must be end with ":"
#
common_command_check(){
   command_check which:hexdump:hostname:getent:sudo:ping:tar:passwd:usermod:pidof:\
grep:awk:sed:lsof:dirname:zenity:pax11publish:xauth:xset:xinput:xrandr:lspci:\
domainname:dmidecode:md5sum:
}

#
# Check the commands only used in some specific system
#
SUSE_command_check(){
   if [ "$DISTRO_ID" = "$DISTRO_ID_SUSE" ]; then
      command_check useradd:rcSuSEfirewall2:
   fi
}

none_SUSE_command_check(){
   if [ "$DISTRO_ID" != "$DISTRO_ID_SUSE" ]; then
      if [ "$DISTRO_ID" = "$DISTRO_ID_NEOKYLIN" ]; then
         command_check dconf:
      fi
      if [ "$DISTRO_ID" != "$DISTRO_ID_UBUNTU" ]; then
         command_check useradd:
      fi
      command_check adduser:
   fi
}

chkconfig_command_check(){
   if [ "$SERVICE_MGR" = "$SERVICE_MGR_SYSV" ]; then
   command_check chkconfig:
   fi
}

special_command_check()
{
   SUSE_command_check
   none_SUSE_command_check
   chkconfig_command_check
}

virtual_graphic_driver_check()
{
   #
   # For RedHat 7.1 & CentOS 7.1, we need to check the following drivers version if the target machine does not use NVIDIA graphic driver,
   # these dirvers version can influence the vSGA function.
   #

   lspci -v |grep "Kernel driver in use: nvidia"
   if [ "$?" != "0" ]; then

      #
      # Check the package mesa-libxatracker version is correct or not
      # If the version is lower than 10.6.5, the installer should print error message and exit;
      # If the version is equal to 10.6.5, the release number must not be lower than 3.20150824.el7
      #
      package_minimum_version_compare mesa-libxatracker $MINVERSION_MESA_LIBXATRACKER $MINRELEASE_MESA_LIBXATRACKER

      #
      # Check the package xorg-x11-drv-vmware version is correct or not
      # If the version is lower than 13.0.2, the installer should print error message and exit;
      # If the version is equal to 13.0.2, the release number must not be lower than 7.20150211git8f0cf7c.el7
      #
      # TODO: we skip to check xorg-x11-drv-vmware since the new
      # xorg-x11-drv-vmware version can't work in RHEL 7.1 now. This
      # will result we can't support vSGA in RHEL 7.1. We should add
      # this checking back after the RHEL fix the xorg-x11-drv-vmware bug.
      #
      #package_minimum_version_compare xorg-x11-drv-vmware $MINVERSION_XORG_X11_DRV_VMWARE $MINRELEASE_XORG_X11_DRV_VMWARE

      #
      # Check the package mesa-private-llvm version is correct or not
      # If the version is lower than 3.6.2, the installer should print error message and exit;
      # If the version is equal to 3.6.2, the release number must not be lower than 2.el7
      #
      package_minimum_version_compare mesa-private-llvm $MINVERSION_MESA_PRIVATE_LLVM $MINRELEASE_MESA_PRIVATE_LLVM

      #
      # Check the package mesa-dri-drivers version is correct or not
      # If the version is lower than 10.6.5, the installer should print error message and exit;
      # If the version is equal to 10.6.5, the release number must not be lower than 3.20150824.el7
      #
      package_minimum_version_compare mesa-dri-drivers $MINVERSION_MESA_DRI_DRIVERS $MINRELEASE_MESA_DRI_DRIVERS

   fi
}

python_module_check_for_ubuntu()
{
   local meet_all_requirements="yes"
   local meet_python="yes"
   local meet_dbus="yes"
   local meet_gobject="yes"

   if [ ${OS_MAJOR} = "18" ] && [ ${OS_MINOR} = "04" ]; then
      command -v python >/dev/null 2>&1
      if [ "$?" != "0" ]; then
         meet_python="no"
      else
         python --version 2>&1 | grep -q "Python 2" || meet_python="no"
      fi
   fi

   dpkg --get-selections | grep python-dbus > /dev/null 2>&1
   if [ "$?" != "0" ]; then
      meet_dbus="no"
   fi

   dpkg --get-selections | grep python-gobject > /dev/null 2>&1
   if [ "$?" != "0" ]; then
      meet_gobject="no"
   fi

   if [ "${meet_python}" = "no" ] ||
      [ "${meet_dbus}" = "no" ]   ||
      [ "${meet_gobject}" = "no" ]; then
      meet_all_requirements="no"
   fi

   if [ "${meet_all_requirements}" = "no" ]; then
      echo_term -n "Failed to install Horizon agent on Ubuntu"${OS_MAJOR}${OS_MINOR}", "
      echo_term "because the distribution doesn't meet the following requirement(s):"
      error_num=1

      if [ "${OS_MAJOR}" = "18" ] && [ "${meet_python}" = "no" ]; then
         echo_term "$error_num. The host distribution should have python module (python2.7) installed." && error_num=$(($error_num+1))
      fi

      if [ "${meet_dbus}" = "no" ]; then
         echo_term "$error_num. The host distribution should have python-dbus module installed." && error_num=$(($error_num+1))
      fi

      if [ "${meet_gobject}" = "no" ]; then
         echo_term "$error_num. The host distribution should have python-gobject module installed." && error_num=$(($error_num+1))
      fi

      exit
   fi
}

gdm_module_check_for_ubuntu1804()
{
  grep -q "^/usr/sbin/gdm3" /etc/X11/default-display-manager
  if [ "$?" != "0" ]; then
    echo_term -n "Failed to install Horizon agent on Ubuntu1804, "
    echo_term "because the distribution doesn't meet the following requirement(s):"
    echo_term "The host distribution should select gdm3 as display manager, please select it with command \"dpkg-reconfigure gdm3\"."
    echo_term "Please ignore \"gdm.service is not active, cannot reload.\" error if there is after above command execution, reboot then install viewagent again."
    exit
  fi
}

driver_package_check_for_redhat()
{
   if [ ${OS_MAJOR} -eq 7 -a ${OS_MINOR} -eq 1 ]; then
      virtual_graphic_driver_check
   fi
}

driver_package_check_for_centos()
{
   if [ ${OS_MAJOR} -eq 7 -a ${OS_MINOR} -eq 1 ]; then
      virtual_graphic_driver_check
   fi
}

driver_package_check_for_ubuntu()
{
   if [ ${OS_MAJOR} = "16" -o ${OS_MAJOR} = "18" ] && [ ${OS_MINOR} = "04" ]; then
      python_module_check_for_ubuntu
   fi

   if [ ${OS_MAJOR} = "18" ] && [ ${OS_MINOR} = "04" ]; then
      gdm_module_check_for_ubuntu1804
   fi
}

driver_package_check()
{
   case "$DISTRO_ID" in
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_RHEL_WORKSTATION})
      driver_package_check_for_redhat
      ;;
   ${DISTRO_ID_CENTOS})
      driver_package_check_for_centos
      ;;
   ${DISTRO_ID_SUSE})
      ;;
   ${DISTRO_ID_UBUNTU})
      driver_package_check_for_ubuntu
      ;;
   ${DISTRO_ID_NEOKYLIN}|\
   ${DISTRO_ID_KYLIN})
      ;;
   esac
}

#
# Check the system if there are more than one networks by detecting IP addresses.
#
multiple_networks_detect()
{
   local network_nums="0"
   which ifconfig >/dev/null 2>&1
   if [ "$?" = "0" ]; then
      network_nums="`ifconfig | grep "inet " | wc -l`"
   else
      which ip >/dev/null 2>&1 && network_nums="`ip address | grep "inet " | wc -l`"
   fi

   if [ $network_nums -ge 3 ]; then
      echo "Warning: Multiple networks detected ! You can set the option Subnet in /etc/vmware/viewagent-custom.conf to select the correct IP Address for the Horizon Agent."
   fi
}

#
# Configure limits
#
configure_recource_limits()
{
    umask 022
    # dbus connection limit
    local dbus_1="/usr/share/dbus-1"
    local dbus_systemd="${dbus_1}/system.d"
    local vmw_dbus_conf="${dbus_systemd}/vmware.conf"
    if [ -d "$dbus_1" ]
    then
        if [ -d "$dbus_systemd" ] || mkdir -p $dbus_systemd
        then
            cat > $vmw_dbus_conf <<-\EOF
		<!-- This configuration file extends some dbus limitations to adapt to the multi-session feature. -->
		<!DOCTYPE busconfig PUBLIC "-//freedesktop//DTD D-Bus Bus Configuration 1.0//EN"
		"http://www.freedesktop.org/standards/dbus/1.0/busconfig.dtd">
		<busconfig>
		    <limit name="max_connections_per_user">100000</limit>
		    <limit name="max_completed_connections">100000</limit>
		    <limit name="max_incomplete_connections">100000</limit>
		    <limit name="max_replies_per_connection">100000</limit>
		</busconfig>
EOF
        fi
    fi

    # Unlimit nproc for gdm group to fit for multi-session,
    # otherwise gnome-session-binary will reach the default limit.
    local security="/etc/security"
    local limits_d="${security}/limits.d"
    local vmw_nproc_conf="${limits_d}/99-vmw-nproc.conf"
    if [ -d "$security" ]
    then
        [ ! -d "$limits_d" ] && mkdir -p "$limits_d"
        cat > $vmw_nproc_conf <<-\EOF
		# Unlimit nproc for gdm group
		@gdm        soft    nproc     unlimited
EOF

        if [ "$SINGLETON_MODE" = "false" ]; then
           cat >> $vmw_nproc_conf <<-\EOF
		# Set max fd number for process of gdm and root
		gdm         -    nofile     65535
		root        -    nofile     65535
EOF
        fi
    fi

    # Sysctl: inotify
    local sysctl_d="/etc/sysctl.d"
    local vmw_sysctl_conf="${sysctl_d}/99-vmw-inotify.conf"
    [ ! -d "$sysctl_d" ] && mkdir -p "$sysctl_d"
    cat > $vmw_sysctl_conf <<-\EOF
	# Increase the limit of inotify
	fs.inotify.max_queued_events = 100000
	fs.inotify.max_user_instances = 100000
	fs.inotify.max_user_watches = 100000
EOF

    # Generate uninstall script
    local uninstall_script="$VMW_UNINSTALL_SCRIPTS_DIR/00-resource-limts"
    cat > $uninstall_script <<-EOF
	#!/bin/sh
	rm -f $vmw_dbus_conf >/dev/null 2>&1
	rm -f $vmw_nproc_conf >/dev/null 2>&1
	rm -f $vmw_sysctl_conf >/dev/null 2>&1
EOF
    chmod +x $uninstall_script
}

configure_xdmcp()
{
    "$(dirname $0)/bin/Xdmcp.sh" -m ${SINGLETON_MODE}
}

#
# main
#

#
# command check first
#
common_command_check

#
# Ensure script is executed as root.
#
if [ "`id -u`" != "0" ]; then
   echo "Executing horizon agent installer as superuser"

   #
   # Note, environment is required to spot remote / local session
   #
   exec sudo -E $0 $@

   #
   # The following should never be executed...
   #
   echo "install_viewagent.sh not run as superuser"
   exit
fi


#
# Ensure install_viewagent.sh is executed from the root of the extracted
# installer directory.
#
if [ ! -d bin ] ||
   [ ! -f Product.txt ] ||
   [ ! -d VMwareBlastServer ]; then
   echo "Please execute install_viewagent.sh from the extracted installer directory"
   exit
fi

SINGLETON_MODE="true"

# Parse multiple-charactors options
for i in "$@"
do
   shift
   case "$i" in
      -h|--help)
         usage_exit
         ;;
      --multiple-session)
         SINGLETON_MODE="false"
         echo "Install multiple session agent..."
         ;;
      *) set -- "$@" "$i"  ;;
   esac
done

#
# Include common constants and routines
#
. "`dirname $0`/bin/commonlib.sh"

#
# Initialise command
#
define_os_version
define_commands
identify_distribution
configure_distribution
special_command_check


#
# check if current installation is upgrade or fresh install
#
freshInstall="true"
if [ -f /usr/lib/vmware/viewagent/bin/uninstall_viewagent.sh ]; then
   freshInstall="false"
fi

#
# Get pre-installation's managed flag
#
preManagedFlag="false"
if [ -f "$AGENT_CFG_FILE" ]; then
   managedName="MANAGED="
   line=`grep "^$managedName" $AGENT_CFG_FILE`
   if [ "$?" = 0 ]; then
      preManagedFlag="`echo "$line" | awk 'BEGIN{FS="="}{print $2}'`"
   fi
fi

#
# check if current is a registered unmanaged agent
#
isRegistered="false"
if [ -f "$MACHINE_CFG_FILE" ] &&
   [ -f "$AGENT_CFG_FILE" ] &&
   [ -f "$CERT" ]; then
   isRegistered="true"
   echo_log "current agent already registered."
fi

#
# Verify architecture
#
if ! isPlatformArchSupported; then
   echo_term "Platform architecture (`uname -i`) not supported by this installer."
   exit
fi

#
# Check packages and drivers
#
driver_package_check

#
# Is agent already registered?.
# FIXME:command line interface varies among distributions
# and whether the agent is registered.
# this is ugly and confusing. May fix to be more
# friendly
if [ "$isRegistered" = "TRUE" ]; then
   INSTALLER_OPTS=$INSTALLER_OPTS_REG
else
   INSTALLER_OPTS=$INSTALLER_OPTS_UNREG
fi

NAME=
BROKER=
DOMAIN=
DN=
LDAP_USERNAME=
LDAP_PASSWORD=
INSTALL_FROM_DESKTOP=
NEW_MACHINE_REGISTRATION=
SCREDIR_ENABLE="no"
AUDIOIN_ENABLE="no"
USB_ENABLE="no"
COLLAB_ENABLE="no"
SHOW_EULA="yes"
AUTO_RESTART="no"
CDRSERVER_ENABLE="yes"
CLIPBOARD_ENABLE="yes"
FIPS_ENABLE="false"
IPC_ENABLE="true"
SSO_ENABLE="yes"
TRUESSO_ENABLE="no"
CUSTOMIZATION_STATE="false"
#
# Horizon Agent defaults to default Ubuntu distribution X server (Xorg)
#
KDC=
# for multi domain case, we need to specify the domain and kdc info for broker.
BROKERDOMAIN=
BROKERKDC=
MANAGED="true"
HOSTNAME="`hostname`"
RUNONCE="false"
LINKCLONE="false"
ISKUBUNTU="false"

while getopts $INSTALLER_OPTS o
do
   case "$o" in
   n)
      NAME="$OPTARG";;
   b)
      BROKER="$OPTARG";;
   d)
      DOMAIN="$OPTARG";;
   s)
      DN="$OPTARG";;
   u)
      LDAP_USERNAME="$OPTARG";;
   U)
      param="$OPTARG"
      case "$param" in
         yes)
            USB_ENABLE="yes"
            ;;
         no)
            USB_ENABLE="no"
            ;;
         *)
            usage_exit
            ;;
      esac
      ;;
   r)
      param="$OPTARG"
      case "$param" in
         yes)
            AUTO_RESTART="yes"
            ;;
         no)
            AUTO_RESTART="no"
            ;;
         *)
            usage_exit
            ;;
      esac
      ;;
   m)
      param="$OPTARG"
      case "$param" in
         yes)
            SCREDIR_ENABLE="yes"
            ;;
         no)
            SCREDIR_ENABLE="no"
            ;;
         *)
            usage_exit
            ;;
      esac
      ;;
   S)
      param="$OPTARG"
      case "$param" in
         yes)
            SSO_ENABLE="yes"
            ;;
         no)
            SSO_ENABLE="no"
            ;;
         *)
            usage_exit
            ;;
      esac
      ;;
   F)
      param="$OPTARG"
      case "$param" in
         yes)
            CDRSERVER_ENABLE="yes"
            ;;
         no)
            CDRSERVER_ENABLE="no"
            ;;
         *)
            usage_exit
            ;;
      esac
      ;;
   f)
     param="$OPTARG"
     case "$DISTRO_ID" in
         ${DISTRO_ID_RHEL_CLIENT}|\
         ${DISTRO_ID_RHEL_SERVER}|\
         ${DISTRO_ID_RHEL_WORKSTATION})
            case "$param" in
                yes)
                   FIPS_ENABLE="true"
                   ;;
                no)
                   FIPS_ENABLE="false"
                   ;;
                *)
                   usage_exit
                   ;;
            esac
            ;;
         *)
            echo "FIPS mode does not support current Distro. Please refer to the usage.\n"
            usage_exit
            ;;
     esac
     ;;
   a)
      param="$OPTARG"
      case "$param" in
         yes)
            AUDIOIN_ENABLE="yes"
            ;;
         no)
            AUDIOIN_ENABLE="no"
            ;;
         *)
            usage_exit
            ;;
      esac
      ;;
   A)
      #Accept EULA
      param="$OPTARG"
      case "$param" in
         yes)
               SHOW_EULA="no"
               ;;
          no)
               usage_exit
               ;;
           *)
               usage_exit
               ;;
      esac
      ;;
   C)
      param="$OPTARG"
      case "$param" in
         yes)
               CLIPBOARD_ENABLE="yes"
               ;;
          no)
               CLIPBOARD_ENABLE="no"
               ;;
           *)
               usage_exit
               ;;
      esac
      ;;
   p)
      LDAP_PASSWORD="$OPTARG";;
   X)
      INSTALL_FROM_DESKTOP="yes";;
   R)
      NEW_MACHINE_REGISTRATION="yes";;
   k)
      KDC="$OPTARG";;
   K)
      BROKERKDC="$OPTARG";;
   B)
      BROKERDOMAIN="$OPTARG";;
   M)
      param="$OPTARG"
      case "$param" in
         yes)
               MANAGED="true"
               echo_log "Install managed agent..."
               ;;
          no)
               MANAGED="false"
               echo_log "Install Unmanaged agent..."
               ;;
           *)
               usage_exit
               ;;
      esac
      ;;
   L)
      param="$OPTARG"
      case "$param" in
         yes)
              #LINKCLONE="true"
              #echo_log "activate composer agent"
              ;;
          no)
              #LINKCLONE="false"
              #echo_log "don't activate composer agent"
              ;;
           *)
              #usage_exit
              ;;
      esac
      ;;
   T)
      param="$OPTARG"
      case "$param" in
         yes)
               TRUESSO_ENABLE="yes"
               ;;
         no)
               TRUESSO_ENABLE="no"
               ;;
         *)
               usage_exit
               ;;
      esac
      ;;
   esac
done

#
# For multiple-session mode, USB and Smartcard are not supported currently
# so, don't install the package accordingly, please remove following check
# when they are supported on multiple-session mode.
# TODO: This piece of code will be covered by a common function which is used
# to verify the arguments input by end-user
#
if [ "$SINGLETON_MODE" = "false" ]; then
   SCREDIR_ENABLE="no"
   USB_ENABLE="no"
   verify_rdeSvc_environment
fi

command_check vmware-rpctool:

# Check if need to show EULA to End-user
if [ "$SHOW_EULA" = "yes" ]; then
   verify_EULA_agreement
   # Check if need to show FIPS statement to End-user
   if [ "$FIPS_ENABLE" = "true" ]; then
      verify_FIPS_statement
   fi
else
   if [ "$FIPS_ENABLE" = "true" ]; then
      show_FIPS_statement
   fi
fi

# Check if meet the requirements
if [ "$SCREDIR_ENABLE" = "yes" ]; then
   verify_scredir_environment
fi

if [ "$AUDIOIN_ENABLE" = "yes" ]; then
   verify_audioin_environment
fi

if [ "$USB_ENABLE" = "yes" ]; then
   verify_usb_environment
fi

if [ "$TRUESSO_ENABLE" = "yes" ]; then
   verify_truesso_environment
fi

#
# Detect multiple networks
#
multiple_networks_detect

#
# Check SLED requirements
#
if [ "$DISTRO_ID" = "$DISTRO_ID_SUSE" ]; then
   verify_SLED_environment
fi

#
# Check Redhat/CentOS requirements
#
verify_RHELOrCentOS_environment

#
# Check Ubuntu requirements
#
if [ "$DISTRO_ID" = "$DISTRO_ID_UBUNTU" ]; then
   verify_Ubuntu_environment
fi

# check if there is VMwareBlastServer running
pidof VMwareBlastServer >/dev/null 2>&1
if [ "$?" = "0" ]; then
   # Prompt user to logout current session, then run install script
   echo_term "Agent installation cannot continue due to an active session has been detected, you must restart or logout your system at first."
   exit
fi


#
# read LDAP password in secure mode
#
read_password_secure

#
# check the input paramester immediately after getopts.
#
verify_fill_input_parameters


uninstall_existing_package

#
# Running on local desktop or remotely?
#
REMOTE_SESSION=
if [ -n "$SSH_CLIENT" ]; then
   REMOTE_SESSION="yes"
fi


if [ "${NEW_MACHINE_REGISTRATION}x" != "x" ]; then
   echo_log "Forcing new machine registration (removing old machine configuration)"
   rm -f $MACHINE_CFG_FILE
   rm -f $AGENT_CFG_FILE
else
   if [ -f "$MACHINE_CFG_FILE" ] && [ -f "$AGENT_CFG_FILE" ]; then
      upgrade_blast_server_port
      echo_term "Warning: Machine config file detected,"
      echo_term "         refreshing installation for pre-existing machine registration."
   fi
fi


#
# Create install path if not present
#
mkdir -p $VIEWAGENT_PATH/bin

# Setup uninstall scripts dir
test -d $VMW_UNINSTALL_SCRIPTS_DIR || mkdir -p $VMW_UNINSTALL_SCRIPTS_DIR

verify_environment

# Enable XDMCP
configure_xdmcp

backup_pam
create_blast_account
extract_viewagent
extract_jre
extract_vncserver
extract_vchan_plugins
if [ "$SINGLETON_MODE" = "false" ]; then
	extract_rdeserver
fi
extract_help_desk

extract_session_collaboration

if [ "$CLIPBOARD_ENABLE" = "yes" ]; then
   extract_mks_vchan_server
fi

#
# Check if cdrserver is enabled
#
if [ "$CDRSERVER_ENABLE" = "yes" ]; then
   extract_cdrserver
fi

if [ "$USB_ENABLE" = "yes" ]; then
   extract_usbredir
fi

if [ "$MANAGED" = "false" ]; then
   register_viewagent
else
   touch "$MACHINE_CFG_FILE"
   chmod 600 "$MACHINE_CFG_FILE"
fi
if [ "$SSO_ENABLE" = "yes" ]; then
   extract_sso
fi
if [ "$SCREDIR_ENABLE" = "yes" ]; then
   extract_scredir
fi
if [ "$AUDIOIN_ENABLE" = "yes" ]; then
   extract_audioin
fi
configure_selinux
write_env_file
load_custom_configuration
setup_viewagent_service
setup_shutdown_script_service
if [ "$SINGLETON_MODE" = "false" ]; then
   customise_multiple_session set
fi
customise_desktop set "$MANAGED"
configure_xorg_conf $SINGLETON_MODE install
configure_firewall
configure_ipc
"$(dirname $0)/bin/XdmcpSecurity.sh" install
configure_recource_limits
start_service
restart_system
