/*
 * ************************************************************************
 *
 * Copyright 2019-2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 *
 * ************************************************************************
 */

const Cairo = imports.cairo;
const Lang = imports.lang;
const Gio = imports.gi.Gio;
const GLib = imports.gi.GLib;
const Gtk = imports.gi.Gtk;
const Gdk = imports.gi.Gdk;

var winType = {
    POPUP: 1,
    TOPLEVEL: 2,
};

var winStyle = {
    SOLID: 0,
    TRANSPARENT: 1,
};

var ShadowWindow = new Lang.Class({
    Name: "ShadowWindow",

    _x: 0,
    _y: 0,
    _width: 0,
    _height: 0,
    _type: winType.POPUP,
    _style: winStyle.SOLID,
    _window: null,

    _init: function(){
    },

    _destroy: function(){
        this._remove();
    },

    _create: function(){
        let css = new Gtk.CssProvider();
        css.load_from_data(' * { box-shadow: none;' + ' border: none;' +
                           ' border-radius: 0px; }');

        if (this._type == winType.POPUP) {
            this._window = new Gtk.Window({type: Gtk.WindowType.POPUP});
        } else if (this._type == winType.TOPLEVEL) {
            this._window = new Gtk.Window({type: Gtk.WindowType.TOPLEVEL});
        } else {
            global.log("windows type error!");
            return;
        }
        this._window.get_style_context().add_provider(css, 0);

        this._window.title = "VMWARE Shadow Window";

        /* Here are a few ways we can customize our window.
           Try uncommenting them or changing their values! */
        this._window.set_default_size(this._width, this._height);
        this._window.move(this._x, this._y);
        this._window.set_opacity(0);
        this._window.set_decorated(false);
        this._window.set_skip_taskbar_hint(true);
        this._window.set_can_focus(false);
        this._window.set_can_default(false);
        this._window.set_keep_above(false);

        if (this._style == winStyle.TRANSPARENT) {
            global.log('Set shadow window transparent\n');
            this._window.connect('draw',
                                 (area, ctx) => { this._onDraw(area, ctx); });
            let screen = this._window.get_screen();
            let visual = screen.get_rgba_visual();
            if (visual && screen.is_composited()) {
                this._window.set_visual(visual);
            }
            this._window.set_app_paintable(true);
        }

        this._window.show_all();
    },

    _remove: function(){
        if (this._window != null) {
            this._window.destroy();
        }
        this._window = null;
    },

    _onDraw(area, cr) {
        cr.setSourceRGBA(0, 0, 0, 0);
        cr.setOperator(Cairo.Operator.SOURCE);
        cr.paint();
        cr.setOperator(Cairo.Operator.OVER);
    },

    draw: function(x, y, width, height, type, style){
        [this._x, this._y, this._width, this._height] = [x, y, width, height];
        this._type = type;
        this._style = style;
        this._remove();
        this._create();
    },

    hide: function(){
        this._remove();
    },

    redraw: function(){
        this._remove();
        this._create();
    },

});
