/*
 * ************************************************************************
 *
 * Copyright 2019-2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 *
 * ************************************************************************
 */
const Config = imports.misc.config;

const BoxPointer = imports.ui.boxpointer;
const Gtk = imports.gi.Gtk;
const Lang = imports.lang;
const Main = imports.ui.main;
const Meta = imports.gi.Meta;
const PopupMenu = imports.ui.popupMenu;
const WindowMenu = imports.ui.windowMenu.WindowMenu;
const WindowMenuMgr = imports.ui.windowMenu.WindowMenuManager;

const Me = imports.misc.extensionUtils.getCurrentExtension();
const ShadowWindowExtension = Me.imports.shadowWindowExtension;

var windowWithMenu = null;
var shadowWindow = {};
var raisedId, minimizedId;

let old_buildMenu = {};

let new_buildMenu = function(window) {
    let type = window.get_window_type();

    let item, itemWidth, itemHeight, itemX, itemY;

    item = this.addAction(_("Minimize"), () => {
        window.minimize();
    });

    if (!window.can_minimize())
        item.setSensitive(false);

    if (window.get_maximized()) {
        item = this.addAction(_("Unmaximize"), () => {
            window.unmaximize(Meta.MaximizeFlags.BOTH);
        });
    } else {
        item = this.addAction(_("Maximize"), () => {
            window.maximize(Meta.MaximizeFlags.BOTH);
        });
    }
    if (!window.can_maximize())
        item.setSensitive(false);

    item = this.addAction(_("Move"), event => {
        window.begin_grab_op(Meta.GrabOp.KEYBOARD_MOVING, true,
                             event.get_time());
    });
    if (!window.allows_move())
        item.setSensitive(false);

    item = this.addAction(_("Resize"), event => {
        window.begin_grab_op(Meta.GrabOp.KEYBOARD_RESIZING_UNKNOWN,
                             true, event.get_time());
    });
    if (!window.allows_resize())
        item.setSensitive(false);

    this.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());

    item = this.addAction(_("Close"), event => {
        window.delete(event.get_time());
    });
    if (!window.can_close())
        item.setSensitive(false);
};

let old_showWindowMenuForWindow = {};

let new_showWindowMenuForWindow_3_28 = function(window, type, rect) {
    let AppMenu = imports.ui.windowMenu.AppMenu;
    let menuType = (type == Meta.WindowMenuType.WM) ? WindowMenu : AppMenu;
    let menu = new menuType(window, this._sourceActor);

    this._manager.addMenu(menu);

    menu.connect('activate', () => {
        window.check_alive(global.get_current_time());
    });
    let destroyId = window.connect('unmanaged', () => {
        menu.close();
    });

    this._sourceActor.set_size(Math.max(1, rect.width),
                               Math.max(1, rect.height));
    this._sourceActor.set_position(rect.x, rect.y);
    this._sourceActor.show();

    menu.open(BoxPointer.PopupAnimation.NONE);
    menu.actor.navigate_focus(null, Gtk.DirectionType.TAB_FORWARD, false);
    menu.connect('open-state-changed', (menu_, isOpen) => {
        hide_shadowwindow();

        if (isOpen)
            return;

        this._sourceActor.hide();
        menu.destroy();
        window.disconnect(destroyId);
    });

    show_shadowwindow(window, menu);

};

let new_showWindowMenuForWindow_3_32 = function(window, type, rect) {
    if (type != Meta.WindowMenuType.WM)
        throw new Error('Unsupported window menu type');
    let menu = new WindowMenu(window, this._sourceActor);

    this._manager.addMenu(menu);

    menu.connect('activate', () => {
        window.check_alive(global.get_current_time());
    });
    let destroyId = window.connect('unmanaged', () => {
        menu.close();
    });

    this._sourceActor.set_size(Math.max(1, rect.width),
                               Math.max(1, rect.height));
    this._sourceActor.set_position(rect.x, rect.y);
    this._sourceActor.show();

    menu.open(BoxPointer.PopupAnimation.NONE);
    menu.actor.navigate_focus(null, Gtk.DirectionType.TAB_FORWARD, false);
    menu.connect('open-state-changed', (menu_, isOpen) => {
        hide_shadowwindow();

        if (isOpen)
            return;

        this._sourceActor.hide();
        menu.destroy();
        window.disconnect(destroyId);
    });

    show_shadowwindow(window, menu);

};

function close_window_menu() {
   let menuManager = Main.wm._windowMenuManager._manager;
   for (let i = 0; i < menuManager._menus.length; i++) {
       let menudata = menuManager._menus[i];
       menudata.menu.close();
   }
}

function show_shadowwindow(window, menu) {
    let themeNode = menu.actor.get_theme_node();
    let rise = themeNode.get_length('-arrow-rise');
    if (rise < 0) {
        global.log('Can\'t find -arrow-rise');
        rise = 0;
    }

    let [menuWidth, menuHeight] = menu.actor.get_transformed_size();
    menuWidth = Math.floor(menuWidth);
    menuHeight = Math.floor(menuHeight);
    menuHeight -= rise;

    let [menuX, menuY] = menu.actor.get_transformed_position();
    menuX = Math.floor(menuX);
    menuY = Math.floor(menuY);
    menuY += rise;

    shadowWindow.draw(menuX, menuY, menuWidth, menuHeight,
                      ShadowWindowExtension.winType.POPUP,
                      ShadowWindowExtension.winStyle.TRANSPARENT);

    windowWithMenu = window;
    if (windowWithMenu) {
        raisedId = windowWithMenu.connect('raised', on_menu_win_raised);
        minimizedId = windowWithMenu.connect('notify::minimized',
                                             on_menu_win_minimized);
    }
}

function hide_shadowwindow() {
    if (windowWithMenu) {
        windowWithMenu.disconnect(raisedId);
        windowWithMenu.disconnect(minimizedId);
        windowWithMenu = null;
    }

    shadowWindow.hide();
}

function on_menu_win_raised() {
    shadowWindow.redraw();
}

function on_menu_win_minimized() {
    close_window_menu();
}

function init() {
    shadowWindow = new ShadowWindowExtension.ShadowWindow();
    old_buildMenu = WindowMenu.prototype._buildMenu;
    old_showWindowMenuForWindow
            = WindowMenuMgr.prototype.showWindowMenuForWindow;
}

function enable() {
    WindowMenu.prototype._buildMenu = new_buildMenu;
    if (Config.PACKAGE_VERSION.startsWith("3.28")) {
        WindowMenuMgr.prototype.showWindowMenuForWindow
                          = new_showWindowMenuForWindow_3_28;
    } else if (Config.PACKAGE_VERSION.startsWith("3.32")) {
        WindowMenuMgr.prototype.showWindowMenuForWindow
                          = new_showWindowMenuForWindow_3_32;
    } else {
        global.log('Not Support ' + Config.PACKAGE_VERSION);
    }
}

function disable() {
    WindowMenu.prototype._buildMenu = old_buildMenu;
    WindowMenuMgr.prototype.showWindowMenuForWindow
                          = old_showWindowMenuForWindow;
}
