/*
 * ************************************************************************
 *
 * Copyright 2019-2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 *
 * ************************************************************************
 */
const Config = imports.misc.config;

const Lang = imports.lang;
const Main = imports.ui.main;
const Tweener = imports.ui.tweener;
const GLib = imports.gi.GLib;
const Meta = imports.gi.Meta;

const ExtensionUtils = imports.misc.extensionUtils;
const ExtensionSystem = imports.ui.extensionSystem;
const Mainloop = imports.mainloop;

const Me = imports.misc.extensionUtils.getCurrentExtension();
const DialogExtension = Me.imports.dialogExtension;
const MenuExtension = Me.imports.menuExtension;
const ScreenSaverExtension = Me.imports.screenSaverExtension;
const IBusPopupExtension = Me.imports.ibusPopupExtension;
const SwitcherPopupExtension = Me.imports.switcherPopupExtension;
const BackgroundExtension = Me.imports.bgExtension;

const panel_actor = Main.panel.actor;
const panel_box = panel_actor.get_parent();

const UBUNTU_DOCK_UUID = 'ubuntu-dock@ubuntu.com';
const RHEL_DOCK_UUID = 'window-list@gnome-shell-extensions.gcampax.github.com';
const RHEL8_DESKTOPICON_UUID = 'desktop-icons@gnome-shell-extensions.gcampax.github.com';

var ANIMATION_SPEED = 0.45;

/*
 * Panel's top does not have to be 0, if the primary screen is some pixels
 * "below" another one.
 */
let panel_y = panel_actor.get_y();
let panel_height = panel_actor.get_height();
let workareaChangeId;

let show_event = false;
let hide_event = false;

let focusWindow = null;
let focusWindowNotifyId;

let disableUbuntuDock = false;
let disableRhelDock = false;
let disableRhel8DesktopIcon = false;
let extensionChangedId;

function on_focus_window_notify() {
    let window = global.display.get_focus_window();
    if (window != null) {
        focusWindow = window;
    }

    // Close menu if the windowWithMenu is not the focus window
    if ((MenuExtension.windowWithMenu != null) &&
        (focusWindow != MenuExtension.windowWithMenu)) {
        MenuExtension.close_window_menu();
    }
}

function hide_panel() {
    Tweener.addTween(panel_actor, {
        translation_y: 0,
        time: ANIMATION_SPEED
    });
}

function show_panel() {
    Tweener.addTween(panel_actor, {
        translation_y: panel_height,
        time: ANIMATION_SPEED
    });
}

function init() {
    ScreenSaverExtension.init();
    BackgroundExtension.init();
    MenuExtension.init();
    DialogExtension.init();
    IBusPopupExtension.init();
    SwitcherPopupExtension.init();
}

function disable_extension(uuid) {
    let extension = ExtensionUtils.extensions[uuid];
    let retValue = false;

    global.log("Will disable extension [vmware-remote-app-helper]");
    if (extension &&
        extension.state == ExtensionSystem.ExtensionState.ENABLED) {
        Mainloop.timeout_add(0, () => {
            global.log("Will disable extension [" + uuid +"]");
            ExtensionSystem.disableExtension(uuid);
            return GLib.SOURCE_REMOVE;
        });
        retValue = true;
    }

    return retValue;
}

function enable() {
    global.log("Will enable extension [vmware-remote-app-helper]");

    // Disable background.
    BackgroundExtension.enable();

    // Init focusWindow
    focusWindow = global.display.get_focus_window();
    focusWindowNotifyId = global.display.connect('notify::focus-window',
                                                 on_focus_window_notify);

    // Hide the panel.
    show_event = Main.overview.connect('showing', show_panel);
    hide_event = Main.overview.connect('hiding', hide_panel);
    panel_box.set_position(0, panel_y - panel_height);
    hide_panel();
    if (Config.PACKAGE_VERSION.startsWith("3.28")) {
        workareaChangeId =
            global.screen.connect('workareas-changed', () => {
                    panel_actor.queue_relayout();
                    panel_box.set_position(0, panel_y - panel_height);
                    hide_panel();
                });
    } else if (Config.PACKAGE_VERSION.startsWith("3.32")) {
        workareaChangeId =
            global.display.connect('workareas-changed', () => {
                    panel_actor.queue_relayout();
                    panel_box.set_position(0, panel_y - panel_height);
                    hide_panel();
                });
    } else {
        global.log('Not Support ' + Config.PACKAGE_VERSION);
    }

    // Disable screen saver.
    ScreenSaverExtension.enable();

    // Enable menu, dialog and IMS popup extension.
    MenuExtension.enable();
    DialogExtension.enable();
    IBusPopupExtension.enable();
    SwitcherPopupExtension.enable();

    // Disable ubuntu's dash-to-dock and rhel's dock
    extensionChangedId = ExtensionSystem.connect('extension-state-changed', (data, extension) => {
        if (extension.state == ExtensionSystem.ExtensionState.ENABLED) {
            if (extension.uuid == UBUNTU_DOCK_UUID) {
                disableUbuntuDock = disable_extension(UBUNTU_DOCK_UUID);
            } else if (extension.uuid == RHEL_DOCK_UUID) {
                disableRhelDock = disable_extension(RHEL_DOCK_UUID);
            } else if (extension.uuid == RHEL8_DESKTOPICON_UUID) {
                disableRhel8DesktopIcon = disable_extension(RHEL8_DESKTOPICON_UUID);
            }
        }
    });
    disableUbuntuDock = disable_extension(UBUNTU_DOCK_UUID);
    disableRhelDock = disable_extension(RHEL_DOCK_UUID);
    disableRhel8DesktopIcon = disable_extension(RHEL8_DESKTOPICON_UUID);
}

function disable() {
    // Recover interrupts.
    ExtensionSystem.disconnect(extensionChangedId);
    if (Config.PACKAGE_VERSION.startsWith("3.28")) {
        global.screen.disconnect(workareaChangeId);
    } else if (Config.PACKAGE_VERSION.startsWith("3.32")) {
        global.display.disconnect(workareaChangeId);
    } else {
        global.log('Not Support ' + Config.PACKAGE_VERSION);
    }
    global.display.disconnect(focusWindowNotifyId);
    if(show_event) Main.overview.disconnect(show_event);
    if(hide_event) Main.overview.disconnect(hide_event);

    // Recover ubuntu's dash-to-dock and rhel's dock
    if (Main.sessionMode.allowExtensions) {
        if (disableUbuntuDock) {
            Mainloop.timeout_add(0, () => {
                global.log("Will enable extension [" + UBUNTU_DOCK_UUID +"]");
                ExtensionSystem.enableExtension(UBUNTU_DOCK_UUID);
            });
            disableUbuntuDock = false;
        }
        if (disableRhelDock) {
            Mainloop.timeout_add(0, () => {
                global.log("Will enable extension [" + RHEL_DOCK_UUID +"]");
                ExtensionSystem.enableExtension(RHEL_DOCK_UUID);
            });
            disableRhelDock = false;
        }
        if (disableRhel8DesktopIcon) {
            Mainloop.timeout_add(0, () => {
                global.log("Will enable extension [" + RHEL8_DESKTOPICON_UUID +"]");
                ExtensionSystem.enableExtension(RHEL8_DESKTOPICON_UUID);
            });
            disableRhel8DesktopIcon = false;
        }
    }

    // Recover panel.
    panel_box.set_position(0, panel_y);
    panel_actor.translation_y = 0;

    // Recover menu, dialog & IMS popup functions.
    SwitcherPopupExtension.disable();
    IBusPopupExtension.disable();
    DialogExtension.disable();
    MenuExtension.disable();

    // Recover background & screen saver.
    BackgroundExtension.disable();
    ScreenSaverExtension.disable();
}
