/*
 * ************************************************************************
 *
 * Copyright 2019-2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 *
 * ************************************************************************
 */
const Config = imports.misc.config;

const Clutter = imports.gi.Clutter;
const Dialog = imports.ui.dialog.Dialog;
const Main = imports.ui.main;
const ModalDialog = imports.ui.modalDialog.ModalDialog;
const St = imports.gi.St;
const Tweener = imports.ui.tweener;

const Me = imports.misc.extensionUtils.getCurrentExtension();
const ShadowWindowExtension = Me.imports.shadowWindowExtension;

const ModalDialogState = imports.ui.modalDialog.State;
var OPEN_AND_CLOSE_TIME = imports.ui.modalDialog.OPEN_AND_CLOSE_TIME;

let window = null;
let shadowWindow = {};
let focusId, minimizedId;

let old_ModalDialog_open = {};

let new_ModalDialog_open = function(timestamp, onPrimary) {
    if (this.state == ModalDialogState.OPENED ||
        this.state == ModalDialogState.OPENING)
        return true;

    if (!this.pushModal(timestamp))
        return false;

    this._fadeOpen(onPrimary);
    this.dialogLayout._dialog.add_style_class_name('right-angle-dialog');

    let signalId = this.connect('opened', () => {
        show_shadowwindow();
        this.disconnect(signalId);
    });

    return true;
};

let old_ModalDialog_close = {};

let new_ModalDialog_close = function(timestamp) {
    hide_shadowwindow();

    this.dialogLayout._dialog.remove_style_class_name('right-angle-dialog');

    if (this.state == ModalDialogState.CLOSED ||
        this.state == ModalDialogState.CLOSING)
        return;

    this.state = ModalDialogState.CLOSING;
    this.popModal(timestamp);
    this._savedKeyFocus = null;

    if (this._shouldFadeOut)
        Tweener.addTween(this._group,
                         { opacity: 0,
                           time: OPEN_AND_CLOSE_TIME,
                           transition: 'easeOutQuad',
                           onComplete: this._closeComplete.bind(this)
                         })
    else
        this._closeComplete();

};

let old_Dialog_addButton = {};

let new_Dialog_addButton = function(buttonInfo) {
    let { label, action, key } = buttonInfo;
    let isDefault = buttonInfo['default'];
    let keys;

    if (key)
        keys = [key];
    else if (isDefault)
        keys = [Clutter.KEY_Return, Clutter.KEY_KP_Enter, Clutter.KEY_ISO_Enter];
    else
        keys = [];

    let button = new St.Button({ style_class: 'modal-dialog-linked-button',
                                 button_mask: St.ButtonMask.ONE | St.ButtonMask.THREE,
                                 reactive:    true,
                                 can_focus:   true,
                                 x_expand:    true,
                                 y_expand:    true,
                                 label:       label });
    button.add_style_class_name('right-angle-dialog-button');
    button.connect('clicked', action);

    buttonInfo['button'] = button;

    if (isDefault)
        button.add_style_pseudo_class('default');

    if (this._initialKeyFocus == null || isDefault)
        this._setInitialKeyFocus(button);

    for (let i in keys)
        this._buttonKeys[keys[i]] = buttonInfo;

    this.buttonLayout.add_actor(button);

    return button;
};

function show_shadowwindow() {
    let screen, display;
    let monitorIndex;
    let monitorRect;

    if (Config.PACKAGE_VERSION.startsWith("3.28")) {
        screen = global.screen;
        monitorIndex = screen.get_current_monitor();
        monitorRect = screen.get_monitor_geometry(monitorIndex);
    } else if (Config.PACKAGE_VERSION.startsWith("3.32")) {
        display = global.display;
        monitorIndex = display.get_current_monitor();
        monitorRect = display.get_monitor_geometry(monitorIndex);
    } else {
        global.log('Not Support ' + Config.PACKAGE_VERSION);
        return;
    }

    let [dialogWidth, dialogHeight]
            = Main.layoutManager.modalDialogGroup.get_transformed_size();
    let [dialogX, dialogY]
            = Main.layoutManager.modalDialogGroup.get_transformed_position();

    dialogWidth = Math.floor(dialogWidth);
    dialogHeight = Math.floor(dialogHeight);
    dialogX
        = Math.floor((monitorRect.width - dialogWidth) / 2 + monitorRect.x);
    dialogY
        = Math.floor((monitorRect.height - dialogHeight) / 2 + monitorRect.y);

    shadowWindow.draw(dialogX, dialogY, dialogWidth, dialogHeight,
                      ShadowWindowExtension.winType.TOPLEVEL,
                      ShadowWindowExtension.winStyle.TRANSPARENT);
    window = global.display.get_focus_window();
    if (window) {
        focusId = window.connect('focus', on_dialog_win_focus);
        minimizedId = window.connect('notify::minimized',
                                     on_dialog_win_minimized);
    }
}

function hide_shadowwindow() {
    if (window) {
        window.disconnect(focusId);
        window.disconnect(minimizedId);
        window = null;
    }
    shadowWindow.hide();
}

function on_dialog_win_focus() {
    shadowWindow.redraw();
}

function on_dialog_win_minimized() {
    shadowWindow.hide();
}

function init() {
    shadowWindow = new ShadowWindowExtension.ShadowWindow();

    old_ModalDialog_open = ModalDialog.prototype.open;
    old_ModalDialog_close = ModalDialog.prototype.close;
    old_Dialog_addButton = Dialog.prototype.addButton;
}

function enable() {
    ModalDialog.prototype.open = new_ModalDialog_open;
    ModalDialog.prototype.close = new_ModalDialog_close;
    Dialog.prototype.addButton = new_Dialog_addButton;
}

function disable() {
    ModalDialog.prototype.open = old_ModalDialog_open;
    ModalDialog.prototype.close = old_ModalDialog_close;
    Dialog.prototype.addButton = old_Dialog_addButton;
}
