#!/bin/sh

###################################################
# Copyright 2015-2020 VMware, Inc.  All rights reserved.
###################################################


configure_distribution() {
   #
   # Common
   #
   configure_distribution_common
}


verify_environment() {
#
# Installation sources
#
   if [ ! -d $VIEWAGENT_PATH ]; then
      echo "Horizon Agent path not found"
      exit
   fi
   if [ ! -d $VMW_SSL_PATH ]; then
      echo "Horizon Agent SSL path not found"
      exit
   fi
}


uninstall_viewagent() {
   # Run uninstall scripts generated by install script
   if [ -d $VMW_UNINSTALL_SCRIPTS_DIR ]
   then
      if command -v run-parts > /dev/null 2>&1
      then
         run-parts $VMW_UNINSTALL_SCRIPTS_DIR
      else
         for f in $(ls $VMW_UNINSTALL_SCRIPTS_DIR)
         do
            test -x ${VMW_UNINSTALL_SCRIPTS_DIR}/$f && ${VMW_UNINSTALL_SCRIPTS_DIR}/$f
         done
      fi
      rm -rf $VMW_UNINSTALL_SCRIPTS_DIR
   fi

   if [ -f "${LOCALE_PROFILE_PATH}/vmware-lang.sh" ]; then
      rm -f ${LOCALE_PROFILE_PATH}/vmware-lang.sh
   fi

   if [ -d "/usr/lib/pcoip" ]; then
      rm -rf /usr/lib/pcoip
   fi
   rm -f /etc/xdg/autostart/$VIEWAGENT_DESKTOPC_SCRIPT
   rm -f /etc/xdg/plasma-workspace/env/desktopc.sh >/dev/null 2>&1

   # Keep the x264 software encoder library
   find $VIEWAGENT_PATH/* -type d ! -name VMwareBlastServer -exec rm -rf {} > /dev/null 2>&1 \;
   find $VIEWAGENT_PATH/* -type f ! -name libx264* -exec rm -rf {} > /dev/null 2>&1 \;

   # Clean the tmp file
   rm -rf $VIEWAGENT_TMP_PATH

   case "$SERVICE_MGR" in
   ${SERVICE_MGR_SYSV})
      chkconfig --del viewagent > /dev/null 2>&1
      if [ "$DISTRO_ID" = "$DISTRO_ID_SUSE" ]; then
         rm $SYSV_CONF_PATH_SUSE/$VIEWAGENT_SYSV
         # Remove symbolic link /sbin/rcviewagent if have
         if [ -L /sbin/rcviewagent ]; then
            rm -f /sbin/rcviewagent
         fi
      else
         rm $SYSV_CONF_PATH/$VIEWAGENT_SYSV
      fi

      # uninstall shutdown service
      chkconfig --del viewagent_shutdown > /dev/null 2>&1
      if [ "$DISTRO_ID" = "$DISTRO_ID_SUSE" ]; then
         rm $SYSV_CONF_PATH_SUSE/$SHUTDOWN_SYSV
         # Remove symbolic link /sbin/rcviewagent if have
         if [ -L /sbin/rcviewagent_shutdown ]; then
            rm -f /sbin/rcviewagent_shutdown
         fi
      else
         rm $SYSV_CONF_PATH/$SHUTDOWN_SYSV
      fi
      ;;
   ${SERVICE_MGR_SYSTEMD})
      systemctl disable viewagent > /dev/null 2>&1
      if [ "$DISTRO_ID" = "$DISTRO_ID_UBUNTU" ]; then
         rm $SYSTEMD_CONF_PATH_UBUNTU/$VIEWAGENT_SYSTEMD
      else
         rm $SYSTEMD_CONF_PATH/$VIEWAGENT_SYSTEMD
      fi

      # uninstall shutdown service
      if [ "$DISTRO_ID" = "$DISTRO_ID_UBUNTU" ]; then
         rm $SYSTEMD_CONF_PATH_UBUNTU/$SHUTDOWN_SYSTEMD
      else
         rm $SYSTEMD_CONF_PATH/$SHUTDOWN_SYSTEMD
      fi
      ;;
   ${SERVICE_MGR_UPSTART})
      rm $UPSTART_CONF_PATH/$VIEWAGENT_UPSTART
      # uninstall shutdown service
      rm $UPSTART_CONF_PATH/$SHUTDOWN_UPSTART
      ;;
   esac

   if [ -f "$VIEWAGENT_ENV_FILE" ]; then
      rm -fr "$VIEWAGENT_ENV_FILE"
   fi

   if [ -f "$VIEWAGENT_ENV_FILE_UBUNTU" ]; then
      rm -fr "$VIEWAGENT_ENV_FILE_UBUNTU"
   fi

   if [ -f "$SHUTDOWN_ENV_FILE" ]; then
      rm -fr "$SHUTDOWN_ENV_FILE"
   fi

   if [ -f "$SHUTDOWN_ENV_FILE_UBUNTU" ]; then
      rm -fr "$SHUTDOWN_ENV_FILE_UBUNTU"
   fi

   if [ -f "$LIGHTDM_PATH/lightdm.conf.orig" ]; then
      HEADLESS="yes"
      mv $LIGHTDM_PATH/lightdm.conf.orig $LIGHTDM_PATH/lightdm.conf
   fi

   # uninstall session collaboration
   if [ -d ${COLLABORATION_ICON_PATH} ]; then
      rm -rf ${COLLABORATION_ICON_PATH} > /dev/null 2>&1
   fi

   if [ -f /usr/share/pixmaps/${COLLABORATION_TRAY_ICON} ]; then
      rm -rf /usr/share/pixmaps/${COLLABORATION_TRAY_ICON} > /dev/null 2>&1
   fi

   if [ -f "$UDEV_RULES_PATH/99-vmw-usb-input.rules" ]; then
      rm -f $UDEV_RULES_PATH/99-vmw-usb-input.rules
   fi

   rm -rf "$IPC_PORT_FILE"
   rm -rf "$IPC_MAGIC_FILE"
}


uninstall_sc_sso() {
   local archpath="lib64"

   if [ -f "/usr/$archpath/gdm/simple-greeter/plugins/vmwsccred.so" ]; then
      rm -f /usr/$archpath/gdm/simple-greeter/plugins/vmwsccred.so
   fi

   if [ -d "/usr/share/gdm/simple-greeter/extensions/vmwsccred" ]; then
      rm -rf /usr/share/gdm/simple-greeter/extensions/vmwsccred
   fi

   if [ -f "/usr/share/icons/hicolor/16x16/apps/gdm-vmwsccred.png" ]; then
      rm -f /usr/share/icons/hicolor/16x16/apps/gdm-vmwsccred.png
   fi

   if [ -f "/usr/share/icons/hicolor/48x48/apps/gdm-vmwsccred.png" ]; then
      rm -f /usr/share/icons/hicolor/48x48/apps/gdm-vmwsccred.png
   fi

   if [ -f "/$archpath/security/pam_vmw_cred.so" ]; then
      rm -f /$archpath/security/pam_vmw_cred.so
   fi

   if [ -f "/etc/pam.d/gdm-vmwsccred" ]; then
      rm -f /etc/pam.d/gdm-vmwsccred
   fi

   if [ -f "$DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf" ]; then
      rm -f $DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf
   fi
}


uninstall_sso_rhel() {
   local archpath="lib64"
   local majorVersion=
   majorVersion=`identify_distribution -r | head -c 1`

   if [ "$majorVersion" = "7" -o "$majorVersion" = "8" ]; then
      # Restore the original gnome-shell
      if [ -f "${SSO_BACKUP_PATH}/gnome-shell-3.26.2-5.el7.x86_64.orig.rpm" ]; then
         rpm -U --force ${SSO_BACKUP_PATH}/gnome-shell-3.26.2-5.el7.x86_64.orig.rpm > /dev/null 2>&1
      elif [ -f "${SSO_BACKUP_PATH}/gnome-shell-3.22.3-17.el7.x86_64.orig.rpm" ]; then
         rpm -U --force ${SSO_BACKUP_PATH}/gnome-shell-3.22.3-17.el7.x86_64.orig.rpm > /dev/null 2>&1
      elif [ -f "${SSO_BACKUP_PATH}/gnome-shell-3.14.4-53.el7.x86_64.orig.rpm" ]; then
         rpm -U --force ${SSO_BACKUP_PATH}/gnome-shell-3.14.4-53.el7.x86_64.orig.rpm > /dev/null 2>&1
      elif [ -f "${SSO_BACKUP_PATH}/gnome-shell-3.14.4-37.el7.x86_64.orig.rpm" ]; then
         rpm -U --force ${SSO_BACKUP_PATH}/gnome-shell-3.14.4-37.el7.x86_64.orig.rpm > /dev/null 2>&1
      elif [ -f "${SSO_BACKUP_PATH}/gnome-shell-3.28.3-6.el7.x86_64.orig.rpm" ]; then
         rpm -U --force ${SSO_BACKUP_PATH}/gnome-shell-3.28.3-6.el7.x86_64.orig.rpm > /dev/null 2>&1
      elif [ -f "${SSO_BACKUP_PATH}/gnome-shell-3.28.3-10.el8.x86_64.orig.rpm" ]; then
         rpm -U --force ${SSO_BACKUP_PATH}/gnome-shell-3.28.3-10.el8.x86_64.orig.rpm > /dev/null 2>&1
      elif [ -f "${SSO_BACKUP_PATH}/gnome-shell-3.32.2-9.el8.x86_64.orig.rpm" ]; then
         rpm -U --force ${SSO_BACKUP_PATH}/gnome-shell-3.32.2-9.el8.x86_64.orig.rpm > /dev/null 2>&1
      else
         if [ -f "${SSO_BACKUP_PATH}/authPrompt.js.orig" ]; then
            cp -p ${SSO_BACKUP_PATH}/authPrompt.js.orig /usr/share/gnome-shell/js/gdm/authPrompt.js
         fi
         if [ -f "${SSO_BACKUP_PATH}/util.js.orig" ]; then
            cp -p ${SSO_BACKUP_PATH}/util.js.orig /usr/share/gnome-shell/js/gdm/util.js
         fi
         if [ -f "/usr/share/gnome-shell/js/gdm/vmCred.js" ]; then
            rm -fr /usr/share/gnome-shell/js/gdm/vmCred.js
         fi
      fi

      if [ -f "/$archpath/security/pam_vmw_cred.so" ]; then
         rm -f /$archpath/security/pam_vmw_cred.so
      fi

      if [ -f "/etc/pam.d/gdm-vmwcred" ]; then
         rm -f /etc/pam.d/gdm-vmwcred
      fi

      if [ -f "$DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf" ]; then
         rm -f $DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf
      fi

      # Remove the config file of vmwpkcs11 module for TrueSSO.
      if [ -f "/usr/share/p11-kit/modules/libvmwpkcs11.module" ]; then
         rm -f /usr/share/p11-kit/modules/libvmwpkcs11.module
      fi

      if [ -f "/usr/share/gdm/greeter/autostart/vmware-greeter-check.desktop" ]; then
         rm -f /usr/share/gdm/greeter/autostart/vmware-greeter-check.desktop
      fi
   fi

   if [ "$majorVersion" = "6" ]; then
      if [ -f "/usr/$archpath/gdm/simple-greeter/plugins/vmwcred.so" ]; then
         rm -f /usr/$archpath/gdm/simple-greeter/plugins/vmwcred.so
      fi

      if [ -d "/usr/share/gdm/simple-greeter/extensions/vmwcred" ]; then
         rm -rf /usr/share/gdm/simple-greeter/extensions/vmwcred
      fi

      if [ -f "/usr/share/icons/hicolor/16x16/apps/gdm-vmwcred.png" ]; then
         rm -f /usr/share/icons/hicolor/16x16/apps/gdm-vmwcred.png
      fi

      if [ -f "/usr/share/icons/hicolor/48x48/apps/gdm-vmwcred.png" ]; then
         rm -f /usr/share/icons/hicolor/48x48/apps/gdm-vmwcred.png
      fi

      if [ -f "/$archpath/security/pam_vmw_cred.so" ]; then
         rm -f /$archpath/security/pam_vmw_cred.so
      fi

      if [ -f "/etc/pam.d/gdm-vmwcred" ]; then
         rm -f /etc/pam.d/gdm-vmwcred
      fi

      if [ -f "$DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf" ]; then
         rm -f $DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf
      fi
   fi
}


uninstall_sso_suse() {
   local archpath="lib64"
   if [ "$DISTRO_RELEASE" = "11" ]; then
      if [ -f "${SSO_BACKUP_PATH}/gdm-simple-greeter.orig" ]; then
         gdm_simple_greeter_pid="`pidof gdm-simple-greeter`"
         if [ "$?" == "0" ]; then
            gdm-stop
            gdm_simple_greeter_status=1
            while [ -n "`pidof gdm`" ]
            do
               sleep 1
            done
         fi
         # Restore the original gdm-simple-greeter
         cp -p "${SSO_BACKUP_PATH}/gdm-simple-greeter.orig" "/usr/lib/gdm/gdm-simple-greeter"
         if [ "${gdm_simple_greeter_status}" = "1" ]; then
            /usr/bin/gdm &
         fi
      fi

      if [ -f "${SSO_BACKUP_PATH}/gdm.orig" ]; then
         # Restore the original gdm PAM config file
         cp -p "${SSO_BACKUP_PATH}/gdm.orig" "/etc/pam.d/gdm"
      fi

      if [ -f "/$archpath/security/pam_vmw_cred.so" ]; then
         rm -f /$archpath/security/pam_vmw_cred.so
      fi

      if [ -f "$DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf" ]; then
         rm -f $DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf
      fi
   fi

   if [ "$OS_MAJOR" = "12" ]; then
      if [ "$DISTRO_RELEASE" = "12.3" -o "$DISTRO_RELEASE" = "12.5" ]; then
         [ -f "${SSO_BACKUP_PATH}/gnome-shell-3.20.4-76.3.x86_64.orig.rpm" ] && \
         rpm -U --force ${SSO_BACKUP_PATH}/gnome-shell-3.20.4-76.3.x86_64.orig.rpm
      elif [ "$DISTRO_RELEASE" = "12.2" ]; then
         [ -f "${SSO_BACKUP_PATH}/gnome-shell-3.20.4-70.4.x86_64.orig.rpm" ] && \
         rpm -U --force ${SSO_BACKUP_PATH}/gnome-shell-3.20.4-70.4.x86_64.orig.rpm
      elif [ "$DISTRO_RELEASE" = "12.1" ]; then
         [ -f "${SSO_BACKUP_PATH}/gnome-shell-3.10.4-40.1.x86_64.orig.rpm" ] && \
         rpm -U --force ${SSO_BACKUP_PATH}/gnome-shell-3.10.4-40.1.x86_64.orig.rpm
      fi

      if [ -f "/etc/pam.d/gdm-vmwcred" ]; then
         rm -f "/etc/pam.d/gdm-vmwcred"
      fi

      if [ -f "/$archpath/security/pam_vmw_cred.so" ]; then
         rm -f /$archpath/security/pam_vmw_cred.so
      fi

      if [ -f "$DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf" ]; then
         rm -f $DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf
      fi

   fi
}


uninstall_sso_ubuntu() {
   local unity_greeter_status=0
   local count=0
   local ret=0
   if [ $OS_MAJOR = 14 ] || [ $OS_MAJOR = 16 ] || [ $OS_MAJOR = 18 ]; then
      if [ -f "${SSO_BACKUP_PATH}/unity-greeter.orig" ]; then
         unity_greeter_pid="`pidof unity-greeter`"
         if [ "$?" = "0" ]; then
            service lightdm stop
            unity_greeter_status=1
            while [ -n "`pidof lightdm`" ]
            do
               sleep 1
            done
         fi
         # Restore the original unity-greeter
         cp -p "${SSO_BACKUP_PATH}/unity-greeter.orig" "/usr/sbin/unity-greeter" > /dev/null 2>&1
         ret=$?
         #
         # Sometimes "Text file busy" leads to unity-greeter uninstallation failed, maybe
         # unity-greeter doesn't finish the stop process, need to retry.
         #
         while [ "$ret" != "0" -a "$count" -lt "5" ]
         do
            sleep 1
            echo_log "Uninstaller is waiting unity-greeter to stop"
            cp -p "${SSO_BACKUP_PATH}/unity-greeter.orig" "/usr/sbin/unity-greeter" > /dev/null 2>&1
            ret=$?
            count=`expr $count + 1`
         done

         if [ "$count" -ge "5" ]; then
            echo_term "Failed to uninstall SSO supported unity-greeter, exit"
            exit
         fi
      fi

      if [ -f "${SSO_BACKUP_PATH}/libgnome-shell.orig.so" ]; then
         install ${SSO_BACKUP_PATH}/libgnome-shell.orig.so /usr/lib/gnome-shell/libgnome-shell.so >/dev/null 2>&1
      fi

      if [ -f "/etc/pam.d/gdm-vmwcred" ]; then
         rm -f /etc/pam.d/gdm-vmwcred
      fi

      if [ -f "/lib/x86_64-linux-gnu/security/pam_vmw_cred.so" ]; then
         rm -f /lib/x86_64-linux-gnu/security/pam_vmw_cred.so
      fi

      if [ -f "/lib/x86_64-linux-gnu/security/pam_vmw_null.so" ]; then
         rm -f /lib/x86_64-linux-gnu/security/pam_vmw_null.so
      fi

      if [ -f "/usr/share/pam-configs/vmw-mkhomedir" ]; then
         rm -f /usr/share/pam-configs/vmw-mkhomedir

         if [ -f "/usr/share/pam-configs/vmw-null" ]; then
            rm -f /usr/share/pam-configs/vmw-null
         fi

         if [ -f "${SSO_BACKUP_PATH}/unix" ]; then
            mv ${SSO_BACKUP_PATH}/unix /usr/share/pam-configs >/dev/null 2>&1
         fi

         pam-auth-update --package --force >/dev/null 2>&1
      fi

      if [ -f "$DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf" ]; then
         rm -f $DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf
      fi

      # If install view via ssh, start lightdm process after replace unity-greeter
      if [ "${unity_greeter_status}" = "1" ]; then
         service lightdm start
      fi
   fi
}


uninstall_sso() {
   local arch="`uname -i`"

   [ "$arch" != "x86_64" ] && return

   if [ ! -f "$VIEWAGENT_PATH/bin/SsoDBusNotify.py" ]; then
      return
   fi

   umask 022

   case "$DISTRO_ID" in
   ${DISTRO_ID_CENTOS}|\
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_RHEL_WORKSTATION})
      uninstall_sso_rhel
      ;;
   ${DISTRO_ID_SUSE})
      uninstall_sso_suse
      ;;
   ${DISTRO_ID_UBUNTU})
      uninstall_sso_ubuntu
      ;;
   *)
      return
   esac
}

uninstall_scredir_ubuntu() {
   if [ ${OS_MAJOR} = "16" ]; then
      sed --in-place -e '/pam_vmw_setenv.so/d' /etc/pam.d/unity
   elif [ ${OS_MAJOR} = "18" ]; then
      sed --in-place -e '/pam_vmw_setenv.so/d' /etc/pam.d/gdm-password
   fi

   if [ -f "/lib/x86_64-linux-gnu/security/pam_vmw_setenv.so" ]; then
      rm -f /lib/x86_64-linux-gnu/security/pam_vmw_setenv.so
   fi
}

uninstall_scredir_redhat() {
   local archpath="/lib64/"

   if [ ${OS_MAJOR} = "7" ]; then
       archpath="/usr"${archpath}
   fi

   if [ ${OS_MAJOR} = "6" ]; then
      sed --in-place -e '/pam_vmw_setenv.so/d' /etc/pam.d/gdm-smartcard
      sed --in-place -e '/pam_vmw_setenv.so/d' /etc/pam.d/gnome-screensaver
   elif [ ${OS_MAJOR} = "7" ]; then
      sed --in-place -e '/pam_vmw_setenv.so/d' /etc/pam.d/smartcard-auth
   fi

   rm -f ${archpath}/security/pam_vmw_setenv.so
}

uninstall_scredir_suse() {
   archpath="/lib64/"

   sed --in-place -e '/pam_vmw_setenv.so/d' /etc/pam.d/gdm-password

   rm -f ${archpath}/security/pam_vmw_setenv.so
}

uninstall_scredir() {
   local ret=0;
   local count=0;

   #
   # Check if smartcard redirection package is installed
   #
   which pcscd > /dev/null 2>&1
   if [ "$?" = "1" ] ||
      [ "`pcscd -v | grep vm`" = "" ]; then
      return
   fi

   if [ ! -d ${SCREDIR_BACKUP_DIR} ]; then
      return
   fi

   #
   # Restore the pcsclite library
   #
   service pcscd stop > /dev/null 2>&1
   install ${SCREDIR_BACKUP_DIR}/pcscd /usr/sbin/
   if [ "$DISTRO_ID" = ${DISTRO_ID_UBUNTU} ]; then
      uninstall_scredir_ubuntu
   elif [ "$DISTRO_ID" = ${DISTRO_ID_RHEL_WORKSTATION} ]; then
      uninstall_scredir_redhat
   elif [ "$DISTRO_ID" = "${DISTRO_ID_SUSE}" ]; then
      uninstall_scredir_suse
   fi
   rm -rf ${SCREDIR_BACKUP_DIR}

   #
   # Delete the plugin and IPC directory
   #
   rm -rf ${SCREDIR_VCHAN_PLUGINS_PATH}
   rm -rf ${SCREDIR_IPC_DIR}

   #
   # Start the pcscd service
   #
   service pcscd start > /dev/null 2>&1

   #
   # Uninstall Smartcard SSO package
   #
   uninstall_sc_sso
}

uninstall_audioin() {
   #
   # Delete the audioin plugin directory
   #
   if [ -d ${AUDIOIN_VCHAN_PLUGINS_PATH} ]; then
      rm -rf ${AUDIOIN_VCHAN_PLUGINS_PATH}
   fi
}

uninstall_rdeserver() {
   rm -rf $APP_THEME_DIR/VMWLight
   rm -rf $APP_ICON_DIR
   rm -rf $APP_XML_FILE

   rm -rf ${GNOME_SHELL_EXTENSIONS_DIR}/vmware-remote-app-helper*
   rm -rf /usr/bin/gnome-shell-extension-tool-vmw
   rm -f /usr/bin/shadowWindowExtension.js
}

stop_service() {
   serviceStop viewagent
   serviceStop viewagent_shutdown
}


enable_local_vm_runstate_change() {
   case $DISTRO_ID in
   ${DISTRO_ID_CENTOS}|\
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_SUSE}|\
   ${DISTRO_ID_RHEL_WORKSTATION})
      rm -f $POLICYKIT_PATH/$VIEWAGENT_POLICYKIT

      while read file
      do
         if [ -f $PAM_PATH/$file.vmw ]; then
            mv $PAM_PATH/$file.vmw $PAM_PATH/$file
            if [ "$?" != "0" ]; then
               echo "Failed to restore $PAM_PATH/$file"
            fi
         fi
      done << !!!
halt
poweroff
reboot
!!!
      ;;
   ${DISTRO_ID_UBUNTU}|\
   ${DISTRO_ID_KYLIN})
      ;;
   esac
}

restore_xorg_VTSwitch()
{
    xorg_conf="/etc/X11/xorg.conf"

    if [ -f $VIEWAGENT_PATH/xorg.conf.orig ]; then
        mv -f  $VIEWAGENT_PATH/xorg.conf.orig $xorg_conf
        return
    fi

    if [ -f $xorg_conf ]; then
        rm -f $xorg_conf
    fi
}

unconfigure_firewall_iptables() {
   local ret=0

   blast_client_rule="`iptables -nL INPUT --line-numbers | grep ${BLAST_PORT}`"

   #
   # Loop to clear all the rules for specfic port
   #
   while [ "$?" = "0" ]; do
      line_number="`echo $blast_client_rule | awk '{ print $1 }'`"
      iptables -D INPUT ${line_number}
      ret=$?
      [ "$ret" = "0" ] && serviceSave iptables

      blast_client_rule="`iptables -nL INPUT --line-numbers | grep ${BLAST_PORT}`"
   done

   return $ret
}


unconfigure_firewall_firewalld() {
   local ret=0
   local protocol="tcp"
   local port=${BLAST_PORT}

   if [ "$#" != "2" ]; then
      echo_term "configure_firewall_firewalld: No command specified."
      return 1
   fi

   protocol="$1"
   port="$2"

   firewall-cmd --remove-port=${port}/${protocol} >/dev/null 2>&1 || ret=1
   [ "$ret" = "1" ] || firewall-cmd --remove-port=${port}/${protocol} \
      --permanent >/dev/null 2>&1 || ret=1

   return $ret
}

unconfigure_firewall_suse() {
   local ret=1
   SuSEfirewall="/etc/sysconfig/SuSEfirewall2"

   if [ -f $VIEWAGENT_PATH/SuSEfirewall2.orig ]; then
      mv -f  $VIEWAGENT_PATH/SuSEfirewall2.orig $SuSEfirewall
      # Restart the service if the original status of firewall is ON
      rcSuSEfirewall2 status | grep "unused" || rcSuSEfirewall2 status | grep "inactive" >/dev/null 2>&1
      if [ "$?" != "0" ]; then
         rcSuSEfirewall2 restart >/dev/null 2>&1
      fi
      ret=0
   fi

   return $ret
}

unconfigure_firewall() {
   local ret=0

   if [ "$DISTRO_ID" = "$DISTRO_ID_SUSE" ]; then
      unconfigure_firewall_suse
      ret=$?
   else
      case "${FIREWALL}" in
      ${FIREWALL_IPTABLES})
         unconfigure_firewall_iptables
         ret=$?
         ;;
      ${FIREWALL_FIREWALLD})
         unconfigure_firewall_firewalld tcp ${BLAST_PORT}
         unconfigure_firewall_firewalld udp ${BLAST_PORT}
         ret=$?
         ;;
      esac
   fi

   if [ "$ret" = "1" ]; then
      echo "Unable to automatically unconfigure firewall."
      echo "Please unconfigure inbound connections on port ${BLAST_PORT}/tcp."
   fi
}


unconfigure_selinux() {

   case "$DISTRO_ID" in
   ${DISTRO_ID_CENTOS}|\
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_RHEL_WORKSTATION})
      which semodule >/dev/null 2>&1
      if [ "$?" != "0" ]; then
         echo_term "The required command \"semodule\" is currently not installed, configure selinux failed. "
         return
      fi

      if [ "$SSO_ENABLE" = "yes" ]; then
         semodule -r vmwsso >/dev/null 2>&1
         # If failed, try to remove again.
         # So that we can print the error info
         if [ "$?" != "0" ]; then
            semodule -r vmwsso
         fi
      fi

      ;;
   ${DISTRO_ID_UBUNTU})
      ;;
   *)
      ;;
   esac
}

unconfigure_ipc() {
   # Restore the sysctl configure.
   install ${SYSCTL_BACKUP_DIR}/sysctl.conf.backup /etc/sysctl.conf
   rm -rf ${SYSCTL_BACKUP_DIR}

   rm -rf "$IPC_PORT_FILE"
   rm -rf "$IPC_MAGIC_FILE"
}

unconfigure_pam() {
   if [ -f "${VIEWAGENT_BACKUP_PATH}/lightdm" ]; then
      # Restore the original lightdm PAM config file
      install ${VIEWAGENT_BACKUP_PATH}/lightdm /etc/pam.d/
   fi

   if [ -f "${VIEWAGENT_BACKUP_PATH}/gdm-password" ]; then
      # Restore the original gdm-password config file
      install ${VIEWAGENT_BACKUP_PATH}/gdm-password /etc/pam.d/
   fi

}

#
# main
#

#
# Ensure script is executed as root.
#
if [ "`id -u`" != "0" ]; then
   echo "Executing horizon agent uninstaller as superuser"
   exec sudo $0 $@

   #
   # The following should never be executed...
   #
   echo "uninstall_viewagent.sh not run as superuser"
   exit
fi

#
# Only execute the corresponding installed uninstaller script
#
script_dir="`dirname $0`"
if [ "$script_dir" != "__INSTALL_LOC__" ]; then
   echo "Please execute __INSTALL_LOC__/uninstall_viewagent.sh"
   exit
fi

#
# Include common constants and routines
#
. "${script_dir}/commonlib.sh"

# check if there is VMwareBlastServer running
pidof VMwareBlastServer >/dev/null 2>&1
if [ "$?" = "0" ]; then
   # Prompt user to logout current session, then run uninstall script
   echo "Agent uninstallation cannot continue due to an active session has been detected, you must restart or logout your system at first."
   exit
fi

identify_distribution
configure_distribution

#
# Get pre-installation's managed and singleton flag
#
SINGLETON_MODE=$(get_singleton_var)
preManagedFlag="false"
if [ -f "$AGENT_CFG_FILE" ]; then
   managedName="MANAGED="
   line=`grep "^$managedName" $AGENT_CFG_FILE`
   if [ "$?" = 0 ]; then
      preManagedFlag="`echo "$line" | awk 'BEGIN{FS="="}{print $2}'`"
   fi
fi

configure_xorg_conf $(get_singleton_var) reset
verify_environment
stop_service
echo_term "Uninstallation start..."
"${script_dir}/XdmcpSecurity.sh" uninstall
unconfigure_pam
unconfigure_firewall
uninstall_sso
unconfigure_selinux
uninstall_scredir
uninstall_audioin
uninstall_rdeserver
unconfigure_ipc

customise_multiple_session reset
customise_desktop reset $preManagedFlag
uninstall_viewagent
if [ "${HEADLESS}x" != "x" ]; then
   initctl restart lightdm
fi
enable_local_vm_runstate_change
if [ "$preManagedFlag" = "true" ] &&
   [ -f "$MACHINE_CFG_FILE" ] ; then
   # delete the machine config
   rm -f "$MACHINE_CFG_FILE"
fi
echo_term "Uninstallation done"

