#!/bin/bash

###################################################
# Copyright 2017-2020 VMware, Inc.  All rights reserved.
###################################################


ngvcAgentPrefix="`date` - [NGVC-Agent-Script]"
logPath="/var/log/vmware/viewagent-ngvc.log"
customizationState="success"
customizationScriptFolder="/var/userScript"


CUSTOMIZATION_FLAG_NONE=0x0000000
CUSTOMIZATION_FLAG_SHUTDOWN_REQUIRED=0x0000001
CUSTOMIZATION_FLAG_SHUTDOWN_REQUESTED=0x0000002
CUSTOMIZATION_FLAG_SHUTDOWN_COMPLETED=0x0000004
CUSTOMIZATION_FLAG_REBOOT_REQUIRED=0x0000008
CUSTOMIZATION_FLAG_REBOOT_REQUESTED=0x0000010
CUSTOMIZATION_FLAG_REBOOT_COMPLETED=0x0000020
CUSTOMIZATION_FLAG_RESYNC=0x0000040

checkMachinePassword() {
   local checksum
   local passwdsum

   if [ -z "$MachinePassword" ]; then
      echo "${ngvcAgentPrefix}: checkMachinePassword: Machine password not specified." >> $logPath
      return 1
   fi

   checksum=`vmware-rpctool "info-get guestinfo.machinePasswdChecksum"`
   if [ "$?" != "0" ]; then
      echo "${ngvcAgentPrefix}: checkMachinePassword: Failed to fetch checksum."  >> $logPath
      return 0
   fi

   if [ "$checksum" = "none" ]; then
      echo "${ngvcAgentPrefix}: checkMachinePassword: checksum is none, ignore machine password checksum."  >> $logPath
      return 0
   fi

   passwdsum=`echo -n $MachinePassword | iconv -f UTF-8 -t UTF-16LE | openssl dgst -sha256 -binary | base64`
   if [ "$?" != "0" ]; then
      echo "${ngvcAgentPrefix}: checkMachinePassword: Failed to calulate summary."  >> $logPath
      return 1
   fi

   if [ "$checksum" != "$passwdsum" ]; then
      echo "${ngvcAgentPrefix}: checkMachinePassword: The checksum doesn't equal to password summary."  >> $logPath
      return 1
   fi

   return 0
}

acquireMachinePassword() {
   local obfuscated
   local hostname
   local uuid
   local digest
   local keyvalue
   local ivvalue

   obfuscated=`vmware-rpctool "info-get guestinfo.machinePasswd"`
   if [ "$?" != "0" ] || [ "$obfuscated" = "none" ]; then
      echo "${ngvcAgentPrefix}: acquireMachinePassword: Failed to fetch obfuscated password."  >> $logPath
      return 1
   fi

   hostname=`vmware-rpctool "info-get guestinfo.hostName"`
   if [ "$?" != "0" ]; then
      echo "${ngvcAgentPrefix}: acquireMachinePassword: Failed to fetch hostname."  >> $logPath
      return 1
   fi

   uuid=`vmware-rpctool "info-get guestinfo.cloneprep.clone.uuid"`
   if [ "$?" != "0" ]; then
      echo "${ngvcAgentPrefix}: acquireMachinePassword: Failed to fetch cloneprep uuid."  >> $logPath
      return 1
   fi

   digest=`echo -n "$hostname$uuid" | openssl dgst -sha256 -r`
   if [ "$?" != "0" ] || [ ${#digest} -lt 64 ]; then
      digest=`echo -n "$hostname$uuid" | openssl dgst -sha256`
      if [ "$?" != "0" ] || [ ${#digest} -lt 64 ]; then
         echo "${ngvcAgentPrefix}: acquireMachinePassword: Failed to calulate key digest."  >> $logPath
         return 1
      fi
   fi

   keyvalue=${digest:0:32}
   ivvalue=${digest:32:32}

   MachinePassword=`echo -n $obfuscated |base64 -d | openssl enc -d -aes-128-cfb -K $keyvalue -iv $ivvalue`
   if [ "$?" != "0" ]; then
      echo "${ngvcAgentPrefix}: acquireMachinePassword: Failed to unobfuscate machine password."  >> $logPath
      return 1
   fi

   checkMachinePassword
   return $?
}

offlineDomainJoin4Samba() {
   local fqdn
   local upFqdn
   local domainName

   dbFile="/var/lib/samba/private/secrets.tdb"
   if [ ! -f "$dbFile" ]
   then
      echo "${ngvcAgentPrefix}: $file found. Please check whether already joined domain with samba method."  >> $logPath
      return 1
   fi

   which tdbtool
   if [ "$?" != "0" ]
   then
      echo "${ngvcAgentPrefix}: tdbtool not installed, instance clone will not support samba domain joined."  >> $logPath
      return 1
   fi

   # read the netbios domain name from config file
   netbiosDomainName=`awk '/^NetbiosDomain/{print}' /etc/vmware/viewagent-custom.conf | cut -f2 -d"="`
   if [ -z "${netbiosDomainName}" ]; then

      fqdn=`vmware-rpctool "info-get guestinfo.primed.fqdn"`
      typeset -u upFqdn
      if [ "$?" = "0" ]; then
          upFqdn="$fqdn"
      else
          upFqdn=$(echo -n "$fqdn" | tr '[a-z]' '[A-Z]')
      fi
      domainName=${upFqdn%.*}
   else
      domainName=${netbiosDomainName}
   fi

   tdbtool ${dbFile} store SECRETS/MACHINE_PASSWORD/${domainName} $MachinePassword

   if [ "$DISTRO_ID" = ${DISTRO_ID_UBUNTU} ] && [ ${OS_MAJOR} -ge "18" ]; then
      tdbtool ${dbFile} delete SECRETS/MACHINE_DOMAIN_INFO/${domainName}
   fi

   return $?
}

offlineDomainJoin4PBIS() {
   local fqdn
   local hostName
   local upFqdn
   local lowHostFqdn
   local samAccountName

   REG_TOOL=/opt/pbis/bin/regshell
   if [ -f $REG_TOOL ]; then
      if [ "$?" != "0" ]; then
         echo "${ngvcAgentPrefix}: offlineDomainJoin: Failed to acquire machine password."  >> $logPath
         return 1
      fi

      hostName=`vmware-rpctool "info-get guestinfo.hostName"`
      if [ "$?" != "0" ]; then
         echo "${ngvcAgentPrefix}: offlineDomainJoin: Failed to acquire hostname."  >> $logPath
         return 1
      fi

      fqdn=`vmware-rpctool "info-get guestinfo.primed.fqdn"`
      if [ "$?" != "0" ]; then
         echo "${ngvcAgentPrefix}: offlineDomainJoin: Failed to acquire fqdn."  >> $logPath
         return 1
      fi

      typeset -u samAccountName
      if [ "$?" = "0" ]; then
         samAccountName="$hostName\$"
      else
         samAccountName=$(echo -n "$hostName\$" | tr '[a-z]' '[A-Z]')
      fi
      typeset -u upFqdn
      if [ "$?" = "0" ]; then
         upFqdn="$fqdn"
      else
         upFqdn=$(echo -n "$fqdn" | tr '[a-z]' '[A-Z]')
      fi
      typeset -l lowHostFqdn
      if [ "$?" = "0" ]; then
         lowHostFqdn="$hostName.$fqdn"
      else
         lowHostFqdn=$(echo -n "$hostName.$fqdn" | tr '[A-Z]' '[a-z]')
      fi

      $REG_TOOL list_keys [HKEY_THIS_MACHINE\\Services\\lsass\\Parameters\\Providers\\ActiveDirectory\\DomainJoin\\$upFqdn\\Pstore]
      if [ "$?" != "0" ]; then
         echo "${ngvcAgentPrefix}: offlineDomainJoin: Failed to list keys in PBIS. Please join master VM in domain."  >> $logPath
         return 1
      fi

      $REG_TOOL set_value [HKEY_THIS_MACHINE\\Services\\lsass\\Parameters\\Providers\\ActiveDirectory\\DomainJoin\\$upFqdn\\Pstore] "Fqdn" "$lowHostFqdn"
      if [ "$?" != "0" ]; then
         echo "${ngvcAgentPrefix}: offlineDomainJoin: Failed to set Fqdn in PBIS."  >> $logPath
         return 1
      fi

      $REG_TOOL set_value [HKEY_THIS_MACHINE\\Services\\lsass\\Parameters\\Providers\\ActiveDirectory\\DomainJoin\\$upFqdn\\Pstore] "SamAccountName" "$samAccountName"
      if [ "$?" != "0" ]; then
         echo "${ngvcAgentPrefix}: offlineDomainJoin: Failed to set SamAccountName in PBIS."  >> $logPath
         return 1
      fi

      $REG_TOOL set_value [HKEY_THIS_MACHINE\\Services\\lsass\\Parameters\\Providers\\ActiveDirectory\\DomainJoin\\$upFqdn\\Pstore\\PasswordInfo] "Password" "$MachinePassword"
      if [ "$?" != "0" ]; then
         echo "${ngvcAgentPrefix}: offlineDomainJoin: Failed to set machine password in PBIS."  >> $logPath
         return 1
      fi
   else
      echo "${ngvcAgentPrefix}: offlineDomainJoin: PBIS doesn't install, please install PBIS to join domain in master VM."  >> $logPath
      return 1
   fi

   return 0
}

changeHostName() {
   echo "${ngvcAgentPrefix}: Beginning changing host name..." >> $logPath
   oldHostname=`hostname`
   hostName=`vmware-rpctool "info-get guestinfo.hostName"`

   echo "${ngvcAgentPrefix}: current hostname = $oldHostname." >> $logPath
   if [ "$oldHostname" = "$hostName" ]; then
      return 0
   fi

   case "$DISTRO_ID" in
   ${DISTRO_ID_UBUNTU})
      hostname $hostName
      bash -c "echo $hostName > /etc/hostname"
      sed -i "s/$oldHostname/$hostName/g" /etc/hosts
      ;;
   ${DISTRO_ID_SUSE})
      hostname $hostName
      bash -c "echo $hostName > /etc/HOSTNAME"
      sed -i "s/$oldHostname/$hostName/g" /etc/hosts
      ;;
   ${DISTRO_ID_CENTOS}|\
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_RHEL_WORKSTATION})
      if [ ${OS_MAJOR} -ge "7" ]; then
         hostname $hostName
         bash -c "echo $hostName > /etc/hostname"
         sed -i "s/$oldHostname/$hostName/g" /etc/hosts
      else
         # redhat 6
         sed -i "s/$oldHostname/$hostName/g" /etc/sysconfig/network
         sed -i "s/$oldHostname/$hostName/g" /etc/hosts
      fi
      ;;
   esac
}

runPostCustomizationScript() {
   # run post-customize script
   postCustomize=`vmware-rpctool "info-get guestinfo.scripts.PostCustomization"`
   if [ -n "$postCustomize" ]; then
      echo "${ngvcAgentPrefix}: PostCustomization: $postCustomize.." >> $logPath
      scriptPath=`echo $postCustomize | awk '{print $1}' | sed 's/\"//g'`

      # script folder should be under /var/userScript which has root:root privilege
      scriptfolder=$(dirname "${scriptPath}")
      if [ "$customizationScriptFolder" = "$scriptfolder" ]; then
         if [ -f "$scriptPath" ]; then
             scriptWithPara=`echo $postCustomize | sed 's/\"//g'`
             owner=`stat -c "%U" "$scriptPath"`
             permission=`stat -c "%a" "$scriptPath"`
             if [ "$owner" = "root" ] && [ "$permission" = "700" ]; then
                echo "${ngvcAgentPrefix}: Beginning Post Customization script command: ${scriptWithPara}" >> $logPath
                $scriptWithPara
                echo "${ngvcAgentPrefix}: Ending Post Customization script, return value is: $?" >> $logPath
             else
                echo "${ngvcAgentPrefix}: Post Customization script owner is not root or the permission is not 700" >> $logPath
             fi
         else
             echo "${ngvcAgentPrefix}: Post Customization script does not exist.." >> $logPath
         fi
      else
         echo "${ngvcAgentPrefix}: Customization script folder($scriptfolder) is not under /var/userScript.." >> $logPath
      fi
   else
      echo "${ngvcAgentPrefix}: PostCustomization does not config.." >> $logPath
   fi
}

setCustomizationStatus() {
   state=$1
   # set customization state to success in clone stage
   vmware-rpctool "info-set guestinfo.clone.CustomizationState $state"
   sed -i '/^'CUSTOMIZATION_STATE'/d' /etc/vmware/viewagent-config.txt  >> $logPath 2>&1
   if [ "$state" = "success" ]; then
      echo "CUSTOMIZATION_STATE=true" >> /etc/vmware/viewagent-config.txt
      echo "${ngvcAgentPrefix}: Customization Done..." >> $logPath
   else
      echo "CUSTOMIZATION_STATE=false" >> /etc/vmware/viewagent-config.txt
      echo "${ngvcAgentPrefix}: Customization Failed..." >> $logPath
   fi
}


checkAgentCustomizationFlags() {
   customizationFlags=`vmware-rpctool "info-get guestinfo.AgentCustomizationFlags"` 2>/dev/null
   if [ "$?" != "0" ]; then
      echo "${ngvcAgentPrefix}: agent customization flag doesn't exist." >> $logPath
      return
   else
      echo "${ngvcAgentPrefix}: agent customization flag is $customizationFlags." >> $logPath
   fi

   OkToCompleteCustomization
   if [ "$?" = "0" ]; then
      ProcessShutdownOrReboot
   fi
}

# return 1: complete customization
# return 0: need to process shutdown or reboot
OkToCompleteCustomization() {

   SHUTDOWN_REQUIRED=$(($customizationFlags & $CUSTOMIZATION_FLAG_SHUTDOWN_REQUIRED))
   SHUTDOWN_REQUESTED=$(($customizationFlags & $CUSTOMIZATION_FLAG_SHUTDOWN_REQUESTED))
   SHUTDOWN_COMPLETED=$(($customizationFlags & $CUSTOMIZATION_FLAG_SHUTDOWN_COMPLETED))
   REBOOT_REQUIRED=$(($customizationFlags & $CUSTOMIZATION_FLAG_REBOOT_REQUIRED))
   REBOOT_REQUESTED=$(($customizationFlags & $CUSTOMIZATION_FLAG_REBOOT_REQUESTED))
   REBOOT_COMPLETED=$(($customizationFlags & $CUSTOMIZATION_FLAG_REBOOT_COMPLETED))

   # If no flags set or all required shutdown/reboots are completed
   # we can complete customization
   if [ $SHUTDOWN_REQUIRED -eq 0 ] && [ $REBOOT_REQUIRED -eq 0 ]; then
        # Neither required, customization is done, so we can complete
      echo "${ngvcAgentPrefix}: No flags set, completing customization" >> $logPath
      return 1
   elif [ $SHUTDOWN_REQUIRED -ne 0 ] && [ $REBOOT_REQUIRED -ne 0 ]; then
      if [ $SHUTDOWN_COMPLETED -ne 0 ] && [ $REBOOT_COMPLETED -ne 0 ]; then
          # Both shutdown and reboot required and completed
          echo "${ngvcAgentPrefix}: Shutdown and reboot were required and have completed, completing customization" >> $logPath
      fi
      return 1
   elif [ $SHUTDOWN_REQUIRED -ne 0 ] && [ $SHUTDOWN_COMPLETED -ne 0 ]; then
      # Only shutdown required and completed
      echo "${ngvcAgentPrefix}: Only a shutdown was required and has completed, completing customization" >> $logPath
      return 1
   elif [ $REBOOT_REQUIRED -ne 0 ] && [ $REBOOT_COMPLETED -ne 0 ]; then
      # Only reboot required and completed
      echo "${ngvcAgentPrefix}: Only a reboot was required and has completed, completing customization" >> $logPath
      return 1
   fi

   return 0
}

ProcessShutdownOrReboot() {
   # Process shutdown and/or reboot
   if [ $SHUTDOWN_REQUIRED -ne 0 ] && [ $SHUTDOWN_COMPLETED -eq 0 ]; then
      # Shutdown is required, but not completed yet, so do the shutdown now
      # Set shutdown requested flag
      echo "${ngvcAgentPrefix}: shut down as required" >> $logPath
      # SetCustomizationFlags $CUSTOMIZATION_FLAG_SHUTDOWN_REQUESTED
      vmware-rpctool "info-set guestinfo.AgentCustomizationFlags $CUSTOMIZATION_FLAG_SHUTDOWN_REQUESTED"

      shutdown -h now
   elif [ $REBOOT_REQUIRED -ne 0 ] && [ $REBOOT_COMPLETED -eq 0 ]; then
      # Reboot is required, but not completed yet, so do the reboot
      # Set reboot requested flag
      echo "${ngvcAgentPrefix}: reboot as required" >> $logPath
      #SetCustomizationFlags $CUSTOMIZATION_FLAG_REBOOT_REQUESTED
      vmware-rpctool "info-set guestinfo.AgentCustomizationFlags $CUSTOMIZATION_FLAG_REBOOT_REQUESTED"

      reboot
   fi
}

# return 1: shut down or reboot complete
# return 0: doesn't need reboot/shutdown
checkShutDownOrRebootComplete() {
   customizationFlags=`vmware-rpctool "info-get guestinfo.AgentCustomizationFlags"` 2>/dev/null
   if [ "$?" != "0" ]; then
      echo "${ngvcAgentPrefix}: agent customization flag doesn't exist." >> $logPath
      return 0
   else
      echo "${ngvcAgentPrefix}: agent customization flag is $customizationFlags." >> $logPath
   fi

   SHUTDOWN_REQUESTED=$(($customizationFlags & $CUSTOMIZATION_FLAG_SHUTDOWN_REQUESTED))
   SHUTDOWN_COMPLETED=$(($customizationFlags & $CUSTOMIZATION_FLAG_SHUTDOWN_COMPLETED))
   REBOOT_REQUESTED=$(($customizationFlags & $CUSTOMIZATION_FLAG_REBOOT_REQUESTED))
   REBOOT_COMPLETED=$(($customizationFlags & $CUSTOMIZATION_FLAG_REBOOT_COMPLETED))

   if [ $SHUTDOWN_REQUESTED -ne 0 ] && [ $SHUTDOWN_COMPLETED -eq 0 ]; then
      flag=$(($customizationFlags|$CUSTOMIZATION_FLAG_SHUTDOWN_COMPLETED))
      vmware-rpctool "info-set guestinfo.AgentCustomizationFlags $flag"
      echo "${ngvcAgentPrefix}: shut down completed." >> $logPath
      return 1
   elif [ $REBOOT_REQUESTED -ne 0 ] && [ $REBOOT_COMPLETED -eq 0 ]; then
      falg=$(($customizationFlags|$CUSTOMIZATION_FLAG_REBOOT_COMPLETED))
      vmware-rpctool "info-set guestinfo.AgentCustomizationFlags $flag"
      echo "${ngvcAgentPrefix}: reboot completed." >> $logPath
      return 1
   fi
   return 0
}

#
# main
#
touch $logPath

#
# Include common constants
#
. "`dirname $0`/commonlib.sh"
identify_distribution
define_constants

echo "${ngvcAgentPrefix}: Running NGVC script..." >> $logPath

it=`vmware-rpctool "info-get guestinfo.it" 2>/dev/null`
if [ "$?" != "0" ]; then
   echo "${ngvcAgentPrefix}: Command get guestinfo.it failed or guestinfo.it do not exist..." >> $logPath
fi
if [ "$it" = "1" ]; then
   # set ngaFork = 1 when IT is ready
   vmware-rpctool "info-set guestinfo.ngaFork 1" >> $logPath 2>&1

   echo "${ngvcAgentPrefix}: It's in Internal Template..." >> $logPath

   # set customization state to sucess in IT stage
   vmware-rpctool "info-set guestinfo.it.CustomizationState success" >> $logPath 2>&1
   if [ "$?" != "0" ]; then
      echo "${ngvcAgentPrefix}: Setting guestinfo.it.CustomizationState failed..." >> $logPath
   fi
   sleep 1
   shutdown -h now
   exit 1
fi

replica=`vmware-rpctool "info-get guestinfo.replica" 2>/dev/null`
if [ "$?" != "0" ]; then
   echo "${ngvcAgentPrefix}: Command get guestinfo.replica failed or guestinfo.replica do not exist..." >> $logPath
fi

ngaFork=`vmware-rpctool "info-get guestinfo.ngaFork" 2>/dev/null`
if [ "$?" != "0" ]; then
   echo "${ngvcAgentPrefix}: Command get guestinfo.ngaFork failed or guestinfo.ngaFork do not exist..." >> $logPath
fi

forked=`vmware-rpctool "info-get guestinfo.forked" 2>/dev/null`
if [ "$?" != "0" ]; then
   echo "${ngvcAgentPrefix}: Command get guestinfo.forked failed or guestinfo.forked do not exist..." >> $logPath
fi

if [ "$ngaFork" = "1" -a "$forked" != "1" ]; then

   if [ "$replica" = '1' ]; then #{
      echo "${ngvcAgentPrefix}: This's in Replica..." >> $logPath
      vmware-rpctool "info-set guestinfo.replica.CustomizationState success" >> $logPath 2>&1
      sleep 3
      shutdown -h  now
      exit 2
   #}
   else #{ replica = '0'

      echo "${ngvcAgentPrefix}: This's Parent VM..." >> $logPath
      # set forked = 1, so this shell will not exec after reboot
      vmware-rpctool "info-set guestinfo.forked 1"

      # set customization state to success in parent stage
      vmware-rpctool "info-set guestinfo.parent.CustomizationState success"
      sleep 10

      icVer=`vmware-rpctool "info-get guestinfo.ic.version"`
      icCommand="vmfork-begin -1 -1"
      if [ "$icVer" = "GEN3" ]; then
         icCommand="instantclone.freeze"
         echo "${ngvcAgentPrefix}: This is GEN3" >> $logPath
      fi
      vmware-rpctool "${icCommand}"

      echo "${ngvcAgentPrefix}: This's Clone VM..." >> $logPath

      # set vmForked value into config file
      sed -i '/^'VMFORKED'/d' /etc/vmware/viewagent-config.txt  >> $logPath 2>&1
      echo "VMFORKED=1" >> /etc/vmware/viewagent-config.txt

      changeHostName

      domainJoinTool=`awk '/^OfflineJoinDomain/{print}' /etc/vmware/viewagent-custom.conf | cut -f2 -d"="`
      if [ -z "$domainJoinTool" ]; then
         domainJoinTool="pbis"
      fi
      MachinePassword=
      acquireMachinePassword
      if [ "$domainJoinTool" = "pbis" ]; then
         echo "${ngvcAgentPrefix}: Beginning join domain offline for PBIS..." >> $logPath
         offlineDomainJoin4PBIS
         if [ "$?" != "0" ]; then
            echo "${ngvcAgentPrefix}: Failed to join domain offline for PBIS..." >> $logPath
            customizationState="fail"
         fi
      elif [ "$domainJoinTool" = "samba" ]; then
         echo "${ngvcAgentPrefix}: Beginning join domain offline for Samba..." >> $logPath
         offlineDomainJoin4Samba
         if [ "$?" != "0" ]; then
            echo "${ngvcAgentPrefix}: Failed to join domain offline for Samba..." >> $logPath
            customizationState="fail"
         fi
      else
         echo "${ngvcAgentPrefix}: Not supported domain join tool: ${domainJoinTool}..." >> $logPath
      fi

      # cleanup secret info
      vmware-rpctool "info-set guestinfo.machinePasswd none"
      vmware-rpctool "info-set guestinfo.machinePasswdChecksum none"

      setCustomizationStatus $customizationState

      checkAgentCustomizationFlags

      runPostCustomizationScript

      # ensure the host name has been set
      changeHostName

      reboot

   fi #} $replica = '0'
else
   if [ "$forked" = "1" ]; then
      echo "${ngvcAgentPrefix}: This VM has been forked..." >> $logPath
      checkShutDownOrRebootComplete
      if [ "$?" = "1" ]; then
         runPostCustomizationScript
      fi
   else
      echo "${ngvcAgentPrefix}: This VM is Master or non-cloned VM..." >> $logPath
   fi
fi
