#!/bin/sh

########################################################
# Copyright 2015-2020 VMware, Inc.  All rights reserved.
########################################################


initialise() {
   HOST="`hostname`"
   NAME="`date +$HOST-vdm-sdct-%Y%m%d-%H%M-agent`"
   FILES=""
   TMP_FILES=""
   TMP_DIR=/tmp/${NAME}
   mkdir ${TMP_DIR}
}


finalise() {
   echo "Creating ${NAME}.tgz"
   tar cfhz "${NAME}.tgz" ${FILES} -C ${TMP_DIR} ${TMP_FILES} 2>/dev/null
   rm -fr ${TMP_DIR}
}


gather_agent() {
   VIEWAGENT_PRODUCT=/usr/lib/vmware/viewagent/Product.txt
   VIEWAGENT_XORG_CONF=/usr/lib/vmware/viewagent/resources/X11/xorg.conf
   VIEWAGENT_LOG=/var/log/vmware/viewagent*.log*
   PAM_PLUGIN_LOG=/var/log/vmware/pam_vmw_log
   VMWARE_LOG=`find /tmp/vmware-* -regex '.*/vmware-.*log'`
   VIEWAGENT_LOG_EVENT=`find /var/vmware -regex '.*/logEvent'`
   SCREEN_LOCK_STATUS=`find /var/vmware -regex '.*/screenLockStatus.*'`
   VIEWAGENT_ENV_INFO=`find /var/vmware -regex '.*/envInfo'`
   FILES="$FILES $VIEWAGENT_PRODUCT $VIEWAGENT_LOG $PAM_PLUGIN_LOG $VMWARE_LOG"
   FILES="$FILES $VIEWAGENT_ENV_FILE $VIEWAGENT_LOG_EVENT"
   FILES="$FILES $SCREEN_LOCK_STATUS"
   FILES="$FILES $BORA_CFG_FILE $CUSTOM_CFG_FILE"
   FILES="$FILES $VIEWAGENT_ENV_INFO $VIEWAGENT_XORG_CONF"

   AGENT_CFG_DCT=viewagent-config.txt

   grep -v JMS_KEYSTORE_PASSWORD $AGENT_CFG_FILE \
      > ${TMP_DIR}/$AGENT_CFG_DCT
   echo "# JMS_KEYSTORE_PASSWORD stripped" >> ${TMP_DIR}/$AGENT_CFG_DCT 2>&1

   TMP_FILES="$TMP_FILES $AGENT_CFG_DCT"
}


gather_UI() {
   which pkaction >/dev/null 2>&1
   if [ "$?" = "0" ];then
      PKACTION=pkaction.txt
      pkaction > ${TMP_DIR}/${PKACTION} 2>&1
      TMP_FILES="$TMP_FILES $PKACTION"
   fi
}


gather_X() {
   XORG_LOG=/var/log/Xorg*
   XORG_CONF=/etc/X11/xorg.conf
   FILES="$FILES $XORG_LOG $XORG_CONF"
}


gather_network() {
   HOSTS=/etc/hosts
   RESOLV=/etc/resolv.conf
   NSSWITCH=/etc/nsswitch.conf

   FILES="$FILES $HOSTS $RESOLV $NSSWITCH"

   IFCONFIG=ifconfig.txt
   FIREWALL=iptables-save.txt
   NETSTAT=netstat.txt

   which ifconfig >/dev/null 2>&1
   if [ "$?" = "0" ]; then
      ifconfig -a > ${TMP_DIR}/${IFCONFIG} 2>&1
   else
      which ip >/dev/null 2>&1 && ip address > ${TMP_DIR}/${IFCONFIG} 2>&1
   fi
   iptables-save > ${TMP_DIR}/${FIREWALL} 2>&1
   which netstat >/dev/null 2>&1
   if [ "$?" = "0" ]; then
      netstat -an > ${TMP_DIR}/${NETSTAT} 2>&1
   else
      which ss >/dev/null 2>&1 && ss -an > ${TMP_DIR}/${NETSTAT} 2>&1
   fi

   TMP_FILES="$TMP_FILES $IFCONFIG $FIREWALL $NETSTAT"
}


gather_system() {
   case "$DISTRO_ID" in
   ${DISTRO_ID_CENTOS}|\
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_RHEL_WORKSTATION}|\
   ${DISTRO_ID_SUSE}|\
   ${DISTRO_ID_NEOKYLIN})
      SYSTEM_LOG="/var/log/messages*"
      ;;
   ${DISTRO_ID_UBUNTU}|\
   ${DISTRO_ID_KYLIN})
      SYSTEM_LOG="/var/log/syslog*"
      ;;
   esac

   #
   # Common files
   #
   BOOT_LOG=/var/log/boot*.log

   FILES="$FILES $BOOT_LOG $SYSTEM_LOG $PAM_PATH"

   HOSTNAME=hostname.txt
   DOMAIN=domain.txt
   PTREE=ptree.txt
   SYSTEM_UUID=system-uuid.txt
   JAVA_DCT=java-version.txt
   DMESG=dmesg.txt

   hostname > ${TMP_DIR}/${HOSTNAME} 2>&1
   domainname > ${TMP_DIR}/${DOMAIN} 2>&1
   dmidecode -s system-uuid > ${TMP_DIR}/${SYSTEM_UUID} 2>&1
   ps -auxf > ${TMP_DIR}/${PTREE} 2>&1
   dmesg > ${TMP_DIR}/${DMESG} 2>&1

   JAVA=/usr/lib/vmware/viewagent/jre/bin/java
   ${JAVA} -version > ${TMP_DIR}/${JAVA_DCT} 2>&1

   TMP_PROC_DIR=${TMP_DIR}/proc
   mkdir -p ${TMP_PROC_DIR}
   while read proc_file
   do
      cat /proc/${proc_file} > ${TMP_PROC_DIR}/${proc_file} 2>&1
      TMP_FILES="$TMP_FILES proc/${proc_file}"
   done << !!!
cpuinfo
meminfo
vmstat
loadavg
!!!

   #initctl list
   #service --status-all

   TMP_FILES="$TMP_FILES $HOSTNAME $DOMAIN $PTREE $SYSTEM_UUID $JAVA_DCT $DMESG"
}

gather_truesso() {
   if [ -f "/usr/lib/vmware/viewagent/sso/libvmwpkcs11.so" ]; then
      SECURE_LOG="/var/log/secure*"
      KRB5_CONFIG="/etc/krb5.conf"
      SSSD_CONFIG="/etc/sssd/sssd.conf"
      FILES="$FILES $SECURE_LOG $KRB5_CONFIG $SSSD_CONFIG"
   fi
}

gather_authentication() {
   gather_truesso

   case "$DISTRO_ID" in
   ${DISTRO_ID_CENTOS}|\
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_RHEL_WORKSTATION}|\
   ${DISTRO_ID_SUSE}|\
   ${DISTRO_ID_NEOKYLIN})
      AUTH_LOG="/var/log/audit/audit.log*"
      DM_LOG="/var/log/gdm"
      ;;
   ${DISTRO_ID_UBUNTU}|\
   ${DISTRO_ID_KYLIN})
      AUTH_LOG="/var/log/auth.log*"
      DM_LOG="/var/log/lightdm"
      ;;
   *)
      echo "$DISTRO_ID not supported"
      exit
      ;;
   esac

   PAM_CONFIG="/etc/pam.d"

   FILES="$FILES $AUTH_LOG $DM_LOG $PAM_CONFIG"
}


gather_coredumps() {
   case "$DISTRO_ID" in
   ${DISTRO_ID_CENTOS}|\
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_RHEL_WORKSTATION}|\
   ${DISTRO_ID_SUSE}|\
   ${DISTRO_ID_NEOKYLIN})
      COREFILES="/usr/lib/vmware/viewagent/core.*"
      ;;
   ${DISTRO_ID_UBUNTU}|\
   ${DISTRO_ID_KYLIN})
      #
      # Dependent upon the release / version of Ubuntu / apport, the following
      # has been tested on Ubuntu 14.04 LTS.
      #
      COREFILES="/usr/lib/vmware/viewagent/core"
      ;;
   *)
      echo "$DISTRO_ID not supported"
      exit
      ;;
   esac

   FILES="$FILES $COREFILES"
}


gather_crashdumps() {
   #
   # On Ubuntu the apport service currently gathers in core dumps from
   # third party binaries, hence look for crashes relating to Linux Agent
   # comments.   Longer term this should be made more generic, i.e. disable
   # apport for Linux Agent binaries and support across all distributions via
   # core dumps.
   #

   case "$DISTRO_ID" in
   ${DISTRO_ID_CENTOS}|\
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_RHEL_WORKSTATION}|\
   ${DISTRO_ID_SUSE}|\
   ${DISTRO_ID_NEOKYLIN})
      ;;
   ${DISTRO_ID_UBUNTU}|\
   ${DISTRO_ID_KYLIN})
      CRASHFILES="/var/crash/_usr_lib_vmware_viewagent*"
      ;;
   *)
      echo "$DISTRO_ID not supported"
      exit
      ;;
   esac

   FILES="$FILES $CRASHFILES"
}


#
# main
#

#
# Ensure script is executed as root.
#
if [ "`id -u`" != "0" ]; then
   echo "dct-debug.sh not run as superuser"
   exit
fi


#
# Include common constants and routines
#
. "`dirname $0`/commonlib.sh"


identify_distribution
configure_distribution_common
initialise
gather_system
gather_authentication
gather_network
gather_X
gather_UI
gather_agent
gather_coredumps
gather_crashdumps
finalise
