#!/bin/bash

###################################################
# Copyright 2015-2020 VMware, Inc.  All rights reserved.
###################################################


#
# This file should not be executed directly.
#
if [ "`basename $0`" = "commonlib.sh" ]; then
   echo "commonlib.sh is not intended to be executed directly."
   exit
fi

#
# Wrap "echo" command to dump log to either terminal/log file or both.
#
vmecho() {
   local showOnTerm="no"
   local newLine=""
   local param=""
   local msgType=""
   local i=1;

   msgType="$1"

   case "$msgType" in
      term)
         showOnTerm="yes"
         ;;
      log)
         showOnTerm="no"
         ;;
   esac

   if [ "$2" = "-n" ]; then
      newLine="no"
      shift
      shift
   else
      newLine="yes"
      shift
   fi

   #
   # Collect all echo's parameters like: echo "XX" "YY" "ZZ"
   #
   while [ "$i" -le "$#" ]; do
      param="$param$1"
      i=`expr $i + 1`
   done

   #
   # Create log dir if it doesn't exist
   #
   if [ ! -e "VMW_LOG_PATH" ]; then
      mkdir -p "$VMW_LOG_PATH"
      chmod 755 "$VMW_LOG_PATH"
   fi

   if [ "$showOnTerm" = "no" ]; then
      if [ "$newLine" = "yes" ]; then
         echo "$param" >> "$VIEWAGENT_INSTALLER_LOG"
      else
         echo -n "$param" >> "$VIEWAGENT_INSTALLER_LOG"
      fi
   else
      if [ "$newLine" = "yes" ]; then
         echo "$param" | tee -a "$VIEWAGENT_INSTALLER_LOG"
      else
         echo -n "$param" | tee -a "$VIEWAGENT_INSTALLER_LOG"
      fi
   fi
}

#
# Dump message to both terminal and log file.
#
echo_term() {
   vmecho term "$@"
   return 0
}

#
# Dump message to log file only.
#
echo_log() {
   vmecho log "$@"
   return 0
}

#
# Command funcions
#

identify_distribution() {
   DISTRO_ID_CENTOS=CentOS
   DISTRO_ID_RHEL_CLIENT=RedHatEnterpriseClient
   DISTRO_ID_RHEL_SERVER=RedHatEnterpriseServer
   DISTRO_ID_RHEL_WORKSTATION=RedHatEnterpriseWorkstation
   #rhel 8 doesn't have client/server/worksation, so we use this to identify
   DISTRO_ID_RHEL_ENTERPRISE=RedHatEnterprise
   DISTRO_ID_NEOKYLIN=" NeoKylin Linux Desktop"
   DISTRO_ID_UBUNTU=Ubuntu
   DISTRO_ID_SUSE="SUSE LINUX"
   #
   # Virtual distro IDs
   #
   DISTRO_ID_KYLIN=UbuntuKylin

   type lsb_release >/dev/null 2>&1
   if [ "$?" = "0" ]; then
      DISTRO_ID="`lsb_release -is`"
      DISTRO_RELEASE="`lsb_release -rs`"
      DISTRO_DESC="`lsb_release -ds`"
      if [ "$DISTRO_ID" = "$DISTRO_ID_RHEL_ENTERPRISE" ]; then
         DISTRO_ID=$DISTRO_ID_RHEL_WORKSTATION
      fi
   else
      if [ -f /etc/centos-release ]; then
         DISTRO_ID=$DISTRO_ID_CENTOS
         DISTRO_DESC="`cat /etc/centos-release`"
      elif [ -f /etc/redhat-release ]; then
         DISTRO_DESC="`cat /etc/redhat-release`"
         DISTRO_ID=
         echo $DISTRO_DESC | grep "Client" >/dev/null 2>&1 && \
            DISTRO_ID=$DISTRO_ID_RHEL_CLIENT
         [ -z "$DISTRO_ID" ] && \
            echo $DISTRO_DESC | grep "Server" >/dev/null 2>&1 && \
            DISTRO_ID=$DISTRO_ID_RHEL_SERVER
         [ -z "$DISTRO_ID" ] && \
            echo $DISTRO_DESC | grep "Workstation" >/dev/null 2>&1 && \
            DISTRO_ID=$DISTRO_ID_RHEL_WORKSTATION
         [ -z "$DISTRO_ID" ] && DISTRO_ID=$DISTRO_ID_RHEL_WORKSTATION
      else
         echo "Unable to identify distribution and release."
         exit
      fi
      DISTRO_RELEASE="`echo $DISTRO_DESC | awk '{ for (i=1; i < NF; i++) { if ($i == "release") { print $(i+1); break }}}'`"
   fi

   #
   # Differentiate between Ubuntu and Ubuntu Kylin.
   #
   if [ "$DISTRO_ID" = "$DISTRO_ID_UBUNTU" ]; then
      #
      # Distinguish between Ubuntu and Ubuntu Kylin where possible.   Release
      # files are common place and the following file exists on
      # Ubuntu Kylin 14.04 LTS, but not on Ubuntu Kylin 13.04.   Your mileage
      # may vary...
      if [ -f /etc/ubuntukylin-release ]; then
         DISTRO_ID=$DISTRO_ID_KYLIN
      fi
   fi

   if [ "$DISTRO_ID" = "SUSE" ]; then
      #
      # Add the distribution ID "SUSE" to the DISTRO_ID_SUSE
      #
      DISTRO_ID=$DISTRO_ID_SUSE
   fi

   #
   # Check supported distributions
   #
   case "$DISTRO_ID" in
   ${DISTRO_ID_CENTOS}|\
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_SUSE}|\
   ${DISTRO_ID_RHEL_WORKSTATION}|\
   ${DISTRO_ID_NEOKYLIN})
      ;;
   ${DISTRO_ID_KYLIN}|${DISTRO_ID_UBUNTU})
      echo $DISTRO_DESC | grep " LTS$" >/dev/null 2>&1
      if [ "$?" != "0" ]; then
         echo "$DISTRO_DESC not an LTS release"
         exit
      fi
      ;;
   *)
      echo "$DISTRO_ID not supported"
      exit
      ;;
   esac

   case "$1" in
      -d)
         echo "$DISTRO_DESC";;
      -r)
         echo "$DISTRO_RELEASE";;
      -i)
         echo "$DISTRO_ID";;
      *)
         ;;
   esac
}


define_constants() {
   OS_MAJOR=`echo "$DISTRO_RELEASE" | cut -f1 -d"."`
   OS_MINOR=`echo "$DISTRO_RELEASE" | cut -f2 -d"."`
   VIEWAGENT_TAR="VMware-viewagent-standalone-*"
   VMWARE_PATH=/usr/lib/vmware
   VIEWAGENT_PATH=/usr/lib/vmware/viewagent
   VIEWAGENT_TMP_ROOT=/var/vmware
   VIEWAGENT_TMP_PATH=$VIEWAGENT_TMP_ROOT/viewagent
   VIEWAGENT_BACKUP_PATH="${VIEWAGENT_PATH}/backup"
   VIEWAGENT_DESKTOP_PATH="${VIEWAGENT_PATH}/desktop"
   JRE_PATH=/usr/lib/vmware/viewagent/jre
   VIEWAGENT_ENV_FILE=/etc/sysconfig/viewagent
   VIEWAGENT_ENV_FILE_UBUNTU=/etc/default/viewagent
   VMW_SSL_PATH="/etc/vmware/ssl"
   AGENT_CFG_FILE=/etc/vmware/viewagent-config.txt
   JMS_KEYSTORE_DIR=/etc/vmware/jms
   VIEWAGENT_UPSTART=viewagent.conf
   UPSTART_CONF_PATH=/etc/init
   VIEWAGENT_SYSV=viewagent
   VIEWAGENT_SYSV_SUSE=viewagent.suse
   SYSV_CONF_PATH=/etc/rc.d/init.d
   SYSV_CONF_PATH_SUSE=/etc/init.d
   VIEWAGENT_SYSTEMD=viewagent.service
   VIEWAGENT_SYSTEMD_UBUNTU=viewagent_ubuntu_systemd.service
   VIEWAGENT_SYSTEMD_UBUNTU1804=viewagent_ubuntu1804_systemd.service
   SYSTEMD_CONF_PATH=/usr/lib/systemd/system
   SYSTEMD_CONF_PATH_UBUNTU=/lib/systemd/system
   # shutdown service constants
   SHUTDOWN_SYSV=viewagent_shutdown
   SHUTDOWN_SYSV_SUSE=viewagent_shutdown.suse
   SHUTDOWN_SYSTEMD_UBUNTU=viewagent_shutdown_ubuntu_systemd.service
   SHUTDOWN_SYSTEMD=viewagent_shutdown.service
   SHUTDOWN_ENV_FILE=/etc/sysconfig/viewagent_shutdown
   SHUTDOWN_ENV_FILE_UBUNTU=/etc/default/viewagent_shutdown
   SHUTDOWN_UPSTART=viewagent_shutdown.conf

   VIEWAGENT_POLICYKIT=10-com.vmware.viewagent.pkla
   POLICYKIT_PATH=/etc/polkit-1/localauthority/90-mandatory.d
   PAM_PATH=/etc/pam.d
   LIGHTDM_PATH=/etc/lightdm
   GLIB_SCHEMAS_PATH=/usr/share/glib-2.0/schemas
   VMWARE_SCHEMA_OVERRIDE=50_vmware_viewagent.gschema.override
   BORA_CFG_FILE=/etc/vmware/config
   MACHINE_CFG_FILE=/etc/vmware/viewagent-machine.cfg
   BLAST_SERVER_PATH=/usr/lib/vmware/viewagent/VMwareBlastServer/BlastServer
   DESKTOP_DAEMON_PATH=/usr/lib/vmware/viewagent/DesktopDaemon/desktopDaemon
   START_CLIPBOARD_SH=/usr/lib/vmware/viewagent/bin/StartClipboard.sh
   START_CDR_SH=/usr/lib/vmware/viewagent/bin/StartCDRServer.sh
   START_RDE_SERVER_SH=/usr/lib/vmware/viewagent/bin/StartRdeServer.sh
   STOP_RDE_SERVER_SH=/usr/lib/vmware/viewagent/bin/StopRdeServer.sh
   COLLABORATION_UI_PATH=/usr/lib/vmware/viewagent/collaboration/collabui
   CONTROL_EXTENSION_PY=${VIEWAGENT_DESKTOP_PATH}/ControlExtension.py
   CONTROL_THEME_SH=${VIEWAGENT_DESKTOP_PATH}/ControlTheme.sh
   LAUNCH_APP_SH=/usr/lib/vmware/viewagent/bin/LaunchApp.sh
   RUN_APPSCANNER_SH=/usr/lib/vmware/viewagent/bin/RunAppScanner.sh
   APP_ICON_DIR=/etc/vmware/icons
   APP_THEME_DIR=/usr/share/themes/
   APP_XML_FILE=/etc/vmware/viewagent-apps.xml
   GNOME_SHELL_EXTENSIONS_DIR=/usr/share/gnome-shell/extensions/
   VMTOOL_XDG_FOND_MENUS=/usr/lib/vmware/viewagent/bin/xdg-find-menus
   VMWARE_XDG_DETECT_DE=/usr/lib/vmware/viewagent/bin/vmware-xdg-detect-de
   START_USBREDIR_SH=/usr/lib/vmware/viewagent/bin/StartUsbRedir.sh
   STOP_USBREDIR_SH=/usr/lib/vmware/viewagent/bin/StopUsbRedir.sh
   CONTROL_DESKTOP_SH=/usr/lib/vmware/viewagent/bin/ControlDesktop.sh
   CLEANLOG_SH=/usr/lib/vmware/viewagent/bin/CleanupLogFiles.sh
   MOUNT_FOLDER_SH=/usr/lib/vmware/viewagent/bin/MountFolder.sh
   UMOUNT_FOLDER_SH=/usr/lib/vmware/viewagent/bin/UMountFolder.sh
   START_XSERVER_SH=/usr/lib/vmware/viewagent/bin/StartXServer.sh
   START_BLASTWORKER_SH=/usr/lib/vmware/viewagent/bin/StartBlastWorker.sh
   POST_LOGIN_SH=/usr/lib/vmware/viewagent/bin/PostLogin.sh
   VMW_LOG_PATH=/var/log/vmware
   AGENT_ICON_PATH=/usr/share/vmware/icon/
   COLLABORATION_ICON_PATH=$AGENT_ICON_PATH/collabui/
   COLLABORATION_TRAY_ICON=vmware-collaboration-tray-icon.png
   VIEWAGENT_REG_LOG=${VMW_LOG_PATH}/viewagent-registration.log
   VIEWAGENT_INSTALLER_LOG=${VMW_LOG_PATH}/viewagent-installer.log
   PUBLIC_KEY_PATH=/usr/share/vmware
   PUBLIC_KEY=$PUBLIC_KEY_PATH/rui.pub
   PRIVATE_KEY=$VMW_SSL_PATH/rui.key
   PRIVATE_KEY_SIZE=2048
   CERT_REQ=$VMW_SSL_PATH/rui.csr
   CERT=$VMW_SSL_PATH/rui.crt
   CERT_VALIDITY_PERIOD=1460
   BLAST_RUNTIME_DIR=/tmp/vmwblast
   BLAST_ACCOUNT=vmwblast
   BLAST_GROUP=vmwblast
   BLAST_PORT=22443
   DNPREFIX="/C=US/ST=California/O=VMware"
   VIEWAGENT_DESKTOPC_SCRIPT="viewagent-desktopc.desktop"
   VIEWAGENT_GDM_GREETER_LOGIN_WINDOW_FILE="gdm-greeter-login-window.glade"
   VIEWAGENT_CUSTOMISE_DESKTOP_LOG=/var/log/vmware/viewagent-customise.log
   VIEWAGENT_RUNTIME_DIR=/var/run/vmware/viewagent
   SYSCTL_BACKUP_DIR="${VIEWAGENT_PATH}/sysctl/"
   UDEV_RULES_PATH="/etc/udev/rules.d"
   LOCALE_PROFILE_PATH="/etc/profile.d"
   XORG_CONF_DIR="/etc/X11"
   VIEWAGENT_XORG_CONF_FILE="$VIEWAGENT_PATH/resources/X11/xorg.conf"
   VVC_PATH="$VIEWAGENT_PATH/VVC"
   VCHAN_PLUGINS_PATH="$VVC_PATH/vchan_plugins"
   HELPDESK_PATH="$VIEWAGENT_PATH/helpdesk"
   COLLABUI_PATH="$VIEWAGENT_PATH/collaboration"
   COLLABUI="$COLLABUI_PATH/collabui"
   USBREDIR_SERVER="$VCHAN_PLUGINS_PATH/usbRedirectionServer"
   CLIPBOARD_SERVER="$VCHAN_PLUGINS_PATH/mksvchanserver"
   CDR_SERVER="$VCHAN_PLUGINS_PATH/cdrserver"
   RDE_SERVER="$VCHAN_PLUGINS_PATH/rdeSvc"
   APPSCANNER="$VCHAN_PLUGINS_PATH/appScanner"
   SCREDIR_BACKUP_DIR="${VIEWAGENT_PATH}/scredir/backup"
   SCREDIR_IPC_DIR="/var/run/vvcipc"
   IPC_PORT_FILE="/var/vmware/ipc.port"
   IPC_MAGIC_FILE="/var/vmware/viewagent/ipc.magic"
   DESKTOP_TMP_PATH="$VIEWAGENT_TMP_ROOT/vmw_desktop"
   # TODO: Use blast instead of pcoip
   BLAST_LIB_PATH="/usr/lib/pcoip"
   BLAST_VCHAN_PLUGINS_PATH="$BLAST_LIB_PATH/vchan_plugins"
   SCREDIR_VCHAN_PLUGINS_PATH="$VMWARE_PATH/vchan_plugins"
   AUDIOIN_VCHAN_PLUGINS_PATH="$VMWARE_PATH/vchan_plugins"
   DBUS_CONF_PATH=/etc/dbus-1/system.d
   SSO_DBUS_PY=/usr/lib/vmware/viewagent/bin/SsoDBusNotify.py
   SSO_CONFIG_DESKTOP_TYPE_SH=/usr/lib/vmware/viewagent/bin/SsoConfigDesktopType.sh
   SSO_BACKUP_PATH="${VIEWAGENT_PATH}/sso/backup"
   # Custom configuration file of viewagent
   CUSTOM_CFG_FILE=/etc/vmware/viewagent-custom.conf
   # Custom configuration file of VMwareBlastServer
   CUSTOM_CFG_FILE_BLAST=/etc/vmware/config
   MINVERSION_XORG_X11_SERVER=7.4
   MINRELEASE_XORG_X11_SERVER=27.111.1
   MINVERSION_XF86_VIDEO_VMWARE=13.0.2
   MINRELEASE_XF86_VIDEO_VMWARE=3.2
   MINVERSION_MESA_LIBXATRACKER=10.6.5
   MINRELEASE_MESA_LIBXATRACKER=3
   MINVERSION_XORG_X11_DRV_VMWARE=13.0.2
   MINRELEASE_XORG_X11_DRV_VMWARE=7
   MINVERSION_MESA_PRIVATE_LLVM=3.6.2
   MINRELEASE_MESA_PRIVATE_LLVM=2
   MINVERSION_MESA_DRI_DRIVERS=10.6.5
   MINRELEASE_MESA_DRI_DRIVERS=3
   BROKER_MESSAGEBOX_TITLE="VMware Horizon View Administrator"
   GNOME_DESKTOP="gnome"
   MATE_DESKTOP="mate"
   KDE_DESKTOP="kde"
   NGVC_SHUTDOWN=vmwareNgvcShutDown.sh
   XSERVER_VMWMEM_ENABLED="true"

   # Auto-generated uninstall scripts directory
   VMW_UNINSTALL_SCRIPTS_DIR="/usr/lib/vmware/viewagent/bin/uninstall"
   VMW_DISPLAY_MANAGER_CONF_BACKUP_DIR=${VIEWAGENT_PATH}/dm-conf/backup
}

identify_xserver_vmwmem_enabled() {
   if grep "^XSERVER_VMWMEM_ENABLED=false" $AGENT_CFG_FILE >/dev/null 2>&1; then
      XSERVER_VMWMEM_ENABLED="false"
   fi
}

define_os_version() {
   # The OS_VERSION 100 means "Linux (other)" type in broker side.
   OS_VERSION=100
   if [ "$SINGLETON_MODE" = "false" ]; then
      # The OS_VERSION 200 means "Linux Server (other)" type in broker side.
      OS_VERSION=200
   fi
}

define_commands() {
   which addgroup >/dev/null 2>&1
   [ "$?" = "0" ] && [ ! -n "$ADDGROUP" ] && ADDGROUP=addgroup
   which groupadd >/dev/null 2>&1
   [ "$?" = "0" ] && [ ! -n "$ADDGROUP" ] && ADDGROUP=groupadd
}

get_singleton_var() {
   if [ -z "$AGENT_CFG_FILE" ]; then
       define_constants
   fi

   if grep "^SINGLETON_MODE=false" $AGENT_CFG_FILE >/dev/null 2>&1
   then
       echo "false"
   else
       echo "true"
   fi
}

configure_distribution_common() {
   #
   # Generic
   #
   define_constants

   #
   # Java binary
   #
   JRE_JAVA="$JRE_PATH/bin/java"

   #
   # Distro specific UI
   #
   case "$DISTRO_ID" in
   ${DISTRO_ID_CENTOS})
      DISTRO_PLATFORM_LABEL=CentOS
      ;;
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_RHEL_WORKSTATION})
      DISTRO_PLATFORM_LABEL=RHEL
      ;;
   ${DISTRO_ID_NEOKYLIN})
      DISTRO_PLATFORM_LABEL=FedoraKylin
      ;;
   ${DISTRO_ID_UBUNTU})
      DISTRO_PLATFORM_LABEL=Ubuntu
      ;;
   ${DISTRO_ID_KYLIN})
      DISTRO_PLATFORM_LABEL=UbuntuKylin
      ;;
   ${DISTRO_ID_SUSE})
      DISTRO_PLATFORM_LABEL=SUSE
      ;;
   esac

   #
   # Identify distro specific service definition and control
   #
   SERVICE_MGR=
   SERVICE_MGR_SYSV=SYSV
   SERVICE_MGR_SYSTEMD=SYSTEMD
   SERVICE_MGR_UPSTART=UPSTART

   [ -z "$SERVICE_MGR" ] && \
      which initctl >/dev/null 2>&1 && SERVICE_MGR=${SERVICE_MGR_UPSTART}

   case "$DISTRO_ID" in
   ${DISTRO_ID_CENTOS}|\
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_RHEL_WORKSTATION})
      #
      # CentOS / RHEL prior to the introduction of systemd (< 7.0) contain
      # Upstart from Fedora however the services are still managed via SysV
      # scripts.   Ignore the availability of Upstart on such systems.
      #
      expr $DISTRO_RELEASE \< 7.0 >/dev/null
      if [ "$?" = "0" ]; then
         SERVICE_MGR=
      fi
      ;;
   ${DISTRO_ID_UBUNTU})
      if [ "$OS_MAJOR" = "16" ] || [ "$OS_MAJOR" = "18" ]; then
         SERVICE_MGR=
      fi
      ;;
   esac

   [ -z "$SERVICE_MGR" ] && \
      which systemctl >/dev/null 2>&1 && SERVICE_MGR=${SERVICE_MGR_SYSTEMD}

   [ -z "$SERVICE_MGR" ] && SERVICE_MGR=${SERVICE_MGR_SYSV}

   #
   # Assume all distributions and releases have compatible service / chkconfig
   # commands regardless of service management employed.   Override in distro
   # specific commands section below if this is not the case.
   #
   SERVICE_START="service"
   SERVICE_START_POST="start"
   SERVICE_STOP="service"
   SERVICE_STOP_POST="stop"
   SERVICE_RESTART="service"
   SERVICE_RESTART_POST="restart"
   SERVICE_SAVE="service"
   SERVICE_SAVE_POST="save"

   #
   # Identify firewall
   #
   FIREWALL=
   FIREWALL_IPTABLES=IPTABLES
   FIREWALL_FIREWALLD=FIREWALLD

   [ -z "$FIREWALL" ] && \
      which firewall-cmd >/dev/null 2>&1 && firewall-cmd --list-all >/dev/null 2>&1 && FIREWALL=${FIREWALL_FIREWALLD}

   [ -z "$FIREWALL" ] && \
      which iptables >/dev/null 2>&1 && FIREWALL=${FIREWALL_IPTABLES}

   #
   # Initialise desktop customisation
   #

   REQUEST_DESKTOP_RESTART=

   GCONFTOOL=
   which gconftool-2 >/dev/null 2>&1 && GCONFTOOL=yes

   GLIB_SCHEMAS=
   which gsettings >/dev/null 2>&1 && [ -d "${GLIB_SCHEMAS_PATH}" ] && \
      which glib-compile-schemas >/dev/null 2>&1 && \
      GLIB_SCHEMAS=yes
   #
   # customise_desktop_gsettings does not support the earlier syntax of
   # gsettings.   Verify the verify present on the system and clear variable
   # if not supported.
   #
   [ "$GLIB_SCHEMAS" = "yes" ] && gsettings list-schemas >/dev/null 2>&1 || \
      GLIB_SCHEMAS=

   POLICYKIT=
   which pkaction >/dev/null 2>&1 && [ -d ${POLICYKIT_PATH} ] && \
      POLICYKIT=yes

   #
   # Distro specific commands
   #
   case "$DISTRO_ID" in
   ${DISTRO_ID_CENTOS}|\
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_RHEL_WORKSTATION})
      PKG_QUERY="rpm -q"
      PKG_INSTALL="yum -q -y install"
      ;;
   ${DISTRO_ID_NEOKYLIN})
      PKG_QUERY="rpm -q"
      PKG_INSTALL="yum -q -y install"
      ;;
   ${DISTRO_ID_KYLIN}|\
   ${DISTRO_ID_UBUNTU})
      PKG_QUERY="dpkg -s"
      PKG_INSTALL="apt-get -q -y install"
      ;;
   ${DISTRO_ID_SUSE})
      PKG_QUERY="rpm -q"
      PKG_INSTALL="zypper -q -n install"
      ;;
   esac
}

cleanSSOFiles() {
   local arch="`uname -i`"
   local archpath="lib64"
   local majorVersion=

   # Don't support 32 bit linux system
   [ "$arch" != "x86_64" ] && return

   case "$DISTRO_ID" in
   ${DISTRO_ID_CENTOS}|\
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_RHEL_WORKSTATION})
      majorVersion=`identify_distribution -r | head -c 1`
      ;;
   *)
      return
   esac

   if [ "$majorVersion" = "6" ]; then
      rm -f /usr/$archpath/gdm/simple-greeter/plugins/vmwcred.so

      rm -fr /usr/share/gdm/simple-greeter/extensions/vmwcred

      rm -f /usr/share/icons/hicolor/16x16/apps/gdm-vmwcred.png

      rm -f /usr/share/icons/hicolor/48x48/apps/gdm-vmwcred.png

      rm -f /$archpath/security/pam_vmw_cred.so

      rm -f /etc/pam.d/gdm-vmwcred

      # Clean the dbus conf file
      rm -f $DBUS_CONF_PATH/org.vmware.viewagent.Credentials.conf
   fi
}

registerRollback() {
   rm -f /etc/xdg/autostart/$VIEWAGENT_DESKTOPC_SCRIPT
   rm -f /etc/xdg/plasma-workspace/env/desktopc.sh >/dev/null 2>&1
   rm -fr $VMWARE_PATH

   # Clean the tmp file
   rm -rf $VIEWAGENT_TMP_PATH

   # Clean the configure files
   rm -fr /etc/vmware
   rm -f $PUBLIC_KEY

   # Clean the blast lib path
   rm -fr $BLAST_LIB_PATH

   # Clean the SSO files
   cleanSSOFiles
}

serviceStart() {
   if [ "$#" != "1" ]; then
      echo "serviceStart: service not specified"
   else
      $SERVICE_START $1 $SERVICE_START_POST > /dev/null 2>&1
   fi
}


serviceRestart() {
   if [ "$#" != "1" ]; then
      echo "serviceRestart: service not specified"
   else
      $SERVICE_RESTART $1 $SERVICE_RESTART_POST > /dev/null 2>&1
   fi
}


serviceStop() {
   if [ "$#" != "1" ]; then
      echo "serviceStop: service not specified"
   else
      $SERVICE_STOP $1 $SERVICE_STOP_POST > /dev/null 2>&1
   fi
}


serviceSave() {
   if [ "$#" != "1" ]; then
      echo "serviceSave: service not specified"
   else
      $SERVICE_SAVE $1 $SERVICE_SAVE_POST > /dev/null 2>&1
   fi
}

setLogFile() {
   exec 3>&1
   exec 4>&2

   exec 1>$1
   exec 2>$1
}

resetLogFile() {
   exec 1>&3
   exec 2>&4
}

customise_desktop() {
   setLogFile $VIEWAGENT_CUSTOMISE_DESKTOP_LOG
   local mode
   local managed

   if [ "$#" != "2" ]; then
      echo "customise_desktop: specify operation"
      return
   fi

   mode="$1"
   managed="$2"

   case "$mode" in
   "set"|"reset")
      customise_desktop_gsettings $mode $managed
      customise_desktop_gconf $mode $managed
      customise_desktop_polkit $mode $managed
      customise_desktop_pam $mode $managed
      if [ "$DISTRO_ID" = "$DISTRO_ID_SUSE" ] && [ "$OS_MAJOR" = "11" ]; then
         customise_desktop_SLED $mode $managed
      fi
      if [ "$DISTRO_ID" = "$DISTRO_ID_SUSE" ] && [ "$OS_MAJOR" = "12" ]; then
         customise_desktop_pulseaudio $mode
         echo $DISTRO_DESC | grep "Server" >/dev/null 2>&1
         if [ "$?" = "0" ]; then
            customise_desktop_SLES $mode
         fi
      fi

      if [ "$DISTRO_ID" = "${DISTRO_ID_UBUNTU}" ]; then
         # customise ubuntu1404 mate, disable the switch user dbus message for ubuntu 1404.
         if [ "${OS_MAJOR}" = "14" ]; then
            customise_desktop_ubuntu1404_mate $mode
         fi

         if [ "${OS_MAJOR}" = "18" ]; then
            # Start pulseaudio before gnome-shell in multilpe session mode
            if [ "${SINGLETON_MODE}" = "false" ]; then
               customise_desktop_pulseaudio $mode
            fi
            # Add polkit color permisson file, to disable authentication popup message at login
            customise_desktop_ubuntu1804_popup $mode
         fi
      fi

      if [ "$DISTRO_ID" = "${DISTRO_ID_CENTOS}" ] ||
         [ "$DISTRO_ID" = "${DISTRO_ID_RHEL_CLIENT}" ] ||
         [ "$DISTRO_ID" = "${DISTRO_ID_RHEL_SERVER}" ] ||
         [ "$DISTRO_ID" = "${DISTRO_ID_RHEL_WORKSTATION}" ]; then
         customise_desktop_RHEL $mode
      fi

      #
      # Check desktop environment before install agent
      #
      ls /usr/share/xsessions | grep "kde" >/dev/null 2>&1 || \
      ls /usr/share/xsessions | grep "plasma" >/dev/null 2>&1
      if [ "$?" = "0" ]; then
         customise_desktop_kde $mode
      fi

      if [ "${SINGLETON_MODE}" = "false" ]; then
         customise_desktop_systemd $mode
      fi
      ;;
   *)
      echo "customise_desktop: specify set or reset"
      ;;
   esac

   if [ "$DISTRO_ID" = "$DISTRO_ID_NEOKYLIN" ]; then
      customise_desktop_NeoKylin $managed
   fi
   resetLogFile
}

customise_desktop_ubuntu1404_mate() {
   local mode
   local DBUS_CONF_FILE_PATH="/etc/dbus-1/system.d/org.freedesktop.DisplayManager.conf"
   local DBUS_CONF_FILE_PATH_ORG="/etc/dbus-1/system.d/org.freedesktop.DisplayManager.conf.org"

   if [ "$#" != "1" ]; then
      echo "customise_desktop_ubuntu1404_mate: specify operation"
      return
   fi


 mode="$1"

   case "$mode" in
   "set")
      # Disable Switch User in the logout screen
      if [ -f "$DBUS_CONF_FILE_PATH" ]; then
         # Backup system original file
         mv -f $DBUS_CONF_FILE_PATH $DBUS_CONF_FILE_PATH_ORG

         # Deploy new file
         cp -fp $VIEWAGENT_PATH/resources/mate/org.freedesktop.DisplayManager.conf $DBUS_CONF_FILE_PATH
      fi
      ;;
   "reset")
      # Restore the system original file
      if [ -f "$DBUS_CONF_FILE_PATH_ORG" ]; then
         mv -f $DBUS_CONF_FILE_PATH_ORG $DBUS_CONF_FILE_PATH
      fi
      ;;
   *)
      echo "customise_desktop_ubuntu1404_mate: specify set or reset"
      ;;
   esac
}


#
# Disable Authentication popup message at login for Redhat/CentOS 6
#
customise_desktop_rhel_popup() {
   local AUTHENTICATION_MESSAGE_FILE
   local AUTHENTICATION_MESSAGE_FILE_ORIG

   if [ ${OS_MAJOR} = "6" ]; then
      AUTHENTICATION_MESSAGE_FILE="/etc/xdg/autostart/gpk-update-icon.desktop"
      AUTHENTICATION_MESSAGE_FILE_ORIG="/etc/xdg/autostart/gpk-update-icon.desktop.orig"
   elif [ ${OS_MAJOR} = "8" ] || ([ ${OS_MAJOR} = "7" ] && ([ ${OS_MINOR} = "2" ] || [ ${OS_MINOR} = "3" ])); then
      AUTHENTICATION_MESSAGE_FILE="/etc/xdg/autostart/gnome-software-service.desktop"
      AUTHENTICATION_MESSAGE_FILE_ORIG="/etc/xdg/autostart/gnome-software-service.desktop.orig"
   else
      return
   fi

   if [ "$1" = "set" ]; then
      if [ -f "$AUTHENTICATION_MESSAGE_FILE" ]; then
         cp -p $AUTHENTICATION_MESSAGE_FILE $AUTHENTICATION_MESSAGE_FILE_ORIG

         grep -q "^X-GNOME-Autostart-enabled=false" $AUTHENTICATION_MESSAGE_FILE ||
         sed -i '$aX-GNOME-Autostart-enabled=false' $AUTHENTICATION_MESSAGE_FILE
      fi
   else
      if [ -f "$AUTHENTICATION_MESSAGE_FILE_ORIG" ]; then
         mv -f $AUTHENTICATION_MESSAGE_FILE_ORIG $AUTHENTICATION_MESSAGE_FILE
      fi
   fi
}


#
# Add polkit color permission file to prevent authentication
# popup message at login for Ubuntu1804
#
customise_desktop_ubuntu1804_popup() {
   local mode
   local POLKIT_SOURCE_FILE_PATH="resources/polkit/50-allow-colord.pkla"
   local POLKIT_TARGET_FILE_PATH="/etc/polkit-1/localauthority/50-local.d/50-allow-colord.pkla"

   if [ "$#" != "1" ]; then
      echo "customize_polkit_color: specify operation"
      return
   fi
   mode="$1"

   if [ "$mode" = "set" ]; then
      [ -f "${POLKIT_SOURCE_FILE_PATH}" ] && \
      cp -p "${POLKIT_SOURCE_FILE_PATH}" "${POLKIT_TARGET_FILE_PATH}"

   else
      [ -f "${POLKIT_TARGET_PATH}" ] && \
      rm -rf "${POLKIT_TARGET_PATH}" >/dev/null 2>&1
   fi
}

customise_desktop_gsettings() {
   local mode
   local schema
   local key
   local value
   local last_schema=""
   local customised=""
   local managed

   if [ "$#" != "2" ]; then
      echo "customise_desktop_gsettings: specify operation"
      return
   fi

   mode="$1"
   managed="$2"

   [ "$GLIB_SCHEMAS" = "yes" ] || return

   #
   # For reset remove viewagent gschema override file.
   #
   if [ "$mode" = "reset" ]; then
      rm -f ${GLIB_SCHEMAS_PATH}/${VMWARE_SCHEMA_OVERRIDE}
      return
   fi

   while read cfg
   do
      schema=`echo $cfg | cut -f1 -d" "`

      gsettings list-schemas | grep "$schema" >/dev/null 2>&1
      if [ "$?" != "0" ]; then
         echo "customise_desktop_gsettings: Info: $schema not present"
         continue;
      fi

      key=`echo $cfg | cut -f2 -d" "`

      gsettings get $schema $key >/dev/null 2>&1
      if [ "$?" != "0" ]; then
         echo "customise_desktop_gsettings: Info: $schema $key not present"
         continue;
      fi

      case "$mode" in
      "set")
         if [ "$managed" = "true" ] &&
            [ "$key" = "suppress-restart-menuitem" -o "$key" = "suppress-shutdown-menuitem" ]; then
            echo "customise_desktop_gsettings: managed agent don't supress shut down and restart menu item"
            continue
         else
            value=`echo $cfg | cut -f3 -d" "`
         fi

         if [ "$last_schema" != "$schema" ]; then
            [ -n "$customised" ] && echo >> ${GLIB_SCHEMAS_PATH}/${VMWARE_SCHEMA_OVERRIDE}
            echo "[${schema}]" >> ${GLIB_SCHEMAS_PATH}/${VMWARE_SCHEMA_OVERRIDE}
            last_schema=$schema
            customised=yes
         fi
         echo "${key}=${value}" >> ${GLIB_SCHEMAS_PATH}/${VMWARE_SCHEMA_OVERRIDE}
         ;;
      esac
   done << !!!
com.canonical.indicator.session suppress-restart-menuitem true
com.canonical.indicator.session suppress-shutdown-menuitem true
org.gnome.desktop.lockdown disable-user-switching true
org.gnome.desktop.screensaver user-switch-enabled false
org.mate.lockdown disable-user-switching true
org.mate.screensaver user-switch-enabled false
org.gnome.settings-daemon.plugins.media-keys logout '<Control><Alt>Insert'
!!!

   if [ -n "$customised" ]; then
      glib-compile-schemas ${GLIB_SCHEMAS_PATH}/
      REQUEST_DESKTOP_RESTART=yes
   else
      echo "customise_desktop_gsettings: Info: no changes"
   fi
}


customise_desktop_gconf() {
   local mode
   local schema
   local key
   local type
   local value
   local customised=""
   local file
   local DBUS_SESSION_BUS_ADDRESS_ENV
   local managed

   if [ "$#" != "2" ]; then
      echo "customise_desktop_gconf: specify operation"
      return
   fi

   mode="$1"
   managed="$2"

   [ "$GCONFTOOL" = "yes" ] || return

   #
   # Here we need a wordaround to handle "GConf Error: Failed to contact configuration server;
   # some possible causes are that you need to enable TCP/IP networking for ORBit,or you have
   # stale NFS locks due to a system crash. See http://project.gnome.org/gconf/ for information.
   # (Details -1: Failed to get connection to session: Did you receive a reply. Possible causes
   # include: the remote application did not send a reply, the message bus security policy blocked
   # the reply, the reply timeout expired, or the network connection was broken.)" on SLED
   #
   if [ "$DISTRO_ID" = "$DISTRO_ID_SUSE" ]; then
      DBUS_SESSION_BUS_ADDRESS_ENV="$DBUS_SESSION_BUS_ADDRESS"
      unset DBUS_SESSION_BUS_ADDRESS
   fi

   while read cfg
   do
      schema=`echo $cfg | cut -f1 -d" "`

      gconftool-2 --dir-exists="$schema" >/dev/null 2>&1
      if [ "$?" != "0" ]; then
         echo "customise_desktop_gconf: Info: $schema not present"
         continue;
      fi

      key=`echo $cfg | cut -f2 -d" "`
      file=`echo $cfg |cut -f5 -d" "`
      if [ ! -d "/etc/gconf/${file}" ]; then
         echo "folder /etc/gconf/${file} don't exist"
         continue;
      fi

      gconftool-2 --get "/schemas${schema}/${key}" 2>&1 | \
         grep "^No value set for" >/dev/null 2>&1
      if [ "$?" = "0" ]; then
         echo "customise_desktop_gconf: Info: $schema/$key not present"
         continue;
      fi

      if [ "$managed" = "true" ] &&
         [ "$key" = "suppress-restart-menuitem" -o "$key" = "suppress-shutdown-menuitem" ]; then
         echo "managed agent don't supress restart and shut down menu item"
         continue;
      fi

      case "$mode" in
      "set")
         type=`echo $cfg | cut -f3 -d" "`
         value=`echo $cfg | cut -f4 -d" "`

         gconftool-2 --direct --config-source xml:readwrite:/etc/gconf/${file} --type $type --set ${schema}/${key} $value >/dev/null 2>&1
         [ "$?" = "0" ] && customised=yes
         ;;
      "reset")
         gconftool-2 --direct --config-source xml:readwrite:/etc/gconf/${file} --unset ${schema}/${key} >/dev/null 2>&1
         [ "$?" = "0" ] && customised=yes
         ;;
      esac
   done << !!!
/apps/gnome-screensaver user_switch_enabled bool false gconf.xml.mandatory
/apps/gnome-power-manager/general can_hibernate bool false gconf.xml.mandatory
/apps/gnome_settings_daemon/keybindings power string <Control><Alt>Insert gconf.xml.defaults
/apps/indicator-session suppress_shutdown_menuitem bool true gconf.xml.mandatory
/apps/indicator-session suppress_restart_menuitem bool true gconf.xml.mandatory
/desktop/gnome/lockdown disable_user_switching bool true gconf.xml.mandatory
!!!

   #
   # Shutting down gconfd should trigger a restart / reload
   #
   if [ -n "$customised" ]; then
      gconftool-2 --shutdown
      REQUEST_DESKTOP_RESTART=yes
   else
      echo "customise_desktop_gconf: Info: no changes"
   fi

   if [ "$DISTRO_ID" = "$DISTRO_ID_SUSE" ]; then
      export "DBUS_SESSION_BUS_ADDRESS=$DBUS_SESSION_BUS_ADDRESS_ENV"
   fi
}


customise_desktop_polkit() {
   local mode
   local policy
   local description
   local action
   local last_description=""
   local last_identity=""
   local customised=""
   local pkfile=${POLICYKIT_PATH}/${VIEWAGENT_POLICYKIT}
   local managed

   if [ "$#" != "2" ]; then
      echo "customise_desktop_polkit: specify operation"
      return
   fi

   mode="$1"
   managed="$2"

   [ "$POLICYKIT" = "yes" ] || return

   #
   # For reset remove viewagent mandatory policykit file.
   #
   if [ "$mode" = "reset" ]; then
      rm -f ${pkfile}
      return
   fi

   while read policy
   do
      case "$mode" in
      "set")
         description=`echo $policy | cut -f1 -d";"`
         action=`echo $policy | cut -f2 -d";"`
         identity=`echo $policy | cut -f3 -d";"`

         if [ "$managed" = "true" ] &&
            [ "$description" = "Disable shutdown" -o "$description" = "Disable restart" ]; then
            echo "customise_desktop_polkit: managed agent don't supress shut down and restart menu item"
            continue
         fi

         pkaction 2>&1 | grep ${action} >/dev/null 2>&1
         if [ "$?" != "0" ]; then
            echo "customise_desktop_polkit: Info: Action $action not registered"
            continue
         fi

         if [ -z "$customised" ]; then
            > ${pkfile}
            chown root:root ${pkfile}
            chmod 644 ${pkfile}
         fi

         if [ "$last_description" != "$description" ] || [ "$last_identity" != "$identity" ]; then
            if [ -n "$customised" ]; then
               echo >> ${pkfile}
               echo "ResultAny=no" >> ${pkfile}
               echo "ResultInactive=no" >> ${pkfile}
               echo "ResultActive=no" >> ${pkfile}
               echo >> ${pkfile}
            fi
            echo "[${description}]" >> ${pkfile}
            echo "Identity=${identity}:*" >> ${pkfile}
            echo -n "Action=${action}" >> ${pkfile}
            last_description=$description
            last_identity=$identity
            customised=yes
         else
            echo -n ";${action}" >> ${pkfile}
         fi

         ;;
      esac
   done << !!!
Disable suspend;org.freedesktop.upower.suspend;unix-user
Disable suspend;org.freedesktop.login1.suspend;unix-user
Disable suspend;org.freedesktop.login1.suspend-multiple-sessions;unix-user
Disable suspend;org.freedesktop.devicekit.power.suspend;unix-group
Disable suspend;org.freedesktop.upower.hibernate;unix-user
Disable suspend;org.freedesktop.login1.hibernate;unix-user
Disable suspend;org.freedesktop.login1.hibernate-multiple-sessions;unix-user
Disable suspend;org.freedesktop.devicekit.power.hibernate;unix-group
Disable shutdown;org.freedesktop.consolekit.system.stop;unix-group
Disable shutdown;org.freedesktop.consolekit.system.stop-multiple-users;unix-group
Disable shutdown;org.freedesktop.login1.power-off;unix-group
Disable shutdown;org.freedesktop.login1.power-off-multiple-sessions;unix-group
Disable restart;org.freedesktop.consolekit.system.restart;unix-group
Disable restart;org.freedesktop.consolekit.system.restart-multiple-users;unix-group
Disable restart;org.freedesktop.login1.reboot;unix-group
Disable restart;org.freedesktop.login1.reboot-multiple-sessions;unix-group
Allow colord;org.freedesktop.color-manager.create-device;unix-user
Allow colord;org.freedesktop.color-manager.create-profile;unix-user
Allow colord;org.freedesktop.color-manager.delete-device;unix-user
Allow colord;org.freedesktop.color-manager.delete-profile;unix-user
Allow colord;org.freedesktop.color-manager.modify-device;unix-user
Allow colord;org.freedesktop.color-manager.modify-profile;unix-user
!!!

   if [ -n "$customised" ]; then
      echo >> ${pkfile}
      echo "ResultAny=no" >> ${pkfile}
      echo "ResultInactive=no" >> ${pkfile}
      echo "ResultActive=no" >> ${pkfile}
      REQUEST_DESKTOP_RESTART=yes
   else
      echo "customise_desktop_polkit: Info: no changes"
   fi
}


customise_desktop_pam() {
   local mode
   local managed

   if [ "$#" != "2" ]; then
      echo "customise_desktop_pam: specify operation"
      return
   fi

   mode="$1"
   managed="$2"

   [ -d ${PAM_PATH} ] || return

   while read file
   do
      case "$mode" in
      "set")
         if [ "$managed" = "true" ] &&
            [ "$file" != "halt" ]; then
            echo "customise_desktop_pam: don't suppress shutdown restart poweroff item"
            continue;
         fi

         if [ -f ${PAM_PATH}/$file ]; then
            cp -p ${PAM_PATH}/$file ${PAM_PATH}/$file.vmw
            if [ "$?" != "0" ]; then
               echo "Failed to move ${PAM_PATH}/$file"
            else
               sed -i '1i\#vmware linux desktop disable '$file'\nauth   include  system-auth' ${PAM_PATH}/$file
            fi
         fi
         ;;
      "reset")
         if [ -f ${PAM_PATH}/$file.vmw ]; then
            mv ${PAM_PATH}/$file.vmw ${PAM_PATH}/$file
            if [ "$?" != "0" ]; then
               echo "Failed to restore ${PAM_PATH}/$file"
            fi
         fi
         ;;
      esac
   done << !!!
halt
poweroff
shutdown
reboot
!!!
}

customise_desktop_systemd() {
   local mode="$1"
   local SYSTEMD_CONFIG_FILE="/etc/systemd/system.conf"
   local SYSTEMD_BACKUP_DIR="${VIEWAGENT_PATH}/systemd"

   if [ "$SERVICE_MGR" = "${SERVICE_MGR_SYSTEMD}" ]; then
      case "$mode" in
      "set")
         if [ -f "$SYSTEMD_CONFIG_FILE" ]; then
            if [ ! -d ${SYSTEMD_BACKUP_DIR} ]; then
               mkdir -p ${SYSTEMD_BACKUP_DIR}
               chown -R root:root ${SYSTEMD_BACKUP_DIR}
            fi
            cp -fp $SYSTEMD_CONFIG_FILE $SYSTEMD_BACKUP_DIR/systemd.conf.backup
            if [ "$SINGLETON_MODE" = "false" ]; then
               sed -i 's^DefaultLimitNOFILE^#DefaultLimitNOFILE^g' $SYSTEMD_CONFIG_FILE
               sed -i '$a DefaultLimitNOFILE=65535' $SYSTEMD_CONFIG_FILE
            fi
         fi
         ;;
       "reset")
         if [ -f "$SYSTEMD_BACKUP_DIR/systemd.conf.backup" ]; then
            install $SYSTEMD_BACKUP_DIR/systemd.conf.backup $SYSTEMD_CONFIG_FILE
            rm -rf $SYSTEMD_BACKUP_DIR
         fi
         ;;
      *)
         echo "customise_desktop_systemd: specify set or reset"
         ;;
      esac
   fi
}

customise_desktop_pulseaudio() {
   local mode="$1"
   local XDG_AUTOSTART_PATH="/etc/xdg/autostart"
   local PULSE_CONFIG_PATH="/etc/pulse"

   case "$mode" in
   "set")
      if [ -f "$XDG_AUTOSTART_PATH/pulseaudio.desktop" ]; then
         cp -fp $XDG_AUTOSTART_PATH/pulseaudio.desktop $XDG_AUTOSTART_PATH/pulseaudio.desktop.orig
         sed -i "s/Exec=start-pulseaudio-x11/Exec=\/usr\/lib\/vmware\/viewagent\/resources\/pulseaudio\/start-pulseaudio-vmw/g" $XDG_AUTOSTART_PATH/pulseaudio.desktop
         if [ "${SINGLETON_MODE}" = "false" ]; then
            if [ "$DISTRO_ID" = "${DISTRO_ID_UBUNTU}" ] && [ "${OS_MAJOR}" = "18" ]; then
               sed -i "s/X-GNOME-Autostart-Phase=Initialization/X-GNOME-Autostart-Phase=EarlyInitialization/g" $XDG_AUTOSTART_PATH/pulseaudio.desktop
            fi
         fi

         chmod 755 $VIEWAGENT_PATH/resources/pulseaudio/start-pulseaudio-vmw
      fi

      # Disable bluetooth in pulseaudio
      if [ -f "$PULSE_CONFIG_PATH/default.pa" ]; then
         cp -fp $PULSE_CONFIG_PATH/default.pa $PULSE_CONFIG_PATH/default.pa.BeforeVMwareHorizonInstall
         sed -i '/^load-module module-bluetooth-policy/ s/^/#/' $PULSE_CONFIG_PATH/default.pa
         sed -i '/^load-module module-bluetooth-discover/ s/^/#/' $PULSE_CONFIG_PATH/default.pa
      fi
      ;;
   "reset")
      if [ -f "$XDG_AUTOSTART_PATH/pulseaudio.desktop.orig" ]; then
         mv -f $XDG_AUTOSTART_PATH/pulseaudio.desktop.orig $XDG_AUTOSTART_PATH/pulseaudio.desktop
      fi

      if [ -f "$PULSE_CONFIG_PATH/default.pa.BeforeVMwareHorizonInstall" ]; then
         mv -f $PULSE_CONFIG_PATH/default.pa.BeforeVMwareHorizonInstall $PULSE_CONFIG_PATH/default.pa
      fi
      ;;
   *)
      echo "customise_desktop_pulseaudio: specify set or reset"
      ;;
   esac
}

customise_desktop_kde() {
   local mode
   local LOCK_SCREEN_FILE_PATH="/usr/share/kde4/apps/ksmserver/screenlocker/org.kde.passworddialog/contents/ui/Greeter.qml"
   local LOCK_SCREEN_FILE_PATH_ORG="/usr/share/kde4/apps/ksmserver/screenlocker/org.kde.passworddialog/contents/ui/Greeter.qml.org"
   local SWITCH_USER_FILE_PATH="/usr/share/kde4/services/plasma-runner-sessions.desktop"
   local SWITCH_USER_FILE_PATH_ORG="/usr/share/kde4/services/plasma-runner-sessions.desktop.org"
   local LOCK_SCREEN_FILE_PATH_PLASMA_5="/usr/share/plasma/look-and-feel/org.kde.breeze.desktop/contents/lockscreen/MainBlock.qml"
   local LOCK_SCREEN_FILE_PATH_PLASMA_5_ORG="/usr/share/plasma/look-and-feel/org.kde.breeze.desktop/contents/lockscreen/MainBlock.qml.org"
   local LOCK_SCREEN_UI_FILE_PATH_PLASMA_5="/usr/share/plasma/look-and-feel/org.kde.breeze.desktop/contents/lockscreen/LockScreenUi.qml"
   local LOCK_SCREEN_UI_FILE_PATH_PLASMA_5_ORG="/usr/share/plasma/look-and-feel/org.kde.breeze.desktop/contents/lockscreen/LockScreenUi.qml.org"
   local SWITCH_USER_FILE_PATH_PLASMA_5="/usr/share/plasma/look-and-feel/org.kde.breeze.desktop/contents/userswitcher/UserSwitcher.qml"
   local SWITCH_USER_FILE_PATH_PLASMA_5_ORG="/usr/share/plasma/look-and-feel/org.kde.breeze.desktop/contents/userswitcher/UserSwitcher.qml.org"
   local SWITCH_USER_IN_UNLOCK_SLED11SP4="/etc/dbus-1/system.d/ConsoleKit.conf"
   local SWITCH_USER_IN_UNLOCK_SLED11SP4_ORG="/etc/dbus-1/system.d/ConsoleKit.conf.org"

   if [ "$#" != "1" ]; then
      echo "customise_desktop_kde: specify operation"
      return
   fi

   mode="$1"

   # sled11sp4 kde hide switch user button in unlock pannel
   if [ "$DISTRO_ID" = "$DISTRO_ID_SUSE" ] && [ "$OS_MAJOR" = "11" ]; then
      case "$mode" in
        "set")
           if [ -f $SWITCH_USER_IN_UNLOCK_SLED11SP4 ]; then
              mv -f $SWITCH_USER_IN_UNLOCK_SLED11SP4 $SWITCH_USER_IN_UNLOCK_SLED11SP4_ORG
              cp -fp $VIEWAGENT_PATH/resource/kde/ConsoleKit.conf $SWITCH_USER_IN_UNLOCK_SLED11SP4
           fi
           ;;
        "reset")
           if [ -f $SWITCH_USER_IN_UNLOCK_SLED11SP4_ORG ]; then
              mv -f $SWITCH_USER_IN_UNLOCK_SLED11SP4_ORG $SWITCH_USER_IN_UNLOCK_SLED11SP4
           fi
           ;;
        *)
           echo "customise_desktop_kde: specify set or reset."
           ;;
     esac
   fi

   case "$mode" in
   "set")
      # Disable Switch User in the lock screen
      if [ -f "$LOCK_SCREEN_FILE_PATH" ]; then
         # Backup system original file
         mv -f $LOCK_SCREEN_FILE_PATH $LOCK_SCREEN_FILE_PATH_ORG

         # Deploy new file
         cp -fp $VIEWAGENT_PATH/resources/kde/Greeter.qml $LOCK_SCREEN_FILE_PATH
      fi
      # Disable Switch User in the desktop
      if [ -f "$SWITCH_USER_FILE_PATH" ]; then
         # Backup system original file
         mv -f $SWITCH_USER_FILE_PATH $SWITCH_USER_FILE_PATH_ORG

      fi

      # Disable Switch User in lock screen for plasma 5
      if [ "$DISTRO_ID" = "$DISTRO_ID_UBUNTU" ] && [ ${OS_MAJOR} = "18" ]; then
         if [ -f "$LOCK_SCREEN_UI_FILE_PATH_PLASMA_5" ]; then
            # Backup system original file
            mv -f $LOCK_SCREEN_UI_FILE_PATH_PLASMA_5 $LOCK_SCREEN_UI_FILE_PATH_PLASMA_5_ORG

            # Deploy new file
            cp -fp $VIEWAGENT_PATH/resources/kde/LockScreenUi.qml $LOCK_SCREEN_UI_FILE_PATH_PLASMA_5
         fi
      else
         if [ -f "$LOCK_SCREEN_FILE_PATH_PLASMA_5" ]; then
            # Backup system original file
            mv -f $LOCK_SCREEN_FILE_PATH_PLASMA_5 $LOCK_SCREEN_FILE_PATH_PLASMA_5_ORG

            # Deploy new file
            cp -fp $VIEWAGENT_PATH/resources/kde/MainBlock.qml $LOCK_SCREEN_FILE_PATH_PLASMA_5
         fi
      fi

      # Disable New Session dialog for plasma 5
      if [ -f "$SWITCH_USER_FILE_PATH_PLASMA_5" ]; then
         # Backup system original file
         mv -f $SWITCH_USER_FILE_PATH_PLASMA_5 $SWITCH_USER_FILE_PATH_PLASMA_5_ORG
      fi
      ;;
   "reset")
      # Restore the system original file
      if [ -f "$LOCK_SCREEN_FILE_PATH_ORG" ]; then
         mv -f $LOCK_SCREEN_FILE_PATH_ORG $LOCK_SCREEN_FILE_PATH
      fi
      if [ -f "$SWITCH_USER_FILE_PATH_ORG" ]; then
         # Backup system original file
         mv -f $SWITCH_USER_FILE_PATH_ORG $SWITCH_USER_FILE_PATH

      fi

      if [ -f "$LOCK_SCREEN_FILE_PATH_PLASMA_5_ORG" ]; then
         mv -f $LOCK_SCREEN_FILE_PATH_PLASMA_5_ORG $LOCK_SCREEN_FILE_PATH_PLASMA_5
      fi

      if [ -f "$SWITCH_USER_FILE_PATH_PLASMA_5_ORG" ]; then
         mv -f $SWITCH_USER_FILE_PATH_PLASMA_5_ORG $SWITCH_USER_FILE_PATH_PLASMA_5
      fi
      ;;
   *)
      echo "customise_desktop_kde: specify set or reset"
      ;;
   esac
}

# Disable the name list in gdm greeter.
customise_desktop_rhel_greeter() {
   if [ "$#" != "1" ]; then
      echo "customise_desktop_rhel_greeter: specify operation"
      return
   fi

   mode="$1"
   case "$mode" in
   "set")
      if [ "${SINGLETON_MODE}" = "false" ]; then
         if [ ! -f "/etc/dconf/profile/gdm" ]; then
            echo -e "user-db:user\nsystem-db:gdm\nfile-db:/usr/share/gdm/greeter-dconf-defaults" > /etc/dconf/profile/gdm
            chmod 644 /etc/dconf/profile/gdm
         fi

         if [ ! -d "/etc/dconf/db/gdm.d" ]; then
            mkdir -m 755 /etc/dconf/db/gdm.d
         fi

         echo -e "[org/gnome/login-screen]\ndisable-user-list=true" > /etc/dconf/db/gdm.d/01-vmw-gdm-settings
         chmod 644 /etc/dconf/db/gdm.d/01-vmw-gdm-settings
         dconf update
      fi
      ;;
   "reset")
      if [ -f "/etc/dconf/db/gdm.d/01-vmw-gdm-settings" ]; then
         rm -f /etc/dconf/db/gdm.d/01-vmw-gdm-settings
         dconf update
      fi
      ;;
   *)
      echo "customise_desktop_rhel_greeter: specify set or reset"
      ;;
   esac
}


customise_desktop_NeoKylin() {
   local managed

   if [ "$#" != "1" ]; then
      echo "customise_desktop_NeoKylin: specify operation"
      return
   fi

   managed="$1"

   # Disable restart/shutdown/switchuser/suspend in the login UI of NeoKylin
   if [ "$managed" != "true" ]; then
      sed -i 's/RESTART_ENABLE\ *=\ *true/RESTART_ENABLE\ =\ false/g' /usr/share/gnome-shell/js/misc/config.js
      sed -i 's/SHUTDOWN_ENABLE\ *=\ *true/SHUTDOWN_ENABLE\ =\ false/g' /usr/share/gnome-shell/js/misc/config.js
   fi
   sed -i 's/OTHER_USER_ENABLE\ *=\ *\(true\|false\)/OTHER_USER_ENABLE\ =\ true/g' /usr/share/gnome-shell/js/misc/config.js
   sed -i 's/SUSPEND_ENABLE\ *=\ *true/SUSPEND_ENABLE\ =\ false/g' /usr/share/gnome-shell/js/misc/config.js

   # Disable restart/shutdown/switchuser/hibernate/suspend in the start menu of NeoKylin
   if [ "$managed" != "true" ]; then
      sed -i 's/shutdown\ *=\ *enable/shutdown=disable/g' /etc/mate-session-dialog.conf
      sed -i 's/reboot\ *=\ *enable/reboot=disable/g' /etc/mate-session-dialog.conf
   fi
   sed -i 's/switch_user\ *=\ *enable/switch_user=disable/g' /etc/mate-session-dialog.conf
   sed -i 's/hibernate\ *=\ *enable/hibernate=disable/g' /etc/mate-session-dialog.conf
   sed -i 's/suspend\ *=\ *enable/suspend=disable/g' /etc/mate-session-dialog.conf
}

customise_desktop_SLED() {
   local mode="$1"
   local managed="$2"
   local GDM_GREETER_LOGIN_WINDOW_GLAD="/usr/share/gdm/gdm-greeter-login-window.glade"
   local GDM_GREETER_LOGIN_WINDOW_GLAD_ORIG="$VIEWAGENT_PATH/resources/suse/gdm-greeter-login-window.glade.origin"
   local GNOME_SESSION_SHUTDOWN_DESKTOP="/usr/share/applications/gnome-session-shutdown.desktop"
   local GNOME_SESSION_SHUTDOWN_DESKTOP_ORIG="/usr/share/applications/gnome-session-shutdown.desktop.orig"
   local YAST2_X11_DESKTOP="/usr/share/applications/YaST2/x11.desktop"
   local YAST2_X11_DESKTOP_ORIG="/usr/share/applications/YaST2/x11.desktop.orig"
   local VIEW_ETC="/etc/vmware"

   customise_desktop_pulseaudio $mode

   case "$mode" in
   "set")
      # fix the reset login window failed issue
      mv -f $GDM_GREETER_LOGIN_WINDOW_GLAD_ORIG $GDM_GREETER_LOGIN_WINDOW_GLAD

      # fix the reset start menu issue
      if [ -f $GNOME_SESSION_SHUTDOWN_DESKTOP_ORIG ]; then
         mv -f $GNOME_SESSION_SHUTDOWN_DESKTOP_ORIG $GNOME_SESSION_SHUTDOWN_DESKTOP
      fi

      # Disable restart/shutdown in the login UI for unmanaged agent
      if [ "$managed" != "true" ] &&
         [ -f "$GDM_GREETER_LOGIN_WINDOW_GLAD" ]; then
         # Backup system original file
         mv -f $GDM_GREETER_LOGIN_WINDOW_GLAD $VIEW_ETC/gdm-greeter-login-window.glade.orig

         # Deploy new file
         cp -fp $VIEWAGENT_PATH/resources/suse/gdm-greeter-login-window.glade $GDM_GREETER_LOGIN_WINDOW_GLAD
      fi

      # Disable shutdown/restart/hibernate/suspend for unmanaged agent in the  start menu of SLED
      if [ "$managed" != "true" ] &&
         [ -f "$GNOME_SESSION_SHUTDOWN_DESKTOP" ]; then
         mv -f $GNOME_SESSION_SHUTDOWN_DESKTOP $GNOME_SESSION_SHUTDOWN_DESKTOP_ORIG
      fi

      # Disable feature "Graphics Card and Monitor" in Control Center
      if [ -f "$YAST2_X11_DESKTOP" ]; then
         mv -f $YAST2_X11_DESKTOP $YAST2_X11_DESKTOP_ORIG
      fi
      ;;
   "reset")
      # Enable restart/shutdown in the login UI
      if [ "$managed" != "true" ] &&
         [ -f "$VIEW_ETC/gdm-greeter-login-window.glade.orig" ]; then
         mv -f $VIEW_ETC/gdm-greeter-login-window.glade.orig $GDM_GREETER_LOGIN_WINDOW_GLAD
      fi

      # Enable shutdown/restart/hibernate/suspend in the start menu of SLED
      if [ "$managed" != "true" ] &&
         [ -f "$GNOME_SESSION_SHUTDOWN_DESKTOP_ORIG" ]; then
         mv -f $GNOME_SESSION_SHUTDOWN_DESKTOP_ORIG $GNOME_SESSION_SHUTDOWN_DESKTOP
      fi

      # Enable feature "Graphics Card and Monitor" in Control Center
      if [ -f "$YAST2_X11_DESKTOP_ORIG" ]; then
         mv -f $YAST2_X11_DESKTOP_ORIG $YAST2_X11_DESKTOP
      fi
      ;;
   *)
      echo "customise_desktop_SLED: specify set or reset"
      ;;
   esac
}

customise_desktop_SLES() {
   local mode="$1"
   local SYSTEM_JS="/usr/share/gnome-shell/js/ui/status/system.js"
   local SYSTEM_JS_ORG="/usr/share/gnome-shell/js/ui/status/system.js.org"
   local GNOME_POWER_PANEL_DESKTOP="/usr/share/applications/gnome-power-panel.desktop"
   local GNOME_POWER_PANEL_DESKTOP_ORG="/usr/share/applications/gnome-power-panel.desktop.org"
   local ENABLE_SWITCH_USER="let visible = allowSwitch && multiUser"
   local DISABLE_SWITCH_USER="let visible = false"

   case "$mode" in
   "set")
      # Disable Switch User in the desktop
      if [ -f "$SYSTEM_JS" ]; then
         # Backup system original file
         mv -f $SYSTEM_JS $SYSTEM_JS_ORG

         # Deploy new file
         sed "s/$ENABLE_SWITCH_USER/$DISABLE_SWITCH_USER/" $SYSTEM_JS_ORG >> $SYSTEM_JS
      fi
      # Disable Automatic suspend button in the control center
      if [ -f "$GNOME_POWER_PANEL_DESKTOP" ]; then
         mv -f $GNOME_POWER_PANEL_DESKTOP $GNOME_POWER_PANEL_DESKTOP_ORG
      fi
      ;;
   "reset")
      # Enable Switch User in the desktop
      if [ -f "$SYSTEM_JS_ORG" ]; then
         mv -f $SYSTEM_JS_ORG $SYSTEM_JS
      fi
      # Enable Automatic suspend button in the control center
      if [ -f "$GNOME_POWER_PANEL_DESKTOP_ORG" ]; then
         mv -f $GNOME_POWER_PANEL_DESKTOP_ORG $GNOME_POWER_PANEL_DESKTOP
      fi
      ;;
   *)
      echo "customise_desktop_SLES: specify set or reset"
      ;;
   esac
}

customise_desktop_RHEL() {
   local mode="$1"

   # Disable Authentication popup message at login
   customise_desktop_rhel_popup $mode

   if [ "${OS_MAJOR}" = "8" -o "${OS_MAJOR}" = "7" ]; then
      customise_desktop_rhel_greeter $mode
      customise_desktop_pulseaudio $mode
   fi
}

#
# Desktop session command functions
#

parse_X_args() {
   local ret=1

   shift

   while [ "$#" -gt 0 ]
   do
      if [ "`expr substr $1 1 1`" = ":" ]; then
         export DISPLAY="$1"
      elif [ "$1" = "-auth" ]; then
         shift
         export XAUTHORITY="$1"
      fi
      shift
   done

   #
   # Workaround for an issue in the KUbuntu16.04.
   # On the KUbuntu16.04, we can't get the DISPLAY from the Xorg process information:
   # /usr/lib/xorg/Xorg -nolisten tcp -auth /var/run/sddm/{ca341de6-b935-42d3-9b63-fb48bb9ff7aa} -background none -noreset -displayfd 18 vt7
   # As the linux agent only support the single session, we can get the DISPLAY from the domain socket in the directory "/tmp/.X11-unix/".
   # https://unix.stackexchange.com/questions/196677/what-is-tmp-x11-unix
   # https://unix.stackexchange.com/questions/17255/is-there-a-command-to-list-all-open-displays-on-a-machine
   # https://stackoverflow.com/questions/11367354/obtaining-list-of-all-xorg-displays
   #
   if [ -z "$DISPLAY" ]; then
      xUnixSockets=$(ls /tmp/.X11-unix/ | egrep ^X[0-9]+)
      for xSocket in $xUnixSockets
      do
         xSocket="${xSocket#X}"
         if [ -n "$xSocket" ]; then
            export DISPLAY=":$xSocket"
            break
         fi
      done
   fi

   [ -n "$DISPLAY" ] && [ -n "$XAUTHORITY" ] && ret=0

   return $ret
}


# Deprecated. References: CLN #3465844, #3541098
configure_xinput() {
   local val

   if [ "$#" -gt 0 ]; then
      case $1 in
      disable)
         val=0;;
      enable)
         val=1;;
      *)
         echo "Failed to configure xinput device: unknown the params $1"
         return
         ;;
      esac
   fi

   #
   # Disable the physics input driver via xinput
   #
   while read devname
   do
      xinput set-prop "$devname" 'Device Enabled' $val >/dev/null 2>&1
   done << !!!
`xinput --list --name-only| grep -v "^Virtual core"`
!!!
}


configure_xorg() {
   local res="1280x1024"

   [ "$#" -gt 0 ] && res=$1

   #
   # Switch to the requested screen resolution, where available.
   # TODO: Add temporary mode to screen to handle arbitrary resolutions.
   #
   xrandr --output Virtual1 --mode $res >/dev/null 2>&1
}


disable_xorg_VTSwitch()
{
   local xorg_conf="$VIEWAGENT_XORG_CONF_FILE"

   if [ $# -gt 0 -a -f "$1" ]; then
       xorg_conf="$1"
   fi

   if [ ! -f $xorg_conf ]; then
      echo "disable_xorg_VTSwith: $xorg_conf doesn't exist"
      return
   fi

   nVT=`grep \"DontVTSwitch\" $xorg_conf |wc -l`
   if [ $nVT -ge 1 ]; then
      sed '/\"DontVTSwitch\"/ c\     Option    \"DontVTSwitch\"    \"true\"' $xorg_conf > $xorg_conf.tmp
      mv -f $xorg_conf.tmp $xorg_conf
      return
   fi

   nSF=`grep \"ServerFlags\" $xorg_conf | wc -l`
   if [ $nSF -ge 1 ]; then
      sed '/\"ServerFlags\"/ a\     Option    \"DontVTSwitch\"    \"true\"' $xorg_conf > $xorg_conf.tmp
      mv -f $xorg_conf.tmp $xorg_conf
      return
   fi

   echo "Section    \"ServerFlags\"" >> $xorg_conf
   echo "    Option    \"DontVTSwitch\"    \"true\"" >> $xorg_conf
   echo "EndSection"  >> $xorg_conf
}


#
# Remove the "Device0" section
#
remove_device_section()
{
   local delflag=0
   local ifs=$IFS
   local line
   local lastline
   local xorg_conf=$1

   IFS=
   while read line
   do
      if echo $line |grep -q "\"Device0\"" && echo $line|grep -q "Identifier"; then
         delflag=1
         continue
      elif echo $line |grep -q "EndSection"; then
         if [ $delflag -eq 1 ]; then
            lastline=
            delflag=0
            continue
         fi
      fi
      if [ $delflag -eq 0 ] && [ -n "$lastline" ]; then
         echo $lastline >> $xorg_conf.tmp
      fi
      lastline="$line"
   done < $xorg_conf
   if [ $delflag -eq 0 ] && [ -n "$lastline" ]; then
         echo $lastline >> $xorg_conf.tmp
   fi
   IFS=$ifs
   mv -f $xorg_conf.tmp $xorg_conf
}


config_xorg_vDGA()
{
   local mode=$1
   local pci_entry
   local pci_bus
   local pci_device
   local pci_function
   local xorg_conf="$VIEWAGENT_XORG_CONF_FILE"
   local edid_path="/usr/lib/vmware/viewagent/resources/X11/edid.bin"

   if [ ! -f $xorg_conf ]; then
      echo "config_xorg_vDGA: $xorg_conf doesn't exist"
      return
   fi

   #
   # Remove "Device0" section from /etc/X11/xorg.conf
   #
   if [ "$mode" = "install" ] && \
      grep -q "/etc/X11/edid.bin\|Driver.*\"nvidia\"" $XORG_CONF_DIR/xorg.conf; then
      cp -fp $XORG_CONF_DIR/xorg.conf $XORG_CONF_DIR/xorg.conf.vmware_backup_vgpu
      remove_device_section $XORG_CONF_DIR/xorg.conf
   fi

   nNVDIA=`lspci -v |grep "Kernel driver in use: nvidia" |wc -l`
   nEDID=`grep "/etc/X11/edid.bin" $xorg_conf |wc -l`

   #
   # If the content of xorg.conf is accordant with lspci,
   # We don't need config the xorg.conf again.
   #
   if [ $nNVDIA -eq $nEDID ]; then
      #
      # If install viewagent in vDGA mode, we force to update
      # the xorg.conf file.
      #
      if [ "$mode" != "install" ] || [ $nNVDIA -eq 0 ]; then
         return
      fi
   fi

   # Remove the old "Device0" section
   remove_device_section $xorg_conf

   #
   # If the VM has vDGA driver, add the vDGA configure to xorg.conf
   #
   if [ $nNVDIA -ge 1 ]; then
      if [ ! -e "$XORG_CONF_DIR/edid.bin" ]; then
         cp -fp $edid_path $XORG_CONF_DIR/edid.bin
      else
         echo "configure_xorg_vDGA: $XORG_CONF_DIR/edid.bin have exist."
      fi
      pci_entry=`lspci |grep -m1 "NVIDIA" |cut -d" " -f1`
      pci_bus=`echo $pci_entry |awk -F'[:.]' '{print $1}'`
      pci_device=`echo $pci_entry |awk -F'[:.]' '{print $2}'`
      pci_function=`echo $pci_entry |awk -F'[:.]' '{print $3}'`
      pci_entry=`printf "%d:%d:%d" 0x${pci_bus} 0x${pci_device} 0x${pci_function}`
      echo "Section    \"Device\"" >> $xorg_conf
      echo "    Identifier \"Device0\"" >> $xorg_conf
      echo "    Driver     \"nvidia\"" >> $xorg_conf
      echo "    VendorName \"NVIDIA Corporation\"" >> $xorg_conf
      echo "    BusID      \"PCI:$pci_entry\"" >> $xorg_conf
      echo "    Option     \"IgnoreDisplayDevices\" \"CRT-0\"" >> $xorg_conf
      echo "    Option     \"NoFlip\" \"TRUE\"" >> $xorg_conf
      echo "    Option     \"ConnectedMonitor\" \"DFP-0, DFP-1, DFP-2, DFP-3\"" >> $xorg_conf
      echo "    Option     \"MetaModes\" \"DVI-D-0:640x480\"" >> $xorg_conf
      echo "    Option     \"CustomEDID\" \"DFP-0:/etc/X11/edid.bin; DFP-1:/etc/X11/edid.bin; DFP-2:/etc/X11/edid.bin; DFP-3:/etc/X11/edid.bin\"" >> $xorg_conf
      echo "    Option     \"ModeValidation\" \"NoVesaModes,NoPredefinedModes,NoXserverModes,NoMaxSizeCheck,NoEdidMaxPClkCheck,NoHorizSyncCheck,NoVertRefreshCheck,AllowNonEdidModes\"" >> $xorg_conf
      echo "EndSection"  >> $xorg_conf
   fi
}


generate_xorg_conf()
{
   if [ -f "$XORG_CONF_DIR/xorg.conf" ]; then
      cp -fp $XORG_CONF_DIR/xorg.conf $VIEWAGENT_XORG_CONF_FILE
   else
      > $VIEWAGENT_XORG_CONF_FILE
   fi

   disable_xorg_VTSwitch
}


generate_xorg_normal_conf()
{
   local xorg_conf="$VIEWAGENT_XORG_CONF_FILE"

   generate_xorg_conf

   nXV=`grep \"XVideo\" $xorg_conf|wc -l`
   if [ $nXV -ge 1 ]; then
      sed '/\"XVideo\"/ c\     Option    \"XVideo\"    \"Disable\"' $xorg_conf > $xorg_conf.tmp
      mv -f $xorg_conf.tmp $xorg_conf
      return
   fi

   nEx=`grep \"Extensions\" $xorg_conf| wc -l`
   if [ $nEx -ge 1 ]; then
      sed '/\"Extensions\"/ a\     Option    \"XVideo\"    \"Disable\"' $xorg_conf > $xorg_conf.tmp
      mv -f $xorg_conf.tmp $xorg_conf
      return
   fi

   echo "Section    \"Extensions\"" >> $xorg_conf
   echo "     Option    \"XVideo\"    \"Disable\"" >> $xorg_conf
   echo "EndSection" >> $xorg_conf
}


generate_xorg_vgpu_conf()
{
   local mode=$1

   generate_xorg_conf
   config_xorg_vDGA $mode
}


generate_xorg_vmwmem_conf()
{
   echo "Section    \"Device\"" > $VIEWAGENT_XORG_CONF_FILE
   echo "    Identifier \"Device0\"" >> $VIEWAGENT_XORG_CONF_FILE
   echo "    Driver     \"vmwmem\"" >> $VIEWAGENT_XORG_CONF_FILE
   echo "EndSection"  >> $VIEWAGENT_XORG_CONF_FILE

   disable_xorg_VTSwitch
}


configure_xorg_conf()
{
   umask 022
   local console_xorg_conf="$XORG_CONF_DIR/xorg.conf"
   local singleton=$1
   local mode=$2
   local vDGA_flag=0

   lspci -v |grep "Kernel driver in use: nvidia"
   if [ "$?" = "0" ]; then
      vDGA_flag=1
   fi

   identify_xserver_vmwmem_enabled
   case "$mode" in
   "reset")
      if [ -f /usr/lib64/xorg/modules/drivers/vmwmem_drv.so ]; then
         unlink /usr/lib64/xorg/modules/drivers/vmwmem_drv.so
      fi

      if [ -f /usr/lib/xorg/modules/drivers/vmwmem_drv.so ]; then
         unlink /usr/lib/xorg/modules/drivers/vmwmem_drv.so
      fi

      # Backup current /etc/X11/xorg.conf.
      if [ -f $console_xorg_conf ]; then
         mv $console_xorg_conf $XORG_CONF_DIR/xorg.conf.vmware_backup
      fi

       # Recover original /etc/X11/xorg.conf
      if [ -s $VIEWAGENT_PATH/xorg.conf.orig ]; then
         mv -f $VIEWAGENT_PATH/xorg.conf.orig $console_xorg_conf
      fi

      if [ -f $XORG_CONF_DIR/edid.bin ]; then
         rm -f $XORG_CONF_DIR/edid.bin
      fi
      ;;
   "set" | "install")
      if [ "$mode" = "install" ]; then
         # Backup /etc/X11/xorg.conf
         if [ ! -f $VIEWAGENT_PATH/xorg.conf.orig ]; then
            if [ $vDGA_flag -eq 0 ]; then
               test -s "$console_xorg_conf" && cp -fp "$console_xorg_conf" $VIEWAGENT_PATH/xorg.conf.orig
             elif [ -f $XORG_CONF_DIR/xorg.conf.nvidia-xconfig-original ]; then
                # In vDGA mode, save the original xorg.conf of NVIDIA
                cp -f $XORG_CONF_DIR/xorg.conf.nvidia-xconfig-original $VIEWAGENT_PATH/xorg.conf.orig
             fi
         fi
         # Create /etc/X11/xorg.conf if need.
         if [ ! -d $XORG_CONF_DIR ]; then
            mkdir -p $XORG_CONF_DIR
         fi
         if [ ! -f $console_xorg_conf ]; then
            touch $console_xorg_conf
         fi

         if [ "$singleton" = "false" ] || \
            [ $vDGA_flag -eq 0 -a "$XSERVER_VMWMEM_ENABLED" = "true" ]; then
            xversion=`Xorg -version 2>&1 | grep "X.Org X Server" |awk -F'.' '{print $3}'`
            driver_file="/usr/lib/vmware/viewagent/resources/X11/vmwmem_drv_${xversion}.so"

            if [ -n "$xversion" ] && [ -f "$driver_file" ]; then
               install_foler="/usr/lib/xorg/modules/drivers"
               if [ -d "/usr/lib64/xorg/modules/drivers" ]; then
                  install_foler="/usr/lib64/xorg/modules/drivers"
               fi

               install $driver_file $install_foler/vmwmem_drv.so
            else
               echo_term "Failed to find xserver vmware memory driver with version $xversion"
               echo_term "Switch to use system default xserver driver, exit."
               echo_term "Please install Linux Agent again."

               sed -i '/^XSERVER_VMWMEM_ENABLED.*/d' $AGENT_CFG_FILE
               sed -i '$aXSERVER_VMWMEM_ENABLED=false' $AGENT_CFG_FILE
               exit
            fi
         fi
      fi

      if [ "$singleton" = "false" ]; then
         generate_xorg_vmwmem_conf
      elif [ $vDGA_flag -eq 1 ]; then
         generate_xorg_vgpu_conf $mode
      elif [ "$XSERVER_VMWMEM_ENABLED" = "true" ]; then
         generate_xorg_vmwmem_conf
      else
         generate_xorg_normal_conf
      fi

      disable_xorg_VTSwitch "$console_xorg_conf"

      # Remote input isolation on non-systemd platform, bug #2389861
      if  ! grep -q systemd /proc/1/comm; then
          if ! grep -q 'InputClass' "$VIEWAGENT_XORG_CONF_FILE"; then
              cat >> "$VIEWAGENT_XORG_CONF_FILE" <<-EOF
		Section "InputClass"
		    Identifier   "VMwareInput"
		    MatchDevicePath "/dev/input/*"
		    Option    "Ignore"    "true"
		EndSection
EOF
          fi
          if [ -f "$XORG_CONF_DIR/xorg.conf" ]; then
             if ! grep -q 'InputClass' "$XORG_CONF_DIR/xorg.conf"; then
              cat >> "$XORG_CONF_DIR/xorg.conf" <<-EOF
		Section "InputClass"
		    Identifier   "VMwareInput"
		    MatchDevicePath "/dev/vmware/*"
		    Option    "Ignore"    "true"
		EndSection
EOF
             fi
          fi
      fi
      ;;
   *)
      echo "configure_xorg_conf: fail to configure Xorg as wrong arguments."
      ;;
   esac
}

#
# If VM is configured as 2D/vSGA, reminder user to check document for
# vRAM Settings for Linux Guest Operating Systems
#
show_vRAM_Settings()
{
   nNVDIA=`lspci -v |grep "Kernel driver in use: nvidia"`
   if [ "$?" != "0" ]; then
      echo "Your VM is configured as 2D/vSGA, please check guide for \"Recommended Video Memory (vRAM) Settings\"."
   fi
}

display_manager_validity_check()
{
   # return 0 means display manager valid,
   # return 1 means display manager invalid.
   lsof -i :xdmcp > /dev/null 2>&1
   if [ "$?" != "0" ]; then
      echo "display manager invalid, XDMCP port isn't enabled"
      return 1
   fi

   return 0
}

directory_validity_check()
{
   dir=$1
   owner=$2
   group=$3
   mode=$4

   if [ -d "$dir" ]; then
      dirowner=`ls -ld $dir | awk '{print $3}'`
      if [ "$dirowner" != "$owner" ]; then
         echo "$dir exists, but the file owner doesn't match $owner"
         exit 18
      fi

      if [ "$mode" != "0" ]; then
         dirmode=`stat -c %a $dir`
         if [ "$dirmode" != "$mode" ]; then
            echo "$dir exists, but the file mode doesn't match $mode"
            exit 18
         fi
      fi
   else
      mkdir -m $mode $dir
      if [ "$?" != "0" ]; then
         echo "fails to create $dir"
         exit 18
      fi
      chown $owner:$group $dir
   fi
}


verify_udev_environment() {
   local meet_libudev=0
   local path_libudev

   case "$DISTRO_ID" in
   ${DISTRO_ID_CENTOS} |\
   ${DISTRO_ID_RHEL_CLIENT} |\
   ${DISTRO_ID_RHEL_SERVER} |\
   ${DISTRO_ID_RHEL_WORKSTATION})
      if [ ${OS_MAJOR} -ge "7" ]; then
         meet_libudev=1
      fi
      ;;

   ${DISTRO_ID_SUSE})
      if [ ${OS_MAJOR} -ge "12" ]; then
         meet_libudev=1
      fi
      ;;

   ${DISTRO_ID_UBUNTU})
      if [ ${OS_MAJOR} -ge "14" ]; then
         meet_libudev=1
      fi
      ;;

   *)
      ;;
   esac

   if [ "${meet_libudev}" = "1" ]; then
      #
      # We only support 64-bit system
      #
      ldconfig -p | grep libudev.so | grep "x86-64" >/dev/null 2>&1
      if [ "$?" = "0" ]; then
         path_libudev=`ldconfig -p | grep libudev.so | grep "x86-64" | awk 'NR==1{print $4}'`
         if [ -f ${path_libudev%/*}/libudev.so.0 ]; then
            meet_libudev=0
         else
            [ -f ${path_libudev} ] && \
            ln -sf ${path_libudev} ${path_libudev%/*}/libudev.so.0
            if [ "$?" = "0" ]; then
               meet_libudev=0
            fi
         fi
      fi
   fi

   return ${meet_libudev}
}

customise_multiple_session() {
   local pulseaudio_service_path="/usr/lib/systemd/user/pulseaudio.service"
   local gnome_software_config_path="/etc/xdg/autostart/gnome-software-service.desktop"
   local logrotate_path="/etc/logrotate.conf"
   local network_manager_config_path="/etc/NetworkManager/conf.d"
   local resource_backup="/usr/lib/vmware/viewagent/resources/backup"

   if [ "${SINGLETON_MODE}" != "false" ]; then
      return
   fi

   if [ "$#" -gt 0 ]; then
      case $1 in
      set)
         which systemctl > /dev/null 2>&1
         if [ "$?" = "0" ]; then
            # Disable suspend
            systemctl mask sleep.target suspend.target hibernate.target hybrid-sleep.target > /dev/null 2>&1
         fi

         if [ ! -d ${resource_backup} ]; then
            mkdir -p ${resource_backup}
            chown -R root:root ${resource_backup}
         fi

         # Clean wtmp file
         if [ -f "/var/log/wtmp" ]; then
            rm -f /var/log/wtmp
            ln -s /dev/null /var/log/wtmp
         fi

         # Set up size of wtmp to 100KB
         if [ -f ${logrotate_path} ]; then
            cp -fp ${logrotate_path} ${resource_backup}/logrotate.conf.backup
            sed -i '/^\/var\/log\/wtmp {/,/}/ s/^/#/' ${logrotate_path}
            cat >> ${logrotate_path} <<-\EOF
/var/log/wtmp {
    missingok
    create 0664 root utmp
    maxsize 100k
    rotate 5
}
EOF
         fi

         # Disable pulseaudio.service
         if [ -f ${pulseaudio_service_path} ]; then
            cp -fp ${pulseaudio_service_path} ${resource_backup}/pulseaudio.service.backup
            rm -f ${pulseaudio_service_path}
         fi

         # Disable the gnome-software
         if [ -f ${gnome_software_config_path} ]; then
            cp -fp ${gnome_software_config_path} ${resource_backup}/gnome-software-service.desktop.backup
            sed -i '/^X-GNOME-Autostart-enabled/ s/^/#/' ${gnome_software_config_path}
            sed -i '/^\[Desktop Entry\]/a\X-GNOME-Autostart-enabled=false' ${gnome_software_config_path}
         fi

         case "$DISTRO_ID" in
         ${DISTRO_ID_UBUNTU})
            if [ ${OS_MAJOR} -ge "18" ]; then
               # Disable polkit in NetworkManager
               if [ -d ${network_manager_config_path} ]; then
                  cat > ${network_manager_config_path}/vmware-disable-polkit.conf <<-\EOF
[main]
auth-polkit=false
EOF
               fi
            fi
            ;;
         esac
         ;;
      reset)
         which systemctl > /dev/null 2>&1
         if [ "$?" = "0" ]; then
            # Enable suspend
            systemctl unmask sleep.target suspend.target hibernate.target hybrid-sleep.target > /dev/null 2>&1
         fi

         if [ -L "/var/log/wtmp" ]; then
            rm -f /var/log/wtmp
         fi

         if [ -f ${resource_backup}/logrotate.conf.backup ]; then
            install -m=644 ${resource_backup}/logrotate.conf.backup ${logrotate_path}
            rm -f ${resource_backup}/logrotate.conf.backup
         fi

         if [ -f ${resource_backup}/pulseaudio.service.backup ]; then
            install -m=644 ${resource_backup}/pulseaudio.service.backup ${pulseaudio_service_path}
            rm -f ${resource_backup}/pulseaudio.service.backup
         fi

         # Revert the gnome-software-service.desktop
         if [ -f ${resource_backup}/gnome-software-service.desktop.backup ]; then
            install -m=644 ${resource_backup}/gnome-software-service.desktop.backup ${gnome_software_config_path}
            rm -f ${resource_backup}/gnome-software-service.desktop.backup
         fi

         case "$DISTRO_ID" in
         ${DISTRO_ID_UBUNTU})
            if [ ${OS_MAJOR} -ge "18" ]; then
               # Revert NetworkManager
               if [ -f ${network_manager_config_path}/vmware-disable-polkit.conf ]; then
                  rm -f ${network_manager_config_path}/vmware-disable-polkit.conf
               fi
            fi
            ;;
         esac

         ;;
      *)
         echo "Failed to customise multiple session: unknown the params $1"
         return
         ;;
      esac
   fi
}
