#!/bin/bash
###################################################
# Copyright 2019-2020 VMware, Inc.  All rights reserved.
###################################################

#
# Ensure script is executed by a superuser.
#
if [ "`id -u`" != "0" ]; then
   echo "Must configure XDMCP as a superuser!"
   exit 1
fi

generate_gdm_custom_conf()
{
    cat > $1/custom.conf <<-\EOF
	[xdmcp]
	Enable=true
EOF
}

generate_lightdm_conf()
{
    cat > /etc/lightdm/lightdm.conf <<-\EOF
	[XDMCPServer]
	enabled=true
EOF
}

# Load constants
. "`dirname $0`/commonlib.sh"
identify_distribution
define_constants

singleton_mode=true

while getopts m: o
do
   case $o in
      m)
         singleton_mode="$OPTARG"
         ;;
      *) echo "Invalid option -- $OPTSTRING"
         exit 1
         ;;
   esac
done

# Script file for uninstall
uninstall_script="$VMW_UNINSTALL_SCRIPTS_DIR/10-xdmcp"
cat > $uninstall_script <<<"#!/bin/sh"
chmod +x $uninstall_script

# Awk script for LightDM config manipulation
lightdm_conf_awk='
BEGIN {
    section="";
    xdmcp_done="no";
}

# Enter section
/^\[[^][]+\]/ {
    if ($0 ~ /\[XDMCPServer\]/) {     # entering section xdmcp
        section="xdmcp";
    } else {
        if (section == "xdmcp") { # comming out from section xdmcp
            if (xdmcp_done == "no") {
                print "enabled=true\n";
                xdmcp_done="yes";
            }
        }

        section="";
    }
}

# In section xdmcp
section == "xdmcp" {
    if ($0 ~ /^[\t ]*enabled=/) {
        xdmcp_done="yes";
        if ($0 !~ /=true/) {
            print "enabled=true";
            next;
        }
    }
}

{ print }

END {
    if (xdmcp_done == "no") {
        if (section == "xdmcp") {
            print "enabled=true";
        } else {
            print "[XDMCPServer]";
            print "enabled=true";
        }
    }
}
'

# Awk script for GDM config manipulation
gdm_conf_awk='
BEGIN {
    section="";
    xdmcp_done="no";
}

# Enter section
/^\[[^][]+\]/ {
    if ($0 ~ /\[xdmcp\]/) {     # entering section xdmcp
        section="xdmcp";
    } else if ($0 ~ /\[daemon\]/) { # entering section daemon
        section="daemon";
    } else {
        if (section == "xdmcp") { # comming out from section xdmcp
            if (xdmcp_done == "no") {
                xdmcp_done="yes";
                print "Enable=true\n";
            }
        }
        section="";
    }
}

# In section xdmcp
section == "xdmcp" {
    if ($0 ~ /^[\t ]*Enable=/) {
        xdmcp_done="yes";
        if ($0 !~ /=true/) {
            print "Enable=true";
            next;
        }
    }
}

# In section daemon
section == "daemon" {
    if ($0 ~ /^[\t ]*#WaylandEnable=false/) {
        print "WaylandEnable=false";
        next;
    }
}

{ print }

END {
    if (xdmcp_done == "no") {
        if (section == "xdmcp") {
            print "Enable=true";
        } else {
            print "[xdmcp]";
            print "Enable=true";
        }
    }
}
'

# Create backup directory if not exist
mkdir -p $VMW_DISPLAY_MANAGER_CONF_BACKUP_DIR > /dev/null 2>&1

# SUSE uses /etc/sysconfig/displaymanager
if [ "$DISTRO_ID" = "$DISTRO_ID_SUSE" ]
then
    suse_conf=/etc/sysconfig/displaymanager
    suse_conf_back=$VMW_DISPLAY_MANAGER_CONF_BACKUP_DIR/displaymanager
    cp -p $suse_conf $suse_conf_back && \
        sed -i -e 's/^\(DISPLAYMANAGER_REMOTE_ACCESS\)=.*$/\1="yes"/' $suse_conf && \
        cat >> $uninstall_script <<-EOF
		test -f "$suse_conf_back" && mv "$suse_conf_back" $suse_conf
EOF
    exit
fi

# GDM
test -d /etc/gdm && gdm_conf_dir=/etc/gdm
test -d /etc/gdm3 && gdm_conf_dir=/etc/gdm3

if [ -n "$gdm_conf_dir" ]
then
    gdm_conf=${gdm_conf_dir}/custom.conf
    if [ -f $gdm_conf ]
    then
        gdm_conf_back=$VMW_DISPLAY_MANAGER_CONF_BACKUP_DIR/gdm_custom.conf
        cp -p $gdm_conf $gdm_conf_back && \
            ( IFS=''; awk $gdm_conf_awk $gdm_conf > ${gdm_conf}.tmp && mv ${gdm_conf}{.tmp,} ) && \
            cat >> $uninstall_script <<-EOF
		test -f "$gdm_conf_back" && mv "$gdm_conf_back" $gdm_conf
EOF
    else
        generate_gdm_custom_conf $gdm_conf_dir && \
            cat >> $uninstall_script <<-EOF
		unlink $gdm_conf
EOF
    fi

    if [ "$singleton_mode" = "false" ]
    then
        # Config MaxSessions for multiple-session.
        sed -i -e '/\[xdmcp\]/{ p; s/.*/MaxSessions=50/; p; s/.*/DisplaysPerHost=50/; :loop; { n; /\[.*\]/b print; /MaxSessions=/s/.*//; /DisplaysPerHost=/s/.*// }; b loop }
$a \
[xdmcp]\
Enable=true \
MaxSessions=50 \
DisplaysPerHost=50
b; :print; n; b print' $gdm_conf

    fi
fi

# LightDM
if [ -d /etc/lightdm ]
then
    lightdm_conf=/etc/lightdm/lightdm.conf
    if [ -f $lightdm_conf ]
   then
       lightdm_conf_back=$VMW_DISPLAY_MANAGER_CONF_BACKUP_DIR/lightdm.conf
       cp -p $lightdm_conf $lightdm_conf_back && \
           ( IFS=''; awk $lightdm_conf_awk $lightdm_conf > ${lightdm_conf}.tmp && mv ${lightdm_conf}{.tmp,} ) && \
           cat >> $uninstall_script <<-EOF
		test -f "$lightdm_conf_back" && mv "$lightdm_conf_back" $lightdm_conf
EOF
   else
       generate_lightdm_conf && \
           cat >> $uninstall_script <<-EOF
		unlink $lightdm_conf
EOF

   fi
fi
