#!/bin/sh

###################################################
# Copyright 2015 -2019 VMware, Inc.  All rights reserved.
###################################################


if [ "`id -u`" != "0" ]; then
   #
   # Requires permission to access the XAUTHORITY file.
   #
   echo "StartXServer.sh must be executed as superuser"
   exit 1
fi

#
# Generate xauth file
# located at /var/vmware/viewagent/xauth/.xauth:display
#
generateXauth() {
   local majorVersion=""
   local xauth_folder=$VIEWAGENT_TMP_PATH/xauth

   if [ ! -f "$xauth_folder" ]; then
      mkdir -p $xauth_folder
      chmod 700 $xauth_folder
   fi

   if [ "$DISTRO_ID" = "$DISTRO_ID_CENTOS" ] || [ "$DISTRO_ID" = "$DISTRO_ID_RHEL_CLIENT" ] || \
      [ "$DISTRO_ID" = "$DISTRO_ID_RHEL_SERVER" ] || [ "$DISTRO_ID" = "$DISTRO_ID_RHEL_WORKSTATION" ]; then
      majorVersion=`identify_distribution -r | head -c 1`
      if [ "$majorVersion" = "6" ]; then
         command -v getenforce >/dev/null 2>&1 && [ `getenforce` != "Disabled" ] && \
         xauth_file="/tmp/.xauth"$display
      fi
   fi

   [ -f "$xauth_file" ] && rm -f ${xauth_file} >/dev/null 2>&1

   touch ${xauth_file}
   chown root:root ${xauth_file}
   chmod 600 ${xauth_file}

   xauth -f ${xauth_file} add $(hostname)/unix${display} \
                             MIT-MAGIC-COOKIE-1 `mcookie`

   if [ "$xauth_file" != "$XAUTHORITY" ]; then
      mv -f $xauth_file $XAUTHORITY
      xauth_file=$XAUTHORITY
   fi
}

#
# Check if need to pass -seat to Xorg.
#
#TODO: As the ancient X server does not support a fake seat, the input
# isolation should be done after X server is ready(Bug #2389861).
is_support_seat()
{
    x_ver=$(Xorg -version 2>&1 | awk '/X.Org X Server/{if (split($4, ver, ".")>1) {print ver[2]; exit}}')

    if [ "$x_ver" '>' "15" ]; then
        return 0;
    else
        return 1;
    fi
}

#
# Main
#

#
# Include common constants
#
. "`dirname $0`/commonlib.sh"
identify_distribution
define_constants
identify_xserver_vmwmem_enabled

#
# Enable core dumps
#
ulimit -c unlimited

# Workaround for bug 1649484.
# Sometimes mksvchanserver/cdrServer crash when logoff.
# Move away the crash files for a better user experience.
if [ "$DISTRO_ID" = "$DISTRO_ID_UBUNTU" ]; then
    find /var/crash -name '_usr_lib_vmware_viewagent*' -exec mv -f '{}' /tmp/vmware-root/ \; >/dev/null 2>&1
fi


display=$DISPLAY
singleton_mode=true

while getopts d:m o
do
    case $o in
        d)
            display="$OPTARG"
            ;;
        m)
            singleton_mode=false
            ;;
        *) echo "Invalid option -- $OPTSTRING"
           exit 1
           ;;
    esac
done

if [ "$singleton_mode" = "false" ]; then
   display_manager_validity_check
   [ "$?" = "1" ] && exit 18
fi

# Generate xauth file
xauth_file=$XAUTHORITY
if [ -z "$xauth_file" ]; then
   echo "Wrong xauth file path, exit"
   exit 1
fi
generateXauth

# Use our own xorg.conf and allocate seat when needed
x_seat=""
x_config="-config $VIEWAGENT_XORG_CONF_FILE"

if [ "$singleton_mode" = "false" -o "$XSERVER_VMWMEM_ENABLED" = "true" ]; then
   x_seat="-seat seat-vmwfake"

   if [ "$singleton_mode" = "true" ]; then
      # For vGPU, see bug #2312593, review #1511345 and #1511789.
      nNVDIA=`lspci -v | grep "Kernel driver in use: nvidia" | wc -l `
      nEDID=`grep "/etc/X11/edid.bin" $VIEWAGENT_XORG_CONF_FILE| wc -l`
      if [ $nNVDIA -ge 1 -a  $nEDID -ge 1 ]; then
         x_seat="-seat seat-vmwfake -keeptty"
      fi
   fi
fi

if ! is_support_seat; then
    x_seat=""
fi

exec Xorg $display -auth ${xauth_file} -once -query 127.0.0.1 $x_config $x_seat > /dev/null 2>&1
