#!/bin/sh

###################################################
# Copyright 2015 -2019 VMware, Inc.  All rights reserved.
###################################################


if [ "`id -u`" != "0" ]; then
   #
   # Requires permission to access the XAUTHORITY file.
   #
   echo "StartBlastWorker.sh must be executed as superuser"
   exit 1
fi

#
# Default vaule for blast worker.
#
singleton=TRUE
enableFips=FALSE
enableHelpDesk=FALSE
xdisplay=
xauthority=
desktopId=
language=


#
# Main
#

#
# Include common constants
#
. "`dirname $0`/commonlib.sh"
define_constants
identify_distribution

#
# Enable core dumps
#
ulimit -c unlimited

while getopts a:x:y:b:d:e:m:u:c:p:k:l:f:i:stv o
do
   case $o in
   a)
      xauthority=$OPTARG;;
   d)
      xdisplay=$OPTARG;;
   m)
      port="-m $OPTARG";;
   c)
      sslCiphers="--sslCiphers $OPTARG";;
   i)
      desktopId=$OPTARG;;
   p)
      sslProtocols="--sslProtocols $OPTARG";;
   s)
      sslOptions="--sslCipherServerPreference";;
   t)
      enableHelpDesk="TRUE";;
   f)
      enableFips="TRUE";;
   l)
      language="$OPTARG";;
   v)
      singleton="FALSE";;
   x)
      xres=$OPTARG;;
   y)
      yres=$OPTARG;;
   esac
done

if [ -z "$xdisplay" ]; then
   echo "Missing parameter display, exit"
   exit 1
fi

if ! getent passwd ${BLAST_ACCOUNT} >/dev/null; then
   echo "The blast server account ${BLAST_ACCOUNT} doesn't exist"
   exit 18
fi

directory_validity_check $BLAST_RUNTIME_DIR $BLAST_ACCOUNT $BLAST_GROUP 700

# We should kill the old VMwareBlastServer first in singleton mode.
# Because the old VMwareBlastServer attach 22443 port and result in
# new VMwareBlastServer failed.
if [ "$singleton" = "TRUE" ]; then
   existPid="$(pidof VMwareBlastServer)"
   [ -n "$existPid" ] && kill -TERM $existPid
fi

isRedhatOrCentOS6="FALSE"
majorVersion=`identify_distribution -r | head -c 1`
if [ "$majorVersion" = "6" ]; then
   distroID=$(identify_distribution -i)
   if [ "$distroID" = "CentOS" ] || [ "$distroID" = "RedHatEnterpriseClient" ] || [ "$distroID" = "RedHatEnterpriseServer" ] || [ "$distroID" = "RedHatEnterpriseWorkstation" ]; then
      isRedhatOrCentOS6="TRUE"
   fi
fi

# SLED11 also has screen-saver autofit issue (Bug 1923479)
isSLED11="FALSE"
majorVersion=$(identify_distribution -r)
if [ "$majorVersion" = "11" ]; then
   distroID=$(identify_distribution -i)
   if [ "$distroID" = "SUSE LINUX" ]; then
      isSLED11="TRUE"
   fi
fi

#
# Workaround for the screen saver bug for Redhat 6.x, CentOS 6.x and SLED11.
# Disable the resizing when the screen is locked.
#
if [ "$isRedhatOrCentOS6" = "TRUE" ] || [ "$isSLED11" = "TRUE" ]; then
   disableResizeWhenScreenLocked="TRUE"
else
   disableResizeWhenScreenLocked="FALSE"
fi

export LD_LIBRARY_PATH="$LD_LIBRARY_PATH:/usr/lib/vmware/viewagent/VMwareBlastServer"
export HOME=/tmp/vmware-root

if [ -z "$xauthority" ]; then
   xauthority="$VIEWAGENT_TMP_PATH/xauth/.xauth$xdisplay"
fi

export DISPLAY=$xdisplay
export XAUTHORITY=$xauthority

xauthfile=$BLAST_RUNTIME_DIR/.xauth$xdisplay
[ -f "$xauthfile" ] && rm -f $xauthfile >/dev/null 2>&1

#
# Workaround for a system issue.
# On the CentOS and Redhat 6.x, it is possible that xauth failed for the SELinux security context.
# Here we workaround to copy xauth file directly.
#
if [ "$isRedhatOrCentOS6" = "TRUE" ] &&
   command -v getenforce >/dev/null 2>&1 && [ `getenforce` != "Disabled" ]; then
   cp -f $XAUTHORITY $xauthfile
else
   xauth extract $xauthfile $xdisplay
fi

if [ ! -f "$xauthfile" ]; then
   echo "Failed to create the xauthority file, exit"
   exit 1
fi

chown $BLAST_ACCOUNT $xauthfile
chmod 700 $xauthfile
export XAUTHORITY=$xauthfile

if [ -n "$desktopId" ]; then
   export VMW_DESKTOP_ID=$desktopId
fi

# Set language in desktop info file
if [ -n "$language" -a -n "$DISPLAY" ]; then
   displayNum=`echo $DISPLAY | awk -F':' '{print $NF}'`
   desktopInfoFile="${DESKTOP_TMP_PATH}/vmw_desktopid.$displayNum"
   if [ -f "$desktopInfoFile" ]; then
      grep -i "DESKTOPID=${desktopId}" $desktopInfoFile
      if [ "$?" = "0" ]; then
         echo "LANG=$language" >> $desktopInfoFile
      fi
   fi
fi

randomHandShakeFlag="-r "

# In signelton mode, blast ip is all and port is 22443.
blastPort=${BLAST_PORT}
blastIP=
udpExternalPort=${BLAST_PORT}

# In multi-session mode, blast ip is localhost and port is random.
if [ "$singleton" = "FALSE" ]; then
   blastPort=0
   blastIP="-s "RemoteDisplay.vnc.blastSocket.ip=localhost""
fi

# Configure the resolution.
if [ -n "$xres" -a -n "$yres" ]; then
    configure_xorg ${xres}x${yres}
else
    configure_xorg
fi

exec /usr/lib/vmware/viewagent/VMwareBlastServer/VMwareBlastServer ${port} \
   ${randomHandShakeFlag} ${sslProtocols} ${sslCiphers} ${sslOptions} \
   -v -b ${blastPort} -u ${BLAST_ACCOUNT} -c ${blastIP} \
   -s "RemoteDisplay.vnc.webSocket.useSSL=${singleton}" \
   -s "RemoteDisplay.udpExternalPort=${udpExternalPort}" \
   -s "RemoteDisplay.allowMultimon=TRUE" \
   -s "RemoteDisplay.allowSoftwareH264=TRUE" \
   -s "RemoteDisplay.allowVMWServerHeartbeat=TRUE" \
   -s "RemoteDisplay.allowSessionCloseMessage=TRUE" \
   -s "RemoteDisplay.allowRequestStreamState=TRUE" \
   -s "RemoteDisplay.heartbeatInterval=120" \
   -s "mksVNCServer.disableResizeWhenScreenLocked=$disableResizeWhenScreenLocked" \
   -s "mksVNCServer.lastRootSizeCheck=TRUE" \
   -s "mksVNCServer.initialClippedWidth=2560" \
   -s "mksVNCServer.initialClippedHeight=1600" \
   -s "mksVNCServer.checkPulseAudio=TRUE" \
   -s "mks.panicOnTopologyFail=TRUE" \
   -s "mks.enableMKSControl=TRUE" \
   -s "mks.enableFIPSMode=${enableFips}" \
   -s "mks.enableHelpDesk=${enableHelpDesk}" \
   -s "mks.x.skipCoreDumpOnXIOError=TRUE" > /dev/null 2>&1

exit $?

