#!/bin/sh

###################################################
# Copyright 2019-2020 VMware, Inc.  All rights reserved.
###################################################

#
# Check and terminate the old systemd sessions.
#
check_systemd_sessions() {
   which loginctl > /dev/null 2>&1
   if [ "$?" != "0" ]; then
      echo "System doesn't support loginctl, skip to check systemd sessions."
      return
   fi

   local sid
   local dpyNum=`echo $display | awk -F':' '{print $NF}'`
   if [ $dpyNum -ge 100 -a $dpyNum -le 200 ]; then
      if [ -d "/var/run/systemd/sessions" ]; then
         for file in `find /var/run/systemd/sessions -type f`
         do
            if grep ":$dpyNum" "$file" > /dev/null 2>&1
            then
               sid=${file##*/}
               echo "Session $sid for display $display wrongly existes, terminate it."
               loginctl terminate-session $sid
            fi
         done
      else
         while read sid
         do
            loginctl show-session $sid -p Display | grep ":$dpyNum" > /dev/null 2>&1
            if [ "$?" = "0" ]; then
               echo "Session $sid for display $display wrongly existes, terminate it."
               loginctl terminate-session $sid
            fi
         done << !!!
         `loginctl list-sessions --no-legend |awk '{print $1}'`
!!!
      fi
   else
      echo "Invalid display $display, skip to check systemd sessions."
      return
   fi
}

#
# Configure display manager's process limits, i.e. /proc/<pid>/limits
#
configure_dm_prlimit()
{
    local nofile=100000
    local dm_pid=$(lsof -t -i :xdmcp)

    if [ -z "$dm_pid" ]; then
        echo "The display manager seems not work."
        return 1;
    fi

    which prlimit > /dev/null 2>&1
    if [ "$?" = "0" ]; then
       read -r soft hard <<-EOF
	    $(prlimit --pid "$dm_pid" --nofile --noheadings -o soft,hard)
EOF
       if [ "$soft" -lt $nofile -o "$hard" -lt $nofile ]; then
          prlimit --pid "$dm_pid" --nofile=$nofile
       fi
    else
       read -r soft hard <<-EOF
       $(awk '/Max open files/{ print $4,$5 }' /proc/$dm_pid/limits)
EOF
       if [ "$soft" -lt $nofile -o "$hard" -lt $nofile ]; then
          echo -n "Max open files=$nofile:$nofile" > /proc/$dm_pid/limits
       fi
    fi
}

#
# Clear the remnants stayed in /tmp/.ICE-unix, bug #2399973, #2414218
#
clean_xsession_tmp_files() {
   local tmpf_ice_opened=/tmp/ice-opened.$$
   local tmpf_ice_all=/tmp/ice-all.$$
   local ice_unix_dir=/tmp/.ICE-unix

   for f in $(find /tmp/ -name 'ice-opened.*')
   do
      if ps -p "${f#*.}" -h -o 'comm' | grep "PostLogout.sh"  >/dev/null 2>&1
      then
         echo "There's another one doing this job, do nothing."
         exit 0
      else
         rm -f $f
      fi
   done

   # Generate tmp files in which record the opened/all ICE-unix socket file names
   lsof -wt +D "$ice_unix_dir" | sort > "$tmpf_ice_opened"
   find "$ice_unix_dir" -type s -exec basename '{}' \; | sort > "$tmpf_ice_all"

   # Remove the unused ICE-unix socket files
   comm -13 "$tmpf_ice_opened" "$tmpf_ice_all" | xargs -I {} rm -f "${ice_unix_dir}/{}"

   # Remove the tmp record files
   rm -rf "$tmpf_ice_opened" "$tmpf_ice_all"
}

display=$DISPLAY

while getopts d: o
do
    case $o in
        d)
            display="$OPTARG"
            ;;
        *) echo "Invalid option -- $OPTSTRING"
           exit 1
           ;;
    esac
done

# The systemd sessions may wrongly exist after desktop logout for some time.
# We should check and terminate the old sessions.
check_systemd_sessions

# Clean the unused files in /tmp/.ICE-unix
clean_xsession_tmp_files

# Configure display manager's process limits
configure_dm_prlimit
