#!/bin/bash

usage=

createUserHomeFolder() {
   local user
   local group

   for file in `ls -a $1`
   do
      if [ x"$file" != x"." -a x"$file" != x".." ]; then
         if [ -d "$1/$file" ]; then
            echo "create $file folder in persistent disk"
            id $file > /dev/null 2>&1
            if [ "$?" == "0" ]; then
               mkdir -p /media/tmp/$file
               if [ "$?" != "0" ]; then
                  echo "Failed to create $file folder"
               fi
               user=`stat -c "%U" $1/$file`
               group=`stat -c "%G" $1/$file`
               echo "user: $user, group: $group"
               chown -R $user:$group /media/tmp/$file
               if [ "$?" != "0" ]; then
                  echo "chown failed, rollback copy folder action"
                  rm -rf /media/tmp/$file
               fi
            else
               echo "$file is not a valid user, don't create folder in persistent disk"
            fi
         fi
      fi
   done
}

getuseage() {
   local diskid=$1
   local line
   local mountFolder
   local mounted="0"
   local persistentDiskType
   local internalDiskType
   local diskType
   local simFileName
   local simDatFileName
   local formatOpt=""

   echo "process disk: $diskid..."

   line=`mount | grep "$diskid" | awk '{print $3}'`
   if [ -z "$line" ]; then
      echo "$diskid isn't mounted, mount it to /media/tmp folder."
      line="/media/tmp"
      #mount to temp folder
      mkdir -p $line
      mount "$diskid"1 $line

      if [ "$?" != "0" ]; then
         echo "mount to tmp folder error for disk: $diskid."
         return
      fi
      mounted="1"
   fi

   # if this is not the disk created by composer, umount it if it mount to /media/tmp
   if [ ! -f "$line/simvol.dat" ] && [ ! -f "$line/.simvol.dat" ]; then
      echo "disk $line is not the composer created disk"
      if [ "$mounted" = "1" ]; then
         umount $line
         if [ "$?" = "0" ]; then
            echo "umount error for $diskid: $line"
         fi
      fi
      return
   fi

   echo "disk $diskid is mount in: $line."

   if [ -f "$line/simvol.dat" ]; then
      simFileName=simvol.dat
   elif [ -f "$line/.simvol.dat" ]; then
      simFileName=.simvol.dat
   fi

         # simvol.dat: "sim.versionMajor=0;sim.versionMinor=4;usage=1;mountPoint=*;"
         usage=`cat $line/$simFileName | awk -F ';' '{print $3}' | awk -F '=' '{print $2}'`
         echo "usage: $usage"

         case "$usage" in
            "1")
               # internal disk
               mountFolder="/var/vmware/viewagent/InternalDisk"
               ;;
            "2")
               # persistent disk
               mountFolder=$persistentDiskPath
               ;;
            "4")
               # disposal disk
               mountFolder="/var/vmware/viewagent/SystemDisposal"
               ;;
         esac

         sleep 1

         # If it's internal disk, copy the sim.dat to local disk,
         # and waiting for standalone to read it.
         if [ "$usage" = "1" ]; then
            echo "copy sim.dat to /tmp/sim.dat"
            cp -p $line/sim.dat /tmp/sim.dat
         fi

         # if it's persistent disk, format it to user customize type
         if [ "$usage" = "2" ]; then
            persistentDiskType=`df -Th | grep "$diskid" | awk '{print $2}'`
            if [ "$persistentDiskType" = "fuseblk" ]; then
               echo "format $diskid to user customize type"
               cp -p $line/$simFileName /tmp/$simFileName
               umount $line

               # format user disk with given type, if exit, otherwise using default type
               if [ -n "$userDiskFormat" ]; then
                  if [ "$userDiskFormat" = "xfs" ]; then
                      formatOpt="-f"
                  fi
                  mkfs -t $userDiskFormat $formatOpt "$diskid"1
                  if [ "$?" != 0 ]; then
                     echo "Failed to format user disk with type $userDiskFormat, using default type $defaultUserDiskFormat"
                     if [ "$defaultUserDiskFormat" = "xfs" ]; then
                         formatOpt="-f"
                     fi
                     mkfs -t $defaultUserDiskFormat $formatOpt "$diskid"1
                     if [ "$?" != 0 ]; then
                        echo "Failed to format user disk with default type $defaultUserDiskFormat"
                     fi
                  fi
               else
                  if [ "$defaultUserDiskFormat" = "xfs" ]; then
                      formatOpt="-f"
                  fi
                  mkfs -t $defaultUserDiskFormat $formatOpt "$diskid"1
               fi

               mount "$diskid"1 $line
               cp -p /tmp/$simFileName $line/.simvol.dat
               chmod 700 $line/$simFileName
               chmod 700 $line/.simvol.dat

               if [ "$isRedhat7" = "true" ]; then
                  createUserHomeFolder $mountFolder
               fi
            else
              echo "don't need to format persistent disk"
            fi
         fi

         # unmount the disk from /mount/tmp/
         umount $line
         if [ "$?" != "0" ]; then
            echo "umount error for disk: $line"
            return
         fi

	 mkdir -p $mountFolder
	 mount "$diskid"1 $mountFolder
	 if [ "$?" != "0" ]; then
	    echo "mount error for disk: $diskid"
	    return
	 fi
         # need to restorecon the mount folder if selinux is enable.
         if [ "$usage" = "2" ]; then
            which restorecon > /dev/null 2>&1
            if [ "$?" = "0" ]; then
               restorecon -R $mountFolder
            fi
         fi
}


#
# main
#

#
# Include common constants
#
. "`dirname $0`/commonlib.sh"

identify_distribution

persistentDiskPath="/home"
defaultUserDiskFormat="ext4"
userDiskFormat=
isRedhat7="false"

OS_MAJOR=`echo "$DISTRO_RELEASE" | cut -f1 -d"."`
OS_MINOR=`echo "$DISTRO_RELEASE" | cut -f2 -d"."`

# sled11 ext4 is readonly, need to format it to ext3
case "$DISTRO_ID" in
   ${DISTRO_ID_CENTOS}|\
   ${DISTRO_ID_RHEL_CLIENT}|\
   ${DISTRO_ID_RHEL_SERVER}|\
   ${DISTRO_ID_RHEL_WORKSTATION})
      if [ "$OS_MAJOR" = "7" ]; then
         echo "rhel7, change default disk type to xfs"
         defaultUserDiskFormat="xfs"
         isRedhat7="true"
      fi
      ;;
   ${DISTRO_ID_SUSE})
      if [ "$OS_MAJOR" = "11" ]; then
         echo "suse11, change default disk type to ext3"
         defaultUserDiskFormat="ext3"
      fi
      ;;
esac

while getopts p:f: o
do
   case $o in
   p)
      persistentDiskPath="$OPTARG"
      echo "persistent disk path: $persistentDiskPath"
      ;;
   f)
      userDiskFormat="$OPTARG"
      echo "User disk format: $userDiskFormat"
      if [ "$userDiskFormat" = "ntfs" ] || [ "$userDiskFormat" = "vfat" ] || [ "$userDiskFormat" = "msdos" ]; then
         echo "user define file system is ntfs/vfat/msdos, ignore it and user default value."
         userDiskFormat=$defaultUserDiskFormat
      fi
      if [ "$DISTRO_ID" = "SUSE LINUX" ] && [ "$OS_MAJOR" = "11" ] && [ "$userDiskFormat" = "ext4" ]; then
         userDiskFormat="ext3"
         echo "ext4 is readonly in SELD11, so set the format to ext3"
      fi
   esac
done


diskcollect=`fdisk -l | grep "Disk /" | awk '{print $2}'`
echo "disk collection: "
echo "$diskcollect"

echo "$diskcollect" | while read line
do
   disk=`echo $line | sed 's/://g'`
   getuseage $disk
done



