#!/bin/sh

########################################################
# Copyright 2015-2019 VMware, Inc.  All rights reserved.
########################################################


if [ "`id -u`" != "0" ]; then
   #
   # Requires permission to access the audio file.
   #
   echo "MonitorAudio.sh must be executed as superuser"
   exit 1
fi

TRY_MAX=7200
tryCount=0
trySleep=0

OLD_FIFO_PATH=""
NEW_FIFO_PATH=""

audioinInstalled="no"

xdisplay=
xauthority=

calculate_while_sleep() {
   local val
   val=`expr $tryCount / 120`
   trySleep=`expr $val + 1`
}

parse_pulse_path() {
   local userName=""
   local logEventPath="/var/vmware/viewagent/logEvent"
   local try=0
   local maxTry=5
   while true
   do
      if [ $try -lt $maxTry ]; then
         try=`expr $try + 1`
      else
         echo "Fail to get the current login user name, exit."
         exit 1
      fi

      sleep $try

      if [ -s "$logEventPath" ]; then
         userName=`cat "$logEventPath" | awk '{print $2}'`
      fi

      if [ -n "$userName" ]; then
         break
      fi
   done

   if [ -n "$userName" ] && [ -n "$oldpulse" ]; then
      oldpulse=`su - "$userName" -c "readlink -f '$oldpulse'"`
   fi
}

parse_pulse_server() {
   [ $# -lt 1 ] && return
   local dir
   local temp
   local oldpulse
   local newpulse

   temp="${1#*/}"
   oldpulse="/${temp%%\'*}"
   # If root can't read the pulse socket, this meams
   # the pulse file may be linked in a mounted dir.
   # We should find the real path of pulse file.
   if [ ! -r "$oldpulse" ]; then
      parse_pulse_path
   fi

   if [ "$audioinInstalled" = "yes" ]; then
      OLD_FIFO_PATH="${oldpulse%/*}"
   fi

   if [ -S "$oldpulse" ]; then
      owner=`ls -ld "$oldpulse" | awk '{print $3}'`
      #
      # The hard link of pulse server don't support
      # Cross block device. So our pulse dir should
      # be in the same device with original.
      #
      dir=`df -P "$oldpulse" | awk 'END{print $NF}'`
      [ "$dir" = "/" ] && dir=""
      dir="$dir/.vmwblastaudio$DISPLAY"

      if [ "$audioinInstalled" = "yes" ]; then
         NEW_FIFO_PATH="$dir"
      fi

      newpulse=$dir/native
      #
      # If pulse server is in our dir, return
      #
      [ "$oldpulse" = "$newpulse" ] && return

      [  -d "$dir" ] && rm -rf "$dir"
      mkdir -p -m 710 "$dir"
      chown $owner "$dir"
      chgrp $BLAST_GROUP "$dir"

      command -v getenforce >/dev/null 2>&1 && [ `getenforce` != "Disabled" ] && \
      chcon -u system_u -r object_r -t user_tmp_t -l s0 -R "$dir"

      ln -P "$oldpulse" "$newpulse"
      chgrp $BLAST_GROUP "$newpulse"
      pax11publish -e -S "$newpulse"
      #
      # "pax11publish -e" may delete the PULSE_COOKIE in RHEL 7.4.
      # So we should add the PULSE_COOKIE in display if it was
      # deleted by pax11publish.
      #
      pax11publish -d | grep -i cookie >/dev/null 2>&1
      if [ "$?" != "0" ]; then
         if [ ! -f $VMWARE_ROOT_PATH/.config/pulse/cookie ]; then
            echo "The pulse audio can't find pulse cookie, exit."
            exit 1
         fi
         local pulse_cookie=`hexdump -e '16/1 "%02x" ""' $VMWARE_ROOT_PATH/.config/pulse/cookie`
         xprop -root -f PULSE_COOKIE 8s -set PULSE_COOKIE $pulse_cookie
      fi
      return
   else
      echo "The pulse audio don't in unix socket mode, exit."
      exit 1
   fi
}


#
# Main
#

#
# Include common constants
#
. "`dirname $0`/commonlib.sh"
identify_distribution
define_constants

while getopts a:d: o
do
   case $o in
   a)
      xauthority=$OPTARG;;
   d)
      xdisplay=$OPTARG;;
   esac
done

if [ -n "$xdisplay" ]; then
   export DISPLAY=$xdisplay
fi

if [ -n "$xauthority" ]; then
   export XAUTHORTIY=$xauthority
fi


if [ -f "/usr/lib/vmware/vchan_plugins/libviewMMDevRedir.so" ]; then
   audioinInstalled="yes"
fi

#
# `pax11publish -i` need ~/.config/pulse dir
# to cache the cookie, otherwise it may
# return false in some linux system.
#
VMWARE_ROOT_PATH=/tmp/vmware-root
directory_validity_check $VMWARE_ROOT_PATH root root 0
mkdir -p $VMWARE_ROOT_PATH/.config/pulse
if [ ! -d $VMWARE_ROOT_PATH/.config/pulse ]; then
   echo "Fail to create the $VMWARE_ROOT_PATH/.config/pulse"
   exit 1
fi
chmod -R 750 $VMWARE_ROOT_PATH/.config
export HOME=$VMWARE_ROOT_PATH

while true
do
   #
   # If we try more than TRY_MAX times, finish
   # this script.
   #
   if [ $tryCount -lt $TRY_MAX ]; then
      tryCount=`expr $tryCount + 1`
   else
      exit 1
   fi

   calculate_while_sleep
   sleep $trySleep

   #
   # We don't need to monitor audio any more. when
   # the blast server exit.
   #
   blastname="vmwblast"
   if [ ! -n "$blastname" ]; then
      exit 1
   else
      pax11publish -d >/dev/null 2>&1
      if [ "$?" != "0" ]; then
         echo "Failed to exec pax11publish with DISPLAY: $DISPLAY and XAUTHORITY: $XAUTHORITY\n"
         exit 1
      else
         while read item
         do
            [ ! -n "$item" ] && continue
            if [ "PULSE_SERVER" = ${item%=*} ]; then
               parse_pulse_server $item

               #
               # We don't need to load module-pipe-source if audioin was not installed
               #
               if [ "$audioinInstalled" = "yes" ]; then
                  [ -p "$OLD_FIFO_PATH/vmwaudioin" ] && pactl unload-module module-pipe-source
                  [ -p "$OLD_FIFO_PATH/vmwaudioin" ] && rm "$OLD_FIFO_PATH/vmwaudioin"
                  [ -p "$NEW_FIFO_PATH/vmwaudioin" ] && rm "$NEW_FIFO_PATH/vmwaudioin"
                  pactl load-module module-pipe-source format=S16LE rate=8000 channels=1 file="$OLD_FIFO_PATH/vmwaudioin" source_name=vmw_fifo_input
                  # Under SLED11, need to set default source manually.
                  if [ "$DISTRO_ID" = "$DISTRO_ID_SUSE" -a "$OS_MAJOR" = "11" ]; then
                      su -c 'pacmd set-default-source vmw_fifo_input' $owner
                  fi
                  ln -P "$OLD_FIFO_PATH/vmwaudioin" "$NEW_FIFO_PATH/vmwaudioin"
                  chgrp $BLAST_GROUP "$OLD_FIFO_PATH/vmwaudioin"
                  chmod 770 "$OLD_FIFO_PATH/vmwaudioin"
               fi

               exit 0
            fi
         done << !!!
`pax11publish -i`
!!!
      fi
   fi
done
