#!/usr/bin/python
#
# This script helps to find any ipv4/ipv6 route that
# is programmed in the NPU but not programmed in the
# fabric.
# That means it scans through the FEC entries and verifies
# whether the MCID embedded in the FEC is programmed in the
# fabric by mrib (show mrib fgid info <mcid>).

import argparse
import subprocess
import sys
import os
import re
import socket
import struct

location=-1
only_on_npu=-1
print_fec = False
verify_fec = True
list_routes_not_in_hw = False
ipv4_kaps_tbl = {}
ipv6_compid_tbl = {}
ipv6_kaps_tbl = {}
alloced_fec_tbl = {}
mcids_inhw_tbl = {}
mcdb = {}
print_zombie_fec = False

def error_exit(prstr):
    print prstr
    print"Exiting.."
    sys.exit(1)


def parse_args():
    global print_fec, verify_fec, location, only_on_npu, print_zombie_fec, \
           list_routes_not_in_hw

    parser = argparse.ArgumentParser(description="Check for any empty mcid in fec database", \
            formatter_class=argparse.RawDescriptionHelpFormatter, \
            epilog="""

mcast_pd_find_emtpy_mcid.py helps to find the traffic black hole
scenarios resulting from an empty mcid in the ingress chain.
For the NPU the mcid is just a number, NPU forwards the
traffic to fabric attaching the mcid and it gets dropped
in the fabric if there are no fapids associated with that mcid.

The drops are only shown in the fabric asic errors, in the
below command. It is the counter drhpQueryEmptyMulticastIdInt
that gets incremented

sysadmin-vm:0_RP0# show asic-errors sfe all all location | i "Name | sfe| count"

This script makes it easy to identify empty mcids without going
through asic errors and pin points all the affected S,G in the
fec database.

To check the sanity of the fec database run:
mcast_pd_find_emtpy_mcid.py

To print all the entries in fec database used by mcast:
mcast_pd_find_emtpy_mcid.py -p

To check for any stale fec in the hardware tables:
mcast_pd_find_emtpy_mcid.py -z
            """)

    parser.add_argument("-p", help="Print all the mcast entries in fec table", \
            required=False, action="store_true")
    parser.add_argument("-l", '--loc', help="Test only on this LC", \
            required=False, type=int, choices=xrange(0, 8), default=-1)
    parser.add_argument("-n", '--npu', help="Test only on this NPU", \
            required=False, type=int, choices=xrange(0, 16), default=-1)
    parser.add_argument("-z", help="Print fecs that are present only in hardware tables", \
            required=False, action="store_true")
    parser.add_argument("-u", help="Print unprogramed routes and mcid. ie route is in mrib not in npu", \
            required=False, action="store_true")
    args = vars(parser.parse_args())

    if args['p']:
        verify_fec = False
        print_fec = True
    if args['loc'] != -1:
        location = args['loc'] * 256
    if args['npu'] != -1:
        only_on_npu = args['npu']
    if args['z']:
        print_zombie_fec = True
    if args['u']:
        verify_fec = False
        list_routes_not_in_hw = True


def cmd_op(cmd):
    try:
        return subprocess.check_output(cmd, shell=True).split('\n')
    except subprocess.CalledProcessError as e:
        return None


def check_for_rplc():
    rplc = False
    cmd = 'show_platform_sysdb'
    show_plat = cmd_op(cmd)
    for line in show_plat:
        if (line.find('N560-RSP4') != -1):
            rplc = True
        # BigBend
        if (line.find('N540-28Z4C-') != -1):
            rplc = True

    return rplc


def count_vm_npu(vm):
    cmd = 'show_platform_sysdb'
    show_plat = cmd_op(cmd)

    node_str = vm.split(' ', 1)[0]
    node = node_str.split('/')[1]

    search_str1 = '0/RP%s/NPU' % (node)
    search_str2 = '0/%s/NPU' % (node)
    npu = 0;
    for line in show_plat:
        count = line.count(search_str1)
        if count:
            npu = npu + 1
        count = line.count(search_str2)
        if count:
            npu = npu + 1

    return npu


def find_node_id():
    node_id = '4096'
    cmd = 'show_sdr_sysdb -m'
    node_det = cmd_op(cmd)
    if node_det:
        for line in node_det:
            if (line.find('SDR lead (Primary)') != -1):
                node_id = int(line.split(':')[1], 16)
                break

    return node_id


def check_for_kbp(node):
    node['kbp'] = True
    node['kbp_idx'] = 0
    cmd = 'fia_driver_show -c "diag pp kbp print" -u 0x0 -n %s' %(node['node'])
    for line in cmd_op(cmd):
        if (line.find('KBP Device is not yet installed') != -1):
            node['kbp'] = False
            break
        if (line.find('IPv6 MC') != -1):
            node['kbp_idx'] = line.split()[0]
            break
    # Kbp is present but ipv6 table is still in internal TCAM
    if (node['kbp_idx'] == 0):
        node['kbp'] = False


def find_platform():
    plat = {}

    rplc = check_for_rplc()
    if rplc:
        node = {}
        node['name'] = '0/RP0/CPU0'
        node['npu_cnt'] = 1
        node['node'] = find_node_id()
        plat['rplc'] = True
        check_for_kbp(node)
        plat[0] = node
        return plat

    plat['rplc'] = False
    cmd = 'show_platform_sysdb -v'
    vm_op = cmd_op(cmd)

    i = 0
    for vm in vm_op:
        if (vm.find('LC') != -1):
            node = {}
            name = vm.split(' ', 1)[0]
            node['name'] = name
            node['npu_cnt'] = count_vm_npu(vm)
            node_num = int(name.split('/')[1]) * 256
            node['node'] = node_num
            check_for_kbp(node)
            plat[i] = node
            i = i + 1

    return plat


# Makes dict with fec id as key and mcid as value,
# by filtering only the fecs used by multicast
def filter_mcast_fec(node, npu):
    global alloced_fec_tbl
    cmd = 'fia_driver_show -c "diag alloc fec info=1" -u 0x%d -n %s' %(npu, node['node'])
    for fec_line in cmd_op(cmd):
        if (fec_line.find('FEC-id:') != -1):
            fec_line = fec_line.split()
            mcid_val = fec_line[6]
            if (mcid_val[0:4] == '0x2c'):
                fec = fec_line[1]
                mcid_val = mcid_val[4:]
                alloced_fec_tbl[fec] = int(mcid_val, 16)


# Makes dict with fec id as key and mcid as value,
# by filtering only the fecs used by multicast
# Also builds a reverse dict
def filter_mcast_fec_and_mcid(node, npu):
    global alloced_fec_tbl
    cmd = 'fia_driver_show -c "diag alloc fec info=1" -u 0x%d -n %s' %(npu, node['node'])
    for fec_line in cmd_op(cmd):
        if (fec_line.find('FEC-id:') != -1):
            fec_line = fec_line.split()
            mcid_val = fec_line[6]
            if (mcid_val[0:4] == '0x2c'):
                fec = fec_line[1]
                mcid_val = int(mcid_val[4:], 16)
                alloced_fec_tbl[fec] = mcid_val
                mcids_inhw_tbl[mcid_val] = fec


# It does this by reading the op of show mrib ipv6 fgid info or
# show mrib fgid info
def find_s_g_of_mcid(mcid):
    sg = []
    cmd = 'mrib_show_stats -i %s' % mcid
    ipv4_op = cmd_op(cmd)
    if (len(ipv4_op) > 2):
        for line in ipv4_op:
            if (line.find('Context         : IP') != -1):
                line = ''.join(line.split())
                line = line.split(",")
                sg.extend(['IP', line[1], line[2]])
                sg[2] = sg[2].replace(')', '')
            elif (line.find('Context         : Label') != -1):
                line = line.split()
                sg.extend(['LABEL', line[3]])
            elif (line.find('Members[ref]    :') != -1):
                members = line.split(":")[1]
                if (members != ' No FAP associated with this FGID'):
                    sg.append(1)
                else:
                    sg.append(0)
                break
        return sg

    cmd = 'mrib6_show_stats -i %s' % mcid
    ipv6_op = cmd_op(cmd)
    if (len(ipv6_op) > 2):
        for line in ipv6_op:
            if (line.find('Context         : IP') != -1):
                line = ''.join(line.split())
                line = line.split(",")
                sg.extend(['IP', line[1], line[2]])
                sg[2] = sg[2].replace(')', '')
            elif (line.find('Context         : Label') != -1):
                line = line.split()
                sg.extend(['LABEL', line[3]])
            elif (line.find('Members[ref]    :') != -1):
                members = line.split(":")[1]
                if (members != ' No FAP associated with this FGID'):
                    sg.append(1)
                else:
                    sg.append(0)
                break

    return sg;


# Read the entries from IPv4 MC KAPS table
# and form a table having fec as the key
def form_ipv4_kaps_table(node, npu):
    global ipv4_kaps_tbl
    ipv4_tid = None

    cmd = 'fia_driver_show -c "diag dbal ti" -u 0x%d -n %s' %(npu, node['node'])
    for tid in cmd_op(cmd):
        if (tid.find('IPv4 MC KAPS') != -1):
            ipv4_tid = tid
            break

    if ipv4_tid:
        result = re.search('\((.*)\)', ipv4_tid)
        tid = result.group(1)

        cmd = 'fia_driver_show -c "diag dbal te %s" -u 0x%d -n %s' %(tid, npu, node['node'])
        ipv4_kaps_entries = cmd_op(cmd)

        for line in ipv4_kaps_entries:
            tmp_line = ''.join(line.split())
            tmp_line = tmp_line.split("|")
            if (len(tmp_line) > 7):
                result = re.search('0x(.*)/', tmp_line[7])
                fec_val = hex(int(result.group(1), 16) & 0x1ffff)
                if fec_val in ipv4_kaps_tbl:
                    # Filter out catch all fecs
                    if not ((tmp_line[4] == '0x00000000/00/28') and (tmp_line[5] == '0x00000000/00/32')):
                        print "fec is already existing in the table appending the line\n: %s" %(line)
                    ipv4_kaps_tbl[fec_val].append(line)
                else:
                    ipv4_kaps_tbl[fec_val] = [line]


def form_ipv6_kaps_table(node, npu):
    global ipv6_kaps_tbl
    ipv6_tid = None

    cmd = 'fia_driver_show -c "diag dbal ti" -u 0x%d -n %s' %(npu, node['node'])
    for tid in cmd_op(cmd):
        if (tid.find('IPv6 MC KAPS') != -1):
            result = re.search('\((.*)\)', tid)
            ipv6_tid = result.group(1)
            break

    if ipv6_tid:
        cmd = 'fia_driver_show -c "diag dbal te %s" -u 0x%d -n %s' %(ipv6_tid, npu, node['node'])
        ipv6_kaps_entries = cmd_op(cmd)

        for line in ipv6_kaps_entries:
            tmp_line = ''.join(line.split())
            tmp_line = tmp_line.split("|")
            if (len(tmp_line) == 9):
                result = re.search('0x(.*)/', tmp_line[7])
                comp_val = result.group(1)
                # Skip the default ipv6 entries
                if (comp_val[0] == '2'):
                    compid = int(int(comp_val, 16) & 0xffff)
                    if compid in ipv6_kaps_tbl:
                        # Filter out catch all fecs
                        print "compid is already existing in the table appending the line\n: %s" %(line)
                        ipv6_kaps_tbl[compid].append(tmp_line)
                    else:
                        ipv6_kaps_tbl[compid] = tmp_line

    if (node['kbp'] == True):
        cmd = 'fia_driver_show -c "diag pp kbp iterator table=%s" -u 0x%d -n %s' %(node['kbp_idx'], npu, node['node'])
        ipv6_kaps_entries = cmd_op(cmd)
        collect_fec = False
        for line in ipv6_kaps_entries:
            tmp_line = line.split()
            if (len(tmp_line) == 9):
                data_line = tmp_line
                collect_fec = True
            else:
                if collect_fec:
                    collect_fec = False
                    if (line.find('Associate Data') != -1):
                        fec_val = tmp_line[3].split('=')
                        fec_val = fec_val[1][3:8]
                        fec_val = hex(int(fec_val, 16) & 0x1ffff)

                        ipv6_data = [data_line[7].split('=')[1], data_line[6].split('=')[1], data_line[5].split('=')[1]]

                        if fec_val in ipv6_kaps_tbl:
                            print "fec is already existing in the table appending the line\n: %s" %(data_line)
                            ipv6_kaps_tbl[fec_val].append(ipv6_data)
                        else:
                            ipv6_kaps_tbl[fec_val] = ipv6_data


def form_ipv6_compid_table(node, npu):
    global ipv6_compid_tbl
    ipv6_tid = None

    cmd = 'fia_driver_show -c "diag field tables" -u 0x%d -n %s' %(npu, node['node'])
    for tid in cmd_op(cmd):
        if (tid.find('IPv6_sip_low                       | dest') != -1):
            ipv6_tid = tid.split()[1]
            break

    if ipv6_tid:
        cmd = 'fia_driver_show -c "diag field tables db=%s" -u 0x%d -n %s' %(ipv6_tid, npu, node['node'])
        ipv6_compid_entries = cmd_op(cmd)

        fec_idx = -1
        for line in ipv6_compid_entries:
            tmp_line = ''.join(line.split())
            tmp_line = tmp_line.split("|")
            if (len(tmp_line) >= 10):
                if (fec_idx == -1):
                    if ('Actions' in tmp_line):
                        continue
                    if ('dest' in tmp_line):
                        fec_idx = tmp_line.index('dest')
                        continue
                else:
                    fec_val = hex(int(tmp_line[fec_idx], 16) & 0x1ffff)
                    ipv6_comp_entry = [tmp_line[3], tmp_line[2], int(tmp_line[4], 16)]
                    if fec_val in ipv6_compid_tbl:
                        print "fec is already existing in the table appending the line\n: %s" %(line)
                        ipv6_compid_tbl[fec_val].append(ipv6_comp_entry)
                    else:
                        ipv6_compid_tbl[fec_val] = ipv6_comp_entry


def extract_ipv4_kaps_tbl_data(fec):
    global ipv4_kaps_tbl
    fec_data_sg = []

    line = str(ipv4_kaps_tbl[fec])
    line = ''.join(line.split())
    line = line.split("|")
    if (len(line) > 7):
        fec_data_sg = [line[i] for i in (4, 5, 7)]

        for i, s in enumerate(fec_data_sg):
            result = re.search('0x(.*?)/', s)
            val = result.group(1)
            #Add oxe to group
            if (i == 0):
                val = int(val, 16) | 0xe0000000
                val = socket.inet_ntoa(struct.pack(">L", val))
                fec_data_sg[i] = val
            elif (i == 1):
                val = int(val, 16)
                val = socket.inet_ntoa(struct.pack(">L", val))
                fec_data_sg[i] = val
            else:
                fec_data_sg[i] = fec;

        fec_data_sg.insert(0, 'v4')
        vrf = re.search('0x(.*?)/', line[3]).group(1)
        fec_data_sg.append(int(vrf, 16))


    return fec_data_sg


def extract_ipv6_kaps_tbl_data_int_tcam(fec):
    global ipv6_kaps_tbl, ipv6_compid_tbl
    fec_data_sg = []

    ipv6_src_info = ipv6_compid_tbl[fec]
    compid = ipv6_src_info[2]

    if compid in ipv6_kaps_tbl:
        ipv6_grp_info = ipv6_kaps_tbl[compid]
        fec_data_sg.append('v6')
        fec_data_sg.extend([ipv6_src_info[0], ipv6_src_info[1]])
        fec_data_sg.append(ipv6_grp_info[1][6:38])
        vrf = re.search('0x(.*?)/', ipv6_grp_info[3]).group(1)
        fec_data_sg.append(int(vrf, 16))

    return fec_data_sg


def extract_ipv6_kaps_tbl_data_with_kbp(fec):
    global ipv6_kaps_tbl
    fec_data_sg = []

    ipv6_grp_info = ipv6_kaps_tbl[fec]
    fec_data_sg.append('v6')
    fec_data_sg.extend([ipv6_grp_info[0], '0', ipv6_grp_info[1], int(ipv6_grp_info[2], 16)])

    return fec_data_sg


def resolve_fec(node, npu, fec):
    fec_data_sg = []

    if fec in ipv4_kaps_tbl:
        return extract_ipv4_kaps_tbl_data(fec)

    if (node['kbp'] == False):
        if fec in ipv6_compid_tbl:
            return extract_ipv6_kaps_tbl_data_int_tcam(fec)

    if (node['kbp'] == True):
        if fec in ipv6_kaps_tbl:
            return extract_ipv6_kaps_tbl_data_with_kbp(fec)

    return fec_data_sg


def print_fec_details(fec, sg, hw):
    if (len(sg) == 0):
        print "!!!!!!!!EMPTY FEC:%s MCID:%s" % (fec[0], fec[1])
        if (len(hw) != 0):
            if (hw[0] == 'v4'):
                print "fec:%s mcid:%s tbldecode s:%s g:%s vrf:%d fec:%s" \
                       % (fec[0], fec[1], hw[2], hw[1], hw[4], hw[3])
            if (hw[0] == 'v6'):
                print "fec:%s mcid:%s tbldecode srch:%s srcl:%s grp:%s vrf:%d" \
                       % (fec[0], fec[1], hw[1], hw[2], hw[3], hw[4])
    else:
        if (len(hw) != 0):
            if (sg[0] == 'IP'):
                if sg[3]:
                    e = 'no'
                else:
                    e = 'yes'

                if (hw[0] == 'v4'):
                    print "fec:%s mcid:%s empty:%s s:%s g:%s tbldecode s:%s g:%s vrf:%d fec:%s" \
                           % (fec[0], fec[1], e, sg[1], sg[2], hw[2], hw[1], hw[4], hw[3])
                if (hw[0] == 'v6'):
                    print "fec:%s mcid:%s empty:%s s:%s g:%s tbldecode srch:%s srcl:%s grp:%s vrf:%d" \
                           % (fec[0], fec[1], e, sg[1], sg[2], hw[1], hw[2], hw[3], hw[4])
            if (sg[0] == 'LABEL'):
                if sg[2]:
                    e = 'no'
                else:
                    e = 'yes'

                if (hw[0] == 'v4'):
                    print "fec:%s mcid:%s empty:%s label:%s tbldecode s:%s g:%s vrf:%d fec:%s" \
                           % (fec[0], fec[1], e, sg[1], hw[2], hw[1], hw[4], hw[3])
                if (hw[0] == 'v6'):
                    print "fec:%s mcid:%s empty:%s label:%s tbldecode srch:%s srcl:%s grp:%s vrf:%d" \
                           % (fec[0], e, fec[1], sg[1], hw[1], hw[2], hw[3], hw[4])
        else:
            if (sg[0] == 'IP'):
                if sg[3]:
                    e = 'no'
                else:
                    e = 'yes'

                print "fec:%s mcid:%s empty:%s s:%s g:%s"  % (fec[0], fec[1], e, sg[1], sg[2])
            if (sg[0] == 'LABEL'):
                if sg[2]:
                    e = 'no'
                else:
                    e = 'yes'

                print "fec:%s mcid:%s empty:%s label:%s"  % (fec[0], fec[1], e, sg[1])


def display_fec(node, npu):
    global alloced_fec_tbl
    for fec in alloced_fec_tbl:
        mcid = alloced_fec_tbl[fec]
        sg = find_s_g_of_mcid(mcid)
        fecl = [fec, mcid]
        hwdata = resolve_fec(node, npu, fec)
        print_fec_details(fecl, sg, hwdata)


def print_zombie_fec_vals(node, npu):
    global alloced_fec_tbl, ipv4_kaps_tbl, ipv6_compid_tbl, ipv6_kaps_tbl

    for fec in ipv4_kaps_tbl:
        if fec not in alloced_fec_tbl:
            hwdata = extract_ipv4_kaps_tbl_data(fec)
            print "A fec:%s found only present in HW:" % (fec), hwdata

    if (node['kbp'] == False):
        for fec in ipv6_compid_tbl:
            if fec not in alloced_fec_tbl:
                hwdata = extract_ipv6_kaps_tbl_data_int_tcam(fec)
                print "A fec:%s found only present in HW:" % (fec), hwdata

    if (node['kbp'] == True):
        for fec in ipv6_kaps_tbl:
            if fec not in alloced_fec_tbl:
                hwdata = extract_ipv6_kaps_tbl_data_with_kbp(fec)
                print "A fec:%s found only present in HW:" % (fec), hwdata


def is_empty_fec_present(node, npu):
    global mcdb, alloced_fec_tbl
    empty_mcid = 0

    for fec in alloced_fec_tbl:
        mcid = alloced_fec_tbl[fec]
        if mcid not in mcdb:
            sg = []
            fecl = [fec, mcid]
            empty_mcid += 1
            hwdata = resolve_fec(node, npu, fec)
            print_fec_details(fecl, sg, hwdata)
        else:
            if (mcdb[mcid] == None):
                mcdb[mcid] = fec
            else:
                print "Duplicate mcid :%d" % (mcid)
                sg = find_s_g_of_mcid(mcid)
                fecl = [fec, mcid]
                empty_mcid += 1
                hwdata = resolve_fec(node, npu, fec)
                print_fec_details(fecl, sg, hwdata)

                oldfec = mcdb[mcid]
                hwdata = resolve_fec(node, npu, oldfec)
                fecl = [oldfec, mcid]
                print_fec_details(fecl, sg, hwdata)

                mcdb[mcid] = fec


    return empty_mcid


def show_routes_not_in_hw(node, npu):
    global mcdb, alloced_fec_tbl, mcids_inhw_tbl

    mrib_mcid_cnt = 0
    routes_not_in_hw_cnt = 0
    for mrib_mcid in mcdb:
        mrib_mcid_cnt = mrib_mcid_cnt + 1
        if mrib_mcid not in mcids_inhw_tbl:
            routes_not_in_hw_cnt = routes_not_in_hw_cnt + 1
            fecl = [0, mrib_mcid]
            sg = find_s_g_of_mcid(mrib_mcid)
            hwdata = []
            print_fec_details(fecl, sg, hwdata)

    print "routes in mrib:%d, routes not in hw:%d" % (mrib_mcid_cnt, routes_not_in_hw_cnt)


# forms mcdb dict of all the ipv4 and ipv6 mcids
def form_mcid_db():
    global mcdb
    check_mcid = False

    cmd = 'mrib_show_stats -i 0x0'
    ipv4_fgid_str = cmd_op(cmd)
    for line in ipv4_fgid_str:
        if (line.find('FGID (type)     :') != -1):
            mcid = int(line.split()[3])
            check_mcid = True

        if check_mcid:
            if (line.find('Members[ref]    :') != -1):
                members = line.split(":")[1]
                check_mcid = False
                if (members != ' No FAP associated with this FGID'):
                    mcdb[mcid] = None

    cmd = 'mrib6_show_stats -i 0x0'
    ipv6_fgid_str = cmd_op(cmd)
    for line in ipv6_fgid_str:
        if (line.find('FGID (type)     :') != -1):
            mcid = int(line.split()[3])
            check_mcid = True

        if check_mcid:
            if (line.find('Members[ref]    :') != -1):
                members = line.split(":")[1]
                check_mcid = False
                if (members != ' No FAP associated with this FGID'):
                    mcdb[mcid] = None


def filter_nodes(plat):
    global location

    for k, node in plat.items():
        if isinstance(node, dict):
            if (node['node'] != location):
                del plat[k]


def print_node(node):
    print "====================================="
    print "\tVerifying in %s" % node['name']
    print "====================================="


def print_npu(npu):
    print "\nSearching NPU%d:" % (npu)


def main():
    global print_fec, verify_fec, mcdb, location, only_on_npu
    empty_mcid = 0

    parse_args()

    # Extract the details like
    # Number of LCs and NPUs
    # Whether external TCAM is configured
    # is the platform have only RPLC vm
    plat = find_platform()

    if location != -1:
        filter_nodes(plat)

    if print_fec:
        for k, node in plat.iteritems():
            if isinstance(node, dict):
                print_node(node)
                npus = node['npu_cnt']
                for npu in range(int(npus)):
                    if (only_on_npu != -1):
                        if (npu != only_on_npu):
                            continue
                    print_npu(npu)
                    # Form all the databases for look up on that NPU
                    form_ipv4_kaps_table(node, npu)
                    form_ipv6_kaps_table(node, npu)
                    form_ipv6_compid_table(node, npu)
                    filter_mcast_fec(node, npu)

                    display_fec(node, npu)
                    if print_zombie_fec:
                        print_zombie_fec_vals(node, npu)

                    #Clear all the databases for that NPU
                    ipv4_kaps_tbl.clear()
                    ipv6_compid_tbl.clear()
                    ipv6_kaps_tbl.clear()
                    alloced_fec_tbl.clear()

    if verify_fec:
        for k, node in plat.iteritems():
            if isinstance(node, dict):
                print_node(node)
                npus = node['npu_cnt']
                for npu in range(int(npus)):
                    if (only_on_npu != -1):
                        if (npu != only_on_npu):
                            continue
                    print_npu(npu)
                    # Form all the databases for look up on that NPU
                    form_mcid_db()
                    form_ipv4_kaps_table(node, npu)
                    form_ipv6_kaps_table(node, npu)
                    form_ipv6_compid_table(node, npu)
                    filter_mcast_fec(node, npu)

                    if not plat['rplc']:
                        empty_mcid += is_empty_fec_present(node, npu)
                    else:
                        print "This platform don't have fabric. So no empty mcids here !!!!"

                    if print_zombie_fec:
                        print_zombie_fec_vals(node, npu)

                    #Clear all the databases for that NPU
                    ipv4_kaps_tbl.clear()
                    ipv6_compid_tbl.clear()
                    ipv6_kaps_tbl.clear()
                    alloced_fec_tbl.clear()
                    mcdb.clear()

        if empty_mcid == 0:
            print "\nThere are no empty mcids present in ingress tables"

    if list_routes_not_in_hw:
        form_mcid_db()
        for k, node in plat.iteritems():
            if isinstance(node, dict):
                print_node(node)
                npus = node['npu_cnt']
                for npu in range(int(npus)):
                    if (only_on_npu != -1):
                        if (npu != only_on_npu):
                            continue
                    print_npu(npu)
                    # Form all the databases for look up on that NPU
                    form_ipv4_kaps_table(node, npu)
                    form_ipv6_kaps_table(node, npu)
                    form_ipv6_compid_table(node, npu)
                    filter_mcast_fec_and_mcid(node, npu)

                    show_routes_not_in_hw(node, npu)

                    #Clear all the databases for that NPU
                    ipv4_kaps_tbl.clear()
                    ipv6_compid_tbl.clear()
                    ipv6_kaps_tbl.clear()
                    alloced_fec_tbl.clear()
                    mcids_inhw_tbl.clear()


if __name__ == "__main__":
    main()
