'''
Created on Apr 12, 2015

@author: Puneet Garg

Copyright (c) 2015 by Cisco Systems, Inc.
All rights reserved.

Classes used for Interface.
'''

from devpkg.utils.util import asciistr
from devpkg.base.simpletype import SimpleType
from devpkg.base.command_interaction import CommandInteraction
from devpkg.utils.state_type import State
from parsers import interface_executor
from fmc.parsers import delete_interface_executor, device_object_executor, acpolicy_update_description, delete_device_object_executor, dispatch_executor
from devpkg.base.specilized_param.command_param_value import CommandParamValue
from devpkg.base.dmobject import DMObject
from fmc.security_zone import SecurityZoneInterfaceBridgeGroup, SecurityZone
from devpkg.base.dmlist import DMList
from fmc.config_keeper import ConfigKeeper
from static_route import StaticRoute
from interface import NGIPSInterfaceConfig
from fmc.acrule import AccessRule, acrule_update_comments
from devpkg.base.command_service import CommandService
from fmc.selective_probe import SelectiveProbe

class BGIInterface(DMObject):
    """
    BGIInterface Object that stores each interface
    """
    def __init__(self, instance, probe):
        super(BGIInterface, self).__init__(instance, dm_key="BGIInterface", probe=probe)
        self.register_child(SimpleType('InterfaceHolder', 'InterfaceHolder', defaults={}, param_req_formater=CommandInteraction.DO_NOT_ADD_TO_PUT_POST_JSON))
        self.probe = probe

class BGIInterfaces(SimpleType):
    """
    BGIInterfaces that stores the DMList of BGIInterface
    """
    def __init__(self, probe = None):
        super(BGIInterfaces, self).__init__('Interfaces', 'interfaces', defaults={}, param_req_formater=CommandInteraction.DO_NOT_ADD_TO_PUT_POST_JSON)
        self.register_child(DMList('Interface', BGIInterface, probe=probe))
        self.probe = probe

    def ifc2dm(self, no_dm_cfg_stack,  dm_cfg_list):
        super(BGIInterfaces, self).ifc2dm(no_dm_cfg_stack,  dm_cfg_list)
   
class BridgeGroupInterface(DMObject):
    '''
    Interface related configuration

    '''
    GENERAL_INTERFACE = """
    {
      "type": "BridgeGroupInterface",
      "bridgeGroupId": 16,
      "enabled": false,
      "MTU": 1500,
      "managementOnly": false,
      "name": "BVI1"
    }
    """
    BVI_NAME = "BVI"

    def __init__(self, instance, probe):
        super(BridgeGroupInterface, self).__init__(instance, dm_key="BridgeGroupInterface", probe=probe)
        self.dm_key_command = 'fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/logicalinterfaces/<id>'
        self.command_executor = device_object_executor

        'Must be in the order in which to generate the command.'
        self.register_child(SimpleType('bridge_group_id', 'bridgeGroupId', defaults='1'))
        self.register_child(BGIInterfaces(probe=probe))
        self.register_child(SimpleType('IPv4Config', 'ipv4', defaults={}, param_req_formater=CommandInteraction.DO_NOT_ADD_TO_PUT_POST_JSON, probe=probe))
        self.register_child(StaticRoute(probe))
        self.dependent_on = []
        self.probe = probe

    def ifc2dm(self, no_dm_cfg_stack, dm_cfg_list):
        'Populate current instance recursive state based on children and dependent objects.'
        try:
            interfaces = self.find('Interfaces')
            interfaces = interfaces.get_value()['Interface']
            for interface in interfaces:
                self.dependent_on.append("/InterfaceConfig/%s" % interface['InterfaceHolder'])
        except:
            pass
        self.recursive_state = self.get_state_recursive()


        local_dm_cfg_list = []

        if not self.has_ifc_delta_cfg():
            return

        num, pol, name = self.delta_ifc_key

        # Process for commands if state modified.
        if self.get_state() == State.CREATE or self.get_state() == State.MODIFY:
            super(BridgeGroupInterface, self).ifc2dm(no_dm_cfg_stack, local_dm_cfg_list)

            # Add command for later execution.
            # We may have more than one item in local_dm_cfg_list.
            for item in local_dm_cfg_list:
                if item.toDict().has_key('command') and item.command == 'IPv4StaticRoute':
                    if not item in (dm_cfg_list):
                        self.add_cli_to_array(item, dm_cfg_list)
                elif item.params.__len__() > 0:
                    'Set Rest function and executor'
    
                    # set BVI's name, append _Tenant_Device to it to make sure it is unique
                    bvi_group_id = item.params['bridge_group_id'].param_value
                    # FMC 6.2.3 added validation checks in the REST APIs on bridge group interfaces:
                    # 1. name field in the REST has become read-only and it has to be 'BVI'+bvi_group_id
                    # 2. ifname field is not allowed in transparent mode
                    item.add_data_param(BridgeGroupInterface.BVI_NAME + bvi_group_id, "name", paramType=CommandParamValue)
                    # Unlike transparent mode, BVI in routed mode requires nameif, so we'll put nameif first
                    # As there is no FMC REST to tell if a target FTD is in transparent or routed firewall mode, we'll workaround this
                    # by catch the error when sending the CLI to a transparent FTD, modify the payload by removing the nameif and re-send.
                    # TODO: we need to re-visit and adjust the code here after FMC 6.3 where we'll have API to check the FTD mode first.
                    ftd_mode = self.probe.config_keeper.get('ftd_mode')
                    if ftd_mode != 'TRANSPARENT':
                        item.add_data_param(BridgeGroupInterface.BVI_NAME + bvi_group_id + '_' + self.get_top_uuid(), 'ifname')
    
                    item.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/bridgegroupinterfaces", BridgeGroupInterface.BVI_NAME + bvi_group_id, self.command_executor, BridgeGroupInterface.GENERAL_INTERFACE)
                    item.add_data_param(True, "enabled")
                    item.add_param("description", self.get_top_uuid(), acpolicy_update_description, "", "description")
                    if item.params.has_key('IPv4Config'):
                        ipv4 = {}
                        if item.params['IPv4Config'].param_value.has_key('static'): #we need to change static from what is there and what we have
                            addr = item.params['IPv4Config'].param_value['static']['address']
                            if '/' in addr: #we have xxx.xxx.xxx.xxx/xx
                                addr_split = addr.split('/')
                                ipv4 = {"static":{"address":addr_split[0], "netmask":addr_split[1]}}
                            else:
                                ipv4 = item.params['IPv4Config'].param_value['static']
                        
                        item.add_data_param(ipv4, 'ipv4')
                    item.add_data_param('BridgeGroupInterface', 'type')
                    #get and add all appropriate interfaces
                    try:
                        selectedInterface = []
                        interfaces_in_bgi = []
                        interfaces = item.params['Interfaces'].param_value
                        if not isinstance(interfaces['Interface'], list):
                            interfaces['Interface'] = [interfaces['Interface']]
                        for interface in interfaces['Interface']:
                            holder = self.find("/InterfaceConfig/%s" % interface['InterfaceHolder'])
                            inter = holder.cli
                            inter_all = self.find("Interfaces/Interface")
                            inter_name = inter.get_name() if inter is not None else holder.vif
                            for inter_state in inter_all:
                                if inter_state.get_value()['InterfaceHolder'] == interface['InterfaceHolder']:
                                    inter_state = inter_state.get_state()
                                    break
                            interfaces_in_bgi.append(self.find("/InterfaceConfig/%s" % interface['InterfaceHolder']))
                            if inter is not None and (inter.state == State.DESTROY or inter_state == State.DESTROY):
                                item.add_removable('selectedInterfaces', inter_name)
                            else:
                                # this is the type being used in selected interfaces, 'PhysicalInterface' is for regular physical interface and subinterfaces
                                intf_type = "EtherChannelInterface" if holder.dm_key == "EtherChannelInterface" else "PhysicalInterface"
                                if inter is None or inter.command == 'IPv4StaticRoute':
                                    # inter is not defined or IPv4 static route is configured for this interface
                                    if holder.dm_key == 'SubInterface':
                                        if not inter_name:
                                            inter_name = holder.vif # if inter_name is None, get the name from vif
                                        inter_name2 = inter_name + '.' + asciistr(holder.vlan)
                                        intf_id = self.probe.config_keeper.get_id_from_probe(inter_name2, 'subinterfaces')
                                        if len(intf_id) == 0:
                                            # observed in virtual FTD even solid interface is having 'SubInterface' in dm_key
                                            # so in this case we cannot get intf_id from above call and we'd do the following instead 
                                            intf_id = self.probe.config_keeper.get_id_from_probe(inter_name, 'physicalinterfaces')
                                        else:
                                            inter_name = inter_name2
                                    elif holder.dm_key == 'EtherChannelInterface':
                                        intf_id = self.probe.config_keeper.get_id_from_probe(inter_name, 'etherchannelinterfaces')
                                    else:
                                        intf_id = self.probe.config_keeper.get_id_from_probe(inter_name, 'physicalinterfaces')
                                    if intf_id is not None and len(intf_id) > 0:
                                        selectedInterface.append({"type":intf_type, "name":inter_name, "id": intf_id})
                                    else:
                                        # We'll try to get the intf_id later through idParam call
                                        selectedInterface.append({"type": intf_type, "name":inter_name2, "id":inter.idParam})
                                else:
                                    selectedInterface.append({"type":intf_type, "name":inter_name, "id":inter.idParam})
                    except:
                        pass
                    item.add_data_param(selectedInterface, "selectedInterfaces")
    
                    # check overlapped selected interfaces
                    # There is no before-change info, we don't know if the end user changed the BVI_ID or not.
                    # We search probe to see if any selected interface has been used in the bridge group interface on FMC
                    # if any, delete that bridge interface first before creating new one
                    is_deleting_bvi = False
                    if self.get_state() == State.MODIFY:
                        selected_interface_ids = set(intf['id'] for intf in selectedInterface)
                        for bvi_interface in self.probe.config_keeper.get('bridgegroupinterfaces'):
                            bvi_selected_interface = bvi_interface['selectedInterfaces'] if bvi_interface.has_key('selectedInterfaces') else {}
                            # if the selected interface is already selected on probe and BVI_id is different, delete the existing BVI and create new one
                            if bvi_group_id != asciistr(bvi_interface['bridgeGroupId']) and [d for d in bvi_selected_interface if d['id'] in selected_interface_ids]:
                                local_dm_cfg_list_del = []
                                clii = CommandInteraction(self.dm_key, model_key=self.get_config_path(), probe=self.probe)
                                name = BridgeGroupInterface.BVI_NAME + asciistr(bvi_interface['bridgeGroupId'])
                                clii.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/bridgegroupinterfaces", name, delete_interface_executor, "")
                                no_dm_cfg_stack.append(clii)
                                self.set_state(State.DESTROY) # DELETE old BGI
                                is_deleting_bvi = True
                                self.add_cli_to_array(clii, dm_cfg_list)
                                # After the BVI is deleted, the security zone associated with the member interface will get lost, we need to put it back here
                                for selected_intf in selectedInterface:
                                    is_vlan_used = self.get_top().is_vlan_supported()
                                    is_virtual = self.get_top().is_virtual()

                                    intf_clii = CommandInteraction(selected_intf['type'], model_key=selected_intf['type'], probe=self.probe)
                                    interface_type = 'physicalinterfaces' if selected_intf['type'] == 'PhysicalInterface' else 'etherchanelinterfaces'

                                    # if vlan is supported(physical interface or virtual with vlan truncking)
                                    if (is_virtual is False or (is_virtual and is_vlan_used)):
                                        interface_type = 'subinterfaces'
                                        url = "fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/" + interface_type
                                        intf_clii.add_basic_interaction(url, selected_intf['name'].split(".")[0], interface_executor, NGIPSInterfaceConfig.GENERAL_INTERFACE)
                                        intf_clii.idParam.param_value['vlanId'] = selected_intf['name'].split('.')[1]
                                        intf_clii.add_data_param('SubInterface', 'type')
                                    else:
                                        url = "fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/" + interface_type
                                        intf_clii.add_basic_interaction(url, selected_intf['name'], device_object_executor, "")

                                    intf_clii.add_data_param(True, 'enabled')
                                    securityZone = {}
                                    securityZone['type'] = 'SecurityZone'
                                    security_zone_id = self.probe.config_keeper.get_security_zone_id_from_interface_name(selected_intf['name'])
                                    if security_zone_id != '':
                                        securityZone['id'] = security_zone_id
                                        intf_clii.add_data_param(securityZone, 'securityZone')
                                        self.set_state(State.MODIFY) # PUT interface
                                        self.add_cli_to_array(intf_clii, dm_cfg_list)
                    elif self.get_state() == State.CREATE:
                        # when a new BGI is created, the associated security zone of the member interfaces will get lost because the sz mode was 'ROUTED', which should be 'SWITCHED' now
                        '''
                        We need to do the following to add the sz back
                            1. identify all member interfaces
                            2. For each member interface, identify the associated security zone
                            3. identity access rule that is using that security zone, remove it from the access rule
                            4. delete the security zone (which was in ROUTED mode)
                            5. re-create the security zone using SWITCHED mode, for example,
                                {
                                  "name": "InternalSZRT_Tenant_Device",
                                  "type": "SecurityZone",
                                  "interfaceMode": "SWITCHED"
                                }
                            6. re-associate the security zone to access rule
                            7. re-associate the security zone to that member interface
                        '''

                        # get associated security zones
                        associated_szs = self.probe.config_keeper.get_associated_securityzones_of_interfaces(selectedInterface)
                        all_switched_mode = True
                        for sz in associated_szs:
                            if sz.has_key('interfaceMode') and sz['interfaceMode'] != 'SWITCHED':
                                all_switched_mode = False
                                break
                        sz2intf = {}
                        if not all_switched_mode:
                            for sz in associated_szs:
                                if sz.has_key('interfaces'):
                                    sz2intf[sz['name']] = sz['interfaces']
                            accessrules = self.probe.config_keeper.get_accessrules_with_securityzones(associated_szs)
                            self.remove_securityzones_from_accessrules(accessrules, associated_szs)
                            szs_to_be_deleted = self.delete_securityzones(associated_szs)
                            for sz_to_delete in szs_to_be_deleted:
                                no_dm_cfg_stack.append(sz_to_delete)
                            szs_to_be_added = self.add_securityzones(associated_szs, sz2intf, 'SWITCHED')
                            for sz_to_add in szs_to_be_added:
                                self.add_cli_to_array(sz_to_add, dm_cfg_list)
                            rules_to_be_updated = self.add_securityzones_to_accessrules(accessrules, associated_szs)
                            for rule_to_update in rules_to_be_updated:
                                self.add_cli_to_array(rule_to_update, dm_cfg_list)
                            
                    # continue to add or modify BVI
                    if is_deleting_bvi:
                        self.set_state(State.CREATE) # POST new BGI
                    self.add_cli_to_array(item, dm_cfg_list)
                    
                    for interface in interfaces_in_bgi:
                        if interface.cli is not None:
                            if interface.cli.command == 'IPv4StaticRoute':
                                # In serviceModify case the interface may only have IPv4 route change so we bypass adding SecurityZoneInterfaceBridgeGroup here
                                continue
                            else:
                                dm_cfg_list.append(SecurityZoneInterfaceBridgeGroup(interface, interface.cli, probe=self.probe))
                
        elif self.get_state() == State.DESTROY:
            # We need to delete all IPv4 static routes first before deleting interface due to dependency
            all_routes = self.probe.config_keeper.get('ipv4staticroutes')
            bgi_name = None
            if isinstance(self.delta_ifc_key, tuple):
                key_id, key_name, bgi_name = self.delta_ifc_key # e.g.: (13, 'BridgeGroupInterface', u'BVI1')
                bgi_name += '_' + self.probe.ldev_id
            for route in all_routes:
                if bgi_name == route['interfaceName']:
                    route_id = route['id']
                    cld = self.get_child('StaticRoute').get_child('IPv4StaticRoute')
                    clii = CommandInteraction('IPv4StaticRoute', model_key=cld.get_config_path(), probe=self.probe)
                    clii.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/routing/ipv4staticroutes", route_id, delete_device_object_executor, "", id=route_id)
                    clii.add_data_param(bgi_name, 'interfaceName')
                    clii.add_data_param(self.probe.config_keeper.get_network_value_from_ipv4_route(route), 'network')
                    clii.add_data_param(self.probe.config_keeper.get_host_value_from_ipv4_route(route), 'gateway')
                    metric_value = int(route['metricValue']) if route.has_key('metricValue') and route['metricValue'] is not None else 1
                    clii.add_data_param(metric_value, 'metric')
                    isTunneled_value = route['isTunneled'] if route.has_key('isTunneled') else False
                    clii.add_data_param(isTunneled_value, 'isTunneled')
                    clii.command = 'IPv4StaticRoute'
                    clii.state = State.DESTROY
                    self.add_cli_to_array(clii, dm_cfg_list)
            clii = CommandInteraction(self.dm_key, model_key=self.get_config_path(), probe=self.probe)
            if self.children is not None and self.children['bridge_group_id'] is not None and self.children['bridge_group_id'].get_value() is not None:
                bvi_group_id = self.children['bridge_group_id'].get_value()
                name = BridgeGroupInterface.BVI_NAME + bvi_group_id
            clii.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/bridgegroupinterfaces", name, delete_interface_executor, "")
            no_dm_cfg_stack.append(clii)
            self.add_cli_to_array(clii, dm_cfg_list)
            
        else: # self.get_state() == State.NOCHANGE or no state
            # when we do mini-audit for IPv4StaticRoute, we need to keep the config from APIC so it won't be deleted during audit
            super(BridgeGroupInterface, self).ifc2dm(no_dm_cfg_stack, local_dm_cfg_list)
            for item in local_dm_cfg_list:
                if item.toDict().has_key('command') and item.command == 'IPv4StaticRoute':
                    if not item in (dm_cfg_list):
                        item.state = State.CREATE
                        self.add_cli_to_array(item, dm_cfg_list)                

    def found_securityzone_in_list(self, candidate_sz, sz_list):
        for sz in sz_list:
            if candidate_sz['name'] == sz['name']:
                return True
        return False

    def remove_securityzones_from_accessrules(self, accessrules, associated_szs):
        for rule in accessrules:
            clii = CommandInteraction('AccessRule', model_key='AccessRule', probe=self.probe)
            clii.add_data_param(rule['acUUID'], 'acUUID', False)
            clii.add_basic_interaction('fmc_config/v1/domain/<domainUUID>/policy/accesspolicies/<acUUID>/accessrules', rule['name'], device_object_executor, AccessRule.GENERAL_AC_RULE)
            clii.add_data_param(rule['name'], "name")
            clii.add_data_param("true", "enabled")

            for sz in rule['sourceZones']['objects']:
                if self.found_securityzone_in_list(sz, associated_szs):
                    clii.add_removable('sourceZones', sz['name'])
            for sz in rule['destinationZones']['objects']:
                if self.found_securityzone_in_list(sz, associated_szs):
                    clii.add_removable('destinationZones', sz['name'])
            clii.state = State.MODIFY
            executor = CommandService(ConfigKeeper.get_global(self.probe.config_keeper.get('top_uuid') + ':device'), [clii], dispatch_executor)
            executor.execute(False)
            
    def delete_securityzones(self, securityzones):
        szs_to_be_deleted = []
        for sz in securityzones:
            clii = CommandInteraction('SecurityZone', model_key='SecurityZone', probe=self.probe)
            clii.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/object/securityzones", sz['name'], delete_device_object_executor, SecurityZone.GENERAL_SECURITY_ZONE)
            clii.state = State.DESTROY
            szs_to_be_deleted.append(clii)
        return szs_to_be_deleted
        
    def add_securityzones(self, securityzones, sz2intf, mode):
        szs_to_be_created = []
        for sz in securityzones:
            clii = CommandInteraction('SecurityZone', model_key='SecurityZone', probe=self.probe)
            clii.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/object/securityzones", sz['name'], device_object_executor, SecurityZone.GENERAL_SECURITY_ZONE)
            if sz2intf.has_key(sz['name']):
                clii.add_data_param(sz2intf[sz['name']], 'interfaces')
            clii.add_data_param(mode, 'interfaceMode')
            clii.add_data_param(sz['name'], 'name')
            clii.state = State.CREATE
            szs_to_be_created.append(clii)
        return szs_to_be_created
        
    def add_securityzones_to_accessrules(self, accessrules, associated_szs):
        rules_to_be_updated = []
        for rule in accessrules:
            clii = CommandInteraction('AccessRule', model_key='AccessRule', probe=self.probe)
            clii.add_data_param(rule['acUUID'], 'acUUID', False)
            clii.add_basic_interaction('fmc_config/v1/domain/<domainUUID>/policy/accesspolicies/<acUUID>/accessrules', rule['name'], device_object_executor, AccessRule.GENERAL_AC_RULE)
            clii.add_data_param(rule['name'], "name")
            clii.add_data_param("true", "enabled")
            clii.add_param("new_comments", self.get_top_uuid(), acrule_update_comments, "", "newComments")

            all_src_zone_objects = rule['sourceZones']['objects']
            all_dest_zone_objects = rule['destinationZones']['objects']
            all_src_zone_updated = []
            all_dest_zone_updated = []
            for sz in associated_szs:
                found_in_src = False
                found_in_dest = False
                for src_zone in all_src_zone_objects:
                    if src_zone['name'] == sz['name']:
                        found_in_src = True
                        # clear 'id', so it can be updated later with a new sz_id after sz is created
                        all_src_zone_updated.append({'type': 'SecurityZone', 'name': sz['name'], 'id':''})
                if not found_in_src:
                    all_src_zone_updated.append({'type': 'SecurityZone', 'name': sz['name'], 'id':''})
                for dest_zone in all_dest_zone_objects:
                    if dest_zone['name'] == sz['name']:
                        found_in_dest = True
                        # clear 'id', so it can be updated later with a new sz_id after sz is created
                        all_dest_zone_updated.append({'type': 'SecurityZone', 'name': sz['name'], 'id':''})
                if not found_in_dest:
                    all_dest_zone_updated.append({'type': 'SecurityZone', 'name': sz['name'], 'id':''})
                    
            src_zones = {}
            dest_zones = {}
            src_zones['objects'] = all_src_zone_updated   
            dest_zones['objects'] = all_dest_zone_updated
            if len(src_zones) > 0:
                clii.add_data_param(src_zones, 'sourceZones')
            if len(dest_zones) > 0:
                clii.add_data_param(dest_zones, 'destinationZones')
            clii.state = State.MODIFY
            rules_to_be_updated.append(clii)
        return rules_to_be_updated
            
class BridgeGroupInterfaces(DMList):
    'A list of Interfaces'

    def __init__(self, name = BridgeGroupInterface.__name__, child_class = BridgeGroupInterface, probe=None):
        super(BridgeGroupInterfaces, self).__init__(name, child_class, probe=probe)
        self.probe = probe

    def ifc2dm(self, no_dm_cfg_stack,  dm_cfg_list):
        super(BridgeGroupInterfaces, self).ifc2dm(no_dm_cfg_stack,  dm_cfg_list)
