'''
Created on Nov 2, 2016

Copyright (c) 2016 by Cisco Systems, Inc.
All rights reserved.

@author: aahluwalia
'''
from devpkg.utils.util import asciistr
from devpkg.base.command_interaction import CommandInteraction
from devpkg.base.command_service import CommandService
import re
from fmc.config_keeper import ConfigKeeper
from devpkg.utils.util import get_top_uuid_from_device

class Probe(object):
    """
    Class that Probes the FMC to find all data that is in it.
    """

    def __init__(self, ldev_id, network_object_group_probe=False):
        """
        creates all the clis requried to probe the fmc
        """
        self.has_run = False
        self.ldev_id = ldev_id
        self.network_object_group_probe = network_object_group_probe
        self.config_keeper = ConfigKeeper()
        self.config_keeper.set('top_uuid', ldev_id)
        self.init_holders()

    def probe_accesspolicies(self, device):
        self.acpolicy_probe = CommandInteraction("GetACPolicy", probe=self)
        self.cliis = [self.acpolicy_probe]
        self.acpolicy_probe.add_basic_interaction('fmc_config/v1/domain/<domainUUID>/policy/accesspolicies', None, None, None)
        self.acpolicy_probe.add_param("regex", ".*:%s:" % self.ldev_id, find_ac_policy_by_regex, None)
        if self.policy_name != '': # add poclicy_name to the GET parameter, so the GET URL will have ?name=policy_name
            self.acpolicy_probe.params['GET'].param_value['name'] = self.policy_name
        CommandService(device, self.cliis).execute(False)
        self.process_cliis(self.cliis)
        self.config_keeper.set('accesspolicies', self.accesspolicies)
        ConfigKeeper.set_global(self.ldev_id, self)

    def probe_accessrules(self, device):
        self.acrule_probe = CommandInteraction("GetAllACRules", probe=self)
        self.acrule_probe.add_basic_interaction('fmc_config/v1/domain/<domainUUID>/policy/accesspolicies/<acUUID>/accessrules', None, None, None)
        self.acrule_probe.add_data_param(self.acpolicy_probe.idParam, "acUUID")
        self.cliis = [self.acrule_probe]
        CommandService(device, self.cliis).execute(False)
        self.process_cliis(self.cliis)
        # We need to save the acUUID in accessrules in order to keep the relationship information
        updated_rules = []
        for rule in self.accessrules:
            if self.acpolicy_probe.idParam.param_uuid: # if there is no access policy associated, ie. during intial set up, no need to set value for updated_rules
                rule['acUUID'] = self.acpolicy_probe.idParam.param_uuid
                updated_rules.append(rule)
        self.accessrules = updated_rules
        self.config_keeper.set('accessrules', self.accessrules)
        ConfigKeeper.set_global(self.ldev_id, self)

    def probe_securityzones(self, device):
        self.sz_probe = CommandInteraction("GetAllSZ", probe=self)
        self.sz_probe.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/object/securityzones", None, None, None)
        cliis = [self.sz_probe]
        CommandService(device, cliis).execute(False)
        self.process_cliis(cliis)
        self.config_keeper.set('securityzones', self.securityzones)
        ConfigKeeper.set_global(self.ldev_id, self)
        
    def probe_inlinesets(self, device):
        self.inline_set_probe = CommandInteraction("GetAllIS", probe=self)
        self.inline_set_probe.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/inlinesets", None, None, None)
        cliis = [self.inline_set_probe]
        CommandService(device, cliis).execute(False)
        self.process_cliis(cliis)
        self.config_keeper.set('inlinesets', self.inlinesets)
        ConfigKeeper.set_global(self.ldev_id, self)
        
    def probe_physicalinterfaces(self, device):
        self.interface_probe = CommandInteraction("GetAllInterfaces", probe=self)
        self.interface_probe.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/physicalinterfaces", None, None, None)
        cliis = [self.interface_probe]
        CommandService(device, cliis).execute(False)
        self.process_cliis(cliis)
        self.config_keeper.set('physicalinterfaces', self.physicalinterfaces)
        ConfigKeeper.set_global(self.ldev_id, self)
        
    def probe_subinterfaces(self, device):
        self.subinterface_probe = CommandInteraction("GetAllSubInterfaces", probe=self)
        self.subinterface_probe.add_basic_interaction('fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/subinterfaces', None, None, None)
        cliis = [self.subinterface_probe]
        CommandService(device, cliis).execute(False)
        self.process_cliis(cliis)
        self.config_keeper.set('subinterfaces', self.subinterfaces)
        ConfigKeeper.set_global(self.ldev_id, self)

    def probe_etherchannels(self, device):
        self.etherchannel_probe = CommandInteraction("GetAllEtherChannelInterfaces", probe=self)
        self.etherchannel_probe.add_basic_interaction('fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/etherchannelinterfaces', None, None, None)
        cliis = [self.etherchannel_probe]
        CommandService(device, cliis).execute(False)
        self.process_cliis(cliis)
        self.config_keeper.set('etherchannelinterfaces', self.etherchannelinterfaces)
        ConfigKeeper.set_global(self.ldev_id, self)

    def probe_ipv4staticroutes(self, device):
        self.ipv4staticroute_probe = CommandInteraction("GetAllIPv4StaticRoutes", probe=self)
        self.ipv4staticroute_probe.add_basic_interaction('fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/routing/ipv4staticroutes', None, None, None)
        cliis = [self.ipv4staticroute_probe]
        CommandService(device, cliis).execute(False)
        self.process_cliis(cliis)
        self.config_keeper.set('ipv4staticroutes', self.ipv4staticroutes)
        ConfigKeeper.set_global(self.ldev_id, self)

    def probe_networks(self, device):
        self.network_probe = CommandInteraction("GetAllNetworks", probe=self)
        self.network_probe.add_basic_interaction('fmc_config/v1/domain/<domainUUID>/object/networks', None, None, None)
        cliis = [self.network_probe]
        CommandService(device, cliis).execute(False)
        self.process_cliis(cliis)
        self.config_keeper.set('networks', self.networks)
        ConfigKeeper.set_global(self.ldev_id, self)

    def probe_hosts(self, device):
        self.host_probe = CommandInteraction("GetAllHosts", probe=self)
        self.host_probe.add_basic_interaction('fmc_config/v1/domain/<domainUUID>/object/hosts', None, None, None)
        cliis = [self.host_probe]
        CommandService(device, cliis).execute(False)
        self.process_cliis(cliis)
        self.config_keeper.set('hosts', self.hosts)
        ConfigKeeper.set_global(self.ldev_id, self)

    def probe_bridgegroupinterfaces(self, device):
        self.bgi_probe = CommandInteraction("GetAllBGI", probe=self)
        self.bgi_probe.add_basic_interaction('fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/bridgegroupinterfaces', None, None, None)
        cliis = [self.bgi_probe]
        self.bridgegroupinterfaces = []
        CommandService(device, cliis).execute(False)
        self.process_cliis(cliis)
        self.config_keeper.set('bridgegroupinterfaces', self.bridgegroupinterfaces)
        ConfigKeeper.set_global(self.ldev_id, self)

    def probe_networkgroups(self, device):
        self.nog_probe = CommandInteraction("GetAllNOG", probe=self)
        if self.network_object_group_probe:
            self.nog_probe.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/object/networkgroups", None, None, None)
        cliis = [self.nog_probe]
        CommandService(device, cliis).execute(False)
        self.process_cliis(cliis)
        self.config_keeper.set('networkgroups', self.networkgroups)
        ConfigKeeper.set_global(self.ldev_id, self)

    def init_holders(self):
        self.data = []
        self.accesspolicies = []
        self.accessrules = []
        self.securityzones = []
        self.inlinesets = []
        self.physicalinterfaces = []
        self.subinterfaces = []
        self.etherchannelinterfaces = []
        self.ipv4staticroutes = []
        self.networks = []
        self.hosts = []
        self.interfacesecurityzones = []
        self.bridgegroupinterfaces = []
        self.networkgroups = []

    def run(self, device, policy_name=''):
        """
        We run our clis and get the data that we need. Then we store it.
        @param device: the device config
        """
        self.policy_name = policy_name
        self.init_holders()
        self.probe_accesspolicies(device)
        self.probe_accessrules(device)
        self.probe_securityzones(device)
        self.probe_inlinesets(device)
        self.probe_physicalinterfaces(device)
        self.probe_subinterfaces(device)
        self.probe_etherchannels(device)
        self.probe_ipv4staticroutes(device)
        self.probe_networks(device)
        self.probe_hosts(device)
        self.probe_bridgegroupinterfaces(device)
        self.probe_networkgroups(device)
        self.config_keeper.set('interfacesecurityzones', self.interfacesecurityzones)
        self.data.extend(self.interfacesecurityzones)
        ConfigKeeper.set_global(self.ldev_id, self)
        
        # probe should be run as needed in the new scheme as the configuration of a device can be changed dynamically
        
    def process_cliis(self, cliis):
        for cli in cliis:
            if cli.does_id_exist():
                if cli.dataParam.param_uuid.has_key('items'):#multiple
                    for item in cli.dataParam.param_uuid['items']:
                        self.data.append(item)
                        self.add_object(item)
                else:
                    self.data.append(cli.dataParam.param_uuid)
                    self.add_object(cli.dataParam.param_uuid)
            else:
                # See if items are present in 'param_value'
                try:
                    if cli.dataParam and cli.dataParam.param_value and cli.dataParam.param_value.has_key('items'):
                        for item in cli.dataParam.param_value['items']:
                            self.data.append(item)
                            self.add_object(item)
                except Exception:
                    pass

    def item_in_data(self, candidate, data):
        for item in data:
            if item['id'] == candidate['id']:
                return True
        return False
        
    def add_object(self, obj):
        obj_type = obj['type']
        if obj_type == 'AccessPolicy':
            self.accesspolicies.append(obj)
        elif obj_type == 'AccessRule':
            self.accessrules.append(obj)
        elif obj_type == 'SecurityZone':
            # Get unique interfaces
            if isinstance(obj, dict) and obj.has_key('interfaces'):
                unique_intf = [dict(x) for x in set(tuple(item.items()) for item in obj['interfaces'])]
                obj['interfaces'] = unique_intf
            self.securityzones.append(obj)
        elif obj_type == 'InterfaceSecurityZone':
            self.interfacesecurityzones.append(obj)
        elif obj_type == 'InlineSet':
            self.inlinesets.append(obj)
        elif obj_type == 'PhysicalInterface':
            self.physicalinterfaces.append(obj)
        elif obj_type == 'SubInterface':
            self.subinterfaces.append(obj)
        elif obj_type == 'EtherChannelInterface':
            self.etherchannelinterfaces.append(obj)
        elif obj_type == 'IPv4StaticRoute':
            self.ipv4staticroutes.append(obj)
        elif obj_type == 'Network':
            self.networks.append(obj)
        elif obj_type == 'Host':
            self.hosts.append(obj)
        elif obj_type == 'BridgeGroupInterface':
            self.bridgegroupinterfaces.append(obj)
        elif obj_type == 'NetworkGroup':
            self.networkgroups.append(obj)

    def find_all_with_param_regex(self, param, reg):
        """
        return a list of jsons based on the regex of the param
        @param param: paramater to look in based on the database>
        EX: "name", "id", "commentHistoryList"
        @param reg: a regex string
        """
        return_arr = []
        for dat in self.data:
            if isinstance(dat, dict) and dat.has_key(param):
                try:
                    if re.search(reg, asciistr(dat[param])):
                        return_arr.append(dat)
                except:
                    pass
        return return_arr
                
    
    def find_all_with_params_regex(self, **kwargs):
        """
        get a dictionary that contains a param : reg model. returns the intersection of all.
        """
        first = True
        return_arr = []
        for k, v in kwargs.iteritems():
            temp = self.find_all_with_param_regex(k, v)
            if first:
                if temp == None:
                    temp = []
                return_arr = temp
                first = False
            elif len(return_arr) > 0:
                return_arr = intersection_of_arrays_ftd(return_arr, temp)
            else:
                return []
        return return_arr
    
    def get_all_items_by_ldevid(self, ldevid):
        arr = []
        arr += self.find_all_with_param_regex("name", asciistr(ldevid))
        arr += self.find_all_with_param_regex("ifname", asciistr(ldevid))
        arr += self.find_all_with_param_regex("description", asciistr(ldevid))
        arr += self.find_all_with_param_regex("commentHistoryList", asciistr(ldevid))
        return arr

    def get_physicalinterfaces(self):
        return self.physicalinterfaces
    
def intersection_of_arrays_ftd(arr1, arr2):
    """
    returns the intersection of the two arrays
    """
    ret_arr = []
    for item1 in arr1:
        for item2 in arr2:
            if item1 == item2:
                ret_arr.append(item1)
    return ret_arr

def find_ac_policy_by_regex(command_executor):
    """
    Finds the Access Policy from a list of Access Policy based on the regex
    """
    try:
        ac_items = command_executor.command_holder.parent.dataParam.param_value['items']
    except:
        return [None]
    reg = command_executor.command_holder.param_value
    for ac in ac_items:        
        try:
            if re.match(reg, asciistr(ac)) != None:
                command_executor.command_holder.parent.dataParam.param_uuid = ac
                command_executor.command_holder.parent.idParam.param_uuid = ac['id']
        except:
            # should not come here in general, but just in case, we let the process continue
            pass
    command_executor.command_holder.parent.dataParam.param_uuid = None
    return [None]
