'''
Created on Apr 12, 2015

@author: Puneet Garg

Copyright (c) 2015 by Cisco Systems, Inc.
All rights reserved.

Classes used for device request and response parser utilities.
'''
from devpkg.utils.util import asciistr
from devpkg.base.command_interaction import CommandResult, CommandInteraction, CommandParam
from devpkg.base.command_dispatch import TokenObjectHelper
from devpkg.base.command_interaction import item_does_not_exist_error
import requests, traceback
import devpkg.utils.env as env
from requests.auth import HTTPBasicAuth
from devpkg.base.requestexecutor.login_executor import LoginExecutor
from devpkg.base.requestexecutor.login_executor import save_device_information
import json
from devpkg.utils.state_type import State
from devpkg.utils.errors import FaultCode, FMC429Error
import re
from devpkg.utils.util import sleep
from devpkg.utils.errors import DeviceConfigError
from random import randint
from fmc.config_keeper import ConfigKeeper
from ftd.networks import Networks
from ftd.hosts import Hosts
import selective_probe
import time

'''
GET/POST Globals
'''
GET = "GET"
POST= "POST"
DELETE = "DELETE"
PUT = "PUT"

'''
Requests Basic functions.
'''
def get_by_name_and_store(response, name, target, store):
    try:
        items = response.json()['items']
        for item in items:
            if item['name'] == name:
                store.param_value = item
                store.param_uuid = store.param_value
                return item[target]
    except:
        pass

def get_device_uuid(command_executor, force=False):
    """
    Used by the LoginExecutor
    Gets the device uuid and sets it into TOKEN
    """
    
    probe = command_executor.probe
    if force or probe is None or isinstance(probe, list) or len(probe.config_keeper.get(command_executor.dispatch.device_ip + ':' + probe.ldev_id + ':' + 'device_uuid')) == 0:
        command_result = CommandResult(
                    cli='Devices',
                    err_type='conn_error',
                    err_msg=None,
                    model_key='host')
        command_result.resource = command_executor.dispatch
        command_result.resource_key = 'device_uuid'
        command_executor.command_holder.param_value = command_executor.dispatch.device_ip
        command_executor.command_holder.postexecute = item_does_not_exist_error
        params = "?expanded=true&hostName=%s" % command_executor.dispatch.device_ip
        command_executor.command_holder.name = command_executor.dispatch.device_ip
        url = command_executor.dispatch.url +  command_executor.command_holder.get_url(url='fmc_config/v1/domain/<domainUUID>/devices/devicerecords', domainUUID=command_executor.dispatch.domain_uuid)
        command_executor.command_holder.response_parser = save_device_information
        command_executor.command_holder.response_parser_arg = {"command_executor" : command_executor}
        return execute_request(command_executor, url, params, command_result, GET)[0]
    else:
        ldev = probe.ldev_id
        device_uuid = probe.config_keeper.get(command_executor.dispatch.device_ip + ':' + ldev + ':' + 'device_uuid')
        device_name = probe.config_keeper.get(command_executor.dispatch.device_ip + ':' + ldev + ':' + 'device_name')
        if len(device_uuid) > 0 and len(device_name) > 0:
            command_executor.dispatch.device_uuid = device_uuid
            command_executor.dispatch.device_name = device_name
            command_executor.command_holder.param_uuid = device_uuid
            return [None]
        else:
            print("404xxxdevice, unable to get device_uuid, which should have been saved by login_executor.")
        


def k2v_json(toFind, data):
    """
    find the value of a certain key if you dont know where the key is in a nested dict.Like a json
    @param toFind: string that needs to be found in the json. is the key
    @param data: the json that we are looking in
    """
    found = None
    if isinstance(data, dict):
        for k, v in data.iteritems():
            if k == toFind:
                return v
            else:
                t = k2v_json(toFind, v)
                if t is not None:
                    found = t
                    break
    if isinstance(data, list):
        for v in data:
            t = k2v_json(toFind, v)
            if t is not None:
                found = t
                break
    return found

def is_cifs_etherchannel(config):
    ''' Check if Ether-channel (aka Port-channel) is used as cifs
    '''
    intf_d = k2v_json('cifs', config) # e.g. INPUT: 'cifs': {'dev6': 'Port-channel3'}, OUTPUT: {'dev6': 'Port-channel3'}
    if intf_d is None or len(intf_d) == 0:
        return False
    for k, v in intf_d.iteritems():
        return v.startswith('Port-channel')
    
def generate_request_url(command_executor):
    return command_executor.dispatch.url + command_executor.command_holder.get_url(domainUUID=command_executor.dispatch.domain_uuid, deviceUUID=command_executor.dispatch.device_uuid)

def return_json(response, *args, **kwargs):
    return response.json()

def parse_response_for_target(response, target):
    """
    Finds the target within the response
    @param response: the response object from a request command
    @param target: a string that tries the find data associated with it. EX: target = 'id' will find the data for id
    @return: string of the target found or None if nothing is found 
    """
    response_json = response.json()
    if response_json.has_key('items'):
        try:
            response_json = response_json['items'][0]
        except Exception as e:
            env.debug("Unexpected exception: " + asciistr(e) + '\n' + traceback.format_exc())

    if response_json.has_key("metadata"):
        del response_json['metadata']
    if response_json.has_key("defaultAction"):
        del response_json['defaultAction']
    try:
        return response_json[target]
    except:
        return k2v_json(target, response_json)

def parse_response_for_target_in_header(response, target):
    """
    Used internally
    Finds the target in the response header
    @param response: the response object from a request command
    @param target: a string that tries to find data associated with it
    @return: string of the target found or None if nothing is found
    """
    try:
        return response.headers[target]
    except Exception as e:
        env.debug("Unexpected exception: " + asciistr(e) + '\n' + traceback.format_exc())
        return None
    
def parse_response_for_target_and_store_json(response, target, store):
    """
    Parses the response for the target then stores the response into store
    @param response: the response object from a request command
    @param target: a string that tries to find data associated with it
    @param store: the ParamCommand that the data will be stored in.
    @return: string of the target found or None if nothing is found
    """
    try:
        json_data = return_json(response)
        if json_data.has_key('items'):
            if len(json_data['items']) == 1:
                store.param_uuid = json_data['items'][0]
            else:
                store.param_uuid = json_data
        else:
            store.param_uuid = json_data
        store.param_value = store.param_uuid
        uuid =  json_data['items'][0][target]
        if uuid is None:
            store.param_uuid = None
            store.param_value = None
        return uuid
    except:
        uuid = parse_response_for_target(response, target)
        if uuid is None:
            store.param_uuid = None
            store.param_value = None
        return uuid

def execute_request(command_executor, url = None, params = None, command_result = None, request_type =  "GET", deploy_request = False):
    """
    Executes the command_executor
    @param command_executor:  the CommandExecutor object
    @param url: the url being PUT/POST/GET/DELETE/etc to
    @param params: if a PUT/POST the paramaters being sent. If a GET then the paramters that are added at the end of the url 
    @param command_result: a CommandResult class
    @param request_type: the request type. The global variables are listed under GET/POST Globals
    @return: Results
    """
    def retry_next_device(command_executor, url, resource_name):
        next_active_device = command_executor.dispatch.get_next_device()
        if next_active_device == None:
            raise DeviceConfigError([([(0, '', None)], FaultCode.CONFIGURATION_ERROR, "No valid device config found.")])
        command_executor.dispatch.init_device(next_active_device)
        get_device_uuid(command_executor, True)
        url = re.sub(r"devicerecords/.*/" + resource_name, "devicerecords/%s/" + resource_name % command_executor.dispatch.device_uuid, url)
        
    if url is None:
        url = generate_request_url(command_executor)
    
    if params is None:
        if request_type == GET:
            params = generate_params_uuid_json(command_executor, command_executor.command_holder)
        elif request_type != DELETE:
            params = get_json(command_executor)
            'ToDo: CSCvd20931 - Comment out the code block for now to improve code coverage, as the code is not executed in current implementations.'
            """
            if params is None:
                return [None]"""
            
    env.debug('Requesting %s from FSMC URL %s' % (request_type, url))
    env.debug('Requesting FSMC for parameters ' +  asciistr(params))
    TokenObjectHelper.get_token(command_executor.dispatch)
    header = {'Content-Type': 'application/json', 
                      'x-auth-access-token': command_executor.dispatch.auth_token}
    if not deploy_request:
        timeout = command_executor.dispatch.DEFAULT_TIMEOUT
    else:
        timeout = 120 #recomended by FMC team
    #env.debug("Auth-Token: " + str(command_executor.dispatch.auth_token))
    execption_hit_count = 0
    deploy_retry_count = 0
    EXCEPTION_RETRY_COUNT = 10
    while True:
        try:
            if request_type is POST:
                start_time = time.time()
                if isinstance(command_executor, LoginExecutor): #login must be treated differently
                    auth = HTTPBasicAuth(command_executor.dispatch.auth[0], command_executor.dispatch.auth[1])
                    response = requests.post(url,
                              auth=auth,
                              verify=False)
                else:
                    
                    response = requests.post(url,
                                 data=params,
                                 headers=header,
                                 timeout=timeout,
                                 verify=False)
                end_time = time.time()
                env.debug("time spent in POST " + url + ": " + str(end_time - start_time) + " seconds")    
                if response.status_code == 400 and 'ipv4staticroutes' in url: #identify an error case in FMC, see CSvi56405
                    next_active_device = command_executor.dispatch.get_next_device()
                    if next_active_device == None:
                        raise DeviceConfigError([([(0, '', None)], FaultCode.CONFIGURATION_ERROR, "No valid device config found.")])
                    command_executor.dispatch.init_device(next_active_device)
                    get_device_uuid(command_executor)
                    url = re.sub(r"devicerecords/.*/routing/ipv4staticroutes", "devicerecords/%s/routing/ipv4staticroutes" % command_executor.dispatch.device_uuid, url)
                    continue
                elif response.status_code == 500 and 'There is no device that is available for deployment' in response.text:
                    deploy_retry_count += 1
                    if deploy_retry_count <= EXCEPTION_RETRY_COUNT:
                        sleep(5)
                        continue
                    else:
                        break
                elif response.status_code == 422 and 'bridgegroupinterfaces' in url and 'Cannot have ifname for Bridge Group Interface in Transparent mode device' in response.text:
                    # In transparent firewall mode, we need to catch the error (for POST operation the error code is 422) and retry without the ifname
                    js = json.loads(params)
                    del js['ifname']  # remove ifname from the payload
                    params = json.dumps(js)
                    response = requests.post(url, data=params, headers=header, timeout=timeout, verify=False)
                    env.debug('Retry without ifname, response=' + asciistr(response))
                elif response.status_code == 201 and 'ipv4staticroutes' in url:
                    command_executor.command_holder.postexecute = None
                    command_executor.command_holder.response_parser = None
            elif request_type is GET:
                start_time = time.time()
                response = requests.get(url + params, 
                                         headers=header,
                                         timeout=timeout,
                                         verify=False)
                end_time = time.time()
                env.debug("time spent in GET " + url + ": " + str(end_time - start_time) + " seconds")    
            elif request_type is PUT:
                start_time = time.time()
                response = requests.put(url,
                                 data=params,
                                 headers=header,
                                 timeout=timeout,
                                 verify=False)
                end_time = time.time()
                env.debug("time spent in PUT " + url + ": " + str(end_time - start_time) + " seconds")    
                if response.status_code == 400 and 'bridgegroupinterfaces' in url and 'Cannot have ifname for Bridge Group Interface in Transparent mode device' in response.text:
                    # In transparent firewall mode, we need to catch the error (for PUT operation the error code is 400) and retry without the ifname
                    js = json.loads(params)
                    del js['ifname']  # remove ifname from the payload
                    params = json.dumps(js)
                    response = requests.put(url, data=params, headers=header, timeout=timeout, verify=False)
                    env.debug('Retry without ifname, response=' + asciistr(response))
                elif response.status_code == 400 and 'bridgegroupinterfaces' in url and 'This operation cannot be performed as some of the Interfaces are not in sync' in response.text:
                    retry_next_device(command_executor, url, "bridgegroupinterfaces")
                    continue
                elif response.status_code == 400 and 'physicalinterfaces' in url and 'This operation cannot be performed as some of the Interfaces are not in sync' in response.text:
                    retry_next_device(command_executor, url, "physicalinterfaces")
                    continue
                elif response.status_code == 400 and 'etherchannelinterfaces' in url and 'This operation cannot be performed as some of the Interfaces are not in sync' in response.text:
                    retry_next_device(command_executor, url, "etherchannelinterfaces")
                    continue
            elif request_type is DELETE:
                start_time = time.time()
                response = requests.delete(url,
                                 headers=header,
                                 timeout=timeout,
                                 verify=False)
                end_time = time.time()
                env.debug("time spent in DELETE " + url + ": " + str(end_time - start_time) + " seconds")    
                #env.debug('response: ' + str(response))
                #env.debug('response body: ' + str(response.content))
                #env.debug('response: DELETE')
                if response.status_code == 400 and 'bridgegroupinterfaces' in url and 'This operation cannot be performed as some of the Interfaces are not in sync' in response.text:
                    retry_next_device(command_executor, url, "bridgegroupinterfaces")
                    continue
                elif response.status_code == 400 and 'subinterfaces' in url and 'Invalid Session' in response.text:
                    retry_next_device(command_executor, url, "subinterfaces")
                    continue
                else:
                    if response.status_code >= 300:
                        env.debug("DELETE failed but continued: " + asciistr(response.content))
                    return [None] # we dont care about errors on delete
            else:
                return ([None], None)
            break
        except requests.exceptions.ReadTimeout:
            execption_hit_count += 1
            if execption_hit_count <= EXCEPTION_RETRY_COUNT:
                env.debug('Timed out will try again')
                sleep(randint(1,10))
            else:
                raise
        
    env.debug('response: ' + asciistr(response))
    #env.debug('response body: ' + str(response.content))
    
    
    return (parse_response(command_executor, response, command_result, url, params, request_type), response)

def parse_response(command_executor, response, command_result, url, params, request_type):
    '''
    This function is called to parse FSMC API response JSON for a parameter. Internal function
    @param command_executor: the CommandExecutor object
    @param response: the response form the request
    @param command_result: the CommandResult object 
    @return: Results 
    '''
    result = [None] * 1
    if response.status_code >= 200 and response.status_code < 300: #2xx status code returned. Lets assume 200 but not presume
        if command_executor.command_holder is None:
            return result
        if command_executor.command_holder.response_parser is None:
            return result 
        else:
            if command_executor.command_holder.response_parser_arg is not None:
                uuid = command_executor.command_holder.response_parser(response, **command_executor.command_holder.response_parser_arg)
            else:
                uuid = command_executor.command_holder.response_parser(response)
            
            if uuid is not None:
                if command_result:
                    setattr(command_result.resource, command_result.resource_key, uuid) 
    elif response.status_code == 406: # This is a workaround to FMC REST defect CSCvf12205 where PUT operation should be idempotent.
        return result 
    else: #an error has occurred
        result = parse_error(command_executor, response, command_result, url, params, request_type)
        
    return result

def parse_error(command_executor, response, command_result, url, params, request_type):
    """
    Internal Function. This function is called to parse errors from parse_response
    @param command_executor: the CommandExecutor object
    @param response: the response form the request
    @param command_result: the CommandResult object 
    """
    non_error = False #boolean to keep track if the error is something that is real or something that can be ignored
    relog = False
    err_msg = None
    result = [None] * 1
    if command_executor.command_holder.error_parser:
        err_msg = command_executor.command_holder.error_parser(response, command_executor)
        if not isinstance(err_msg, str):
            result[0] = err_msg
        if err_msg is None:
            try:
                err_msg = k2v_json('description', response.json())
            except:
                pass
    else:
        try:
            err_msg = k2v_json('description', response.json())
        except:
            pass
    try:
        if command_result is not None:
            cli = command_result.cli
            model_key = command_result.model_key
            err_type =  command_result.err_type
        else:
            cli = command_executor.command_holder.command
            err_type = 'error'
            model_key = command_executor.command_holder.model_key
    except:
        cli = None
        err_type = 'error'
        model_key = None
    if err_msg is not None and ("Page will be reloaded" in err_msg or "Please retry after sometime" in err_msg):# we need to resend data
        non_error = True
        env.debug("Someone else was using the same resource and it was recently changed, will run call again")
        sleep(10)#Delay in case multiple quries to the same page at the same time.
        errs = execute_request(command_executor, url, params, command_result, request_type)[0] #we will refire the request
        result[0] = errs[0]
        relog = True
    elif response.status_code == 429:# too many requests
        max_retries = 5
        error_429_counter = str(env.get_variable(env.ERROR_429_COUNTER))
        num_retries = '0' if error_429_counter == env.UNDEFINED else error_429_counter
        if int(num_retries) >= max_retries:
            num_retries = '0'
            env.set_variable(env.ERROR_429_COUNTER, num_retries)
            msg = "Got error-code-429 (Too Many Calls) from FMC %s times in a row, looks like something is wrong on FMC. To workaround this, do one of these: 1) Reboot FMC. 2) Delete and re-import FTD device package." % max_retries
            env.debug("========== hit maximum retries for 429-error-code, raising fault now.")
            raise FMC429Error(msg)
        else:
            num_retries = str(int(num_retries) + 1)
            env.debug("Too Many Calls: ========== num_retries is: " + num_retries)
            env.set_variable(env.ERROR_429_COUNTER, num_retries)
            non_error = True
            sleep(60) #give it some additional cooldown period
            errs = execute_request(command_executor, url, params, command_result, request_type)[0] #we will refire the request
            result[0] = errs[0]
            relog = True
    elif response.status_code == 401 or (response.status_code == 500 and err_msg and "Authorization Failure" in err_msg) or (response.status_code == 500 and err_msg and "Session" in err_msg and "is invalid." in err_msg): #unauthorized
        if not isinstance(command_executor, LoginExecutor): #somehow we are not logged in
            non_error = True
            env.debug("Token expired, recreating token")
            sleep(10)#we are going to wait for a bit so that whatever caused us to hit a 401 or Auth Failure goes away. Possible OUB edits.
            login = LoginExecutor(command_executor.dispatch)
            errs = login.execute()
            if errs != None and errs[0] != None:
                return errs
            
            command_executor.dispatch = login.dispatch #this dispatch has the auth token
            errs = execute_request(command_executor, url, params, command_result, request_type)[0] #we will refire the request
            result[0] = errs[0]
            relog = True
        else:
            err_msg = "Can't login to a appliance, configured login information is wrong"
            env.debug('login Error')
    elif response.status_code == 404: #not found
            if isinstance(command_executor.command_holder.param_value, dict):
                for key, val in command_executor.command_holder.param_value.iteritems():
                    if 'name' in key:
                        err_msg = "Can't find configured %s with Value %s." % (command_executor.command_holder.command, val)
                        break
            else:
                err_msg = "Can't find configured %s with Value %s." % (command_executor.command_holder.command, command_executor.command_holder.param_value)
    elif response.status_code == 503: #Service Unavailable. REST not enabled
        err_msg = "FireSIGHT Management Center REST API Preferences not enabled."
    if err_msg == "" or err_msg is None:
        err_msg = "For %s: an unknown error occurred with HTML status %s" % (command_executor.command_holder.command, str(response.status_code))
    if err_msg == [None]:
        return err_msg
    if result[0] is None and not relog:
        result[0] = CommandResult(
            cli = cli,
            err_type = err_type,
            err_msg = err_msg,
            model_key = model_key)
    if not non_error: #was the error a real error
        env.debug('response body: ' + asciistr(response.content))
    return result

'''
Requests parameters JSONs.
'''

def generate_params_uuid_json(dispatcher, param):
    '''
    This function is called to generate FSMC API request JSON for a parameter.

    @param dispatcher: Holds device information.
    @param param: Parameter for which request JSON to be generated.
    @return: string
    '''
    try:
        if param.param_value is None or param.param_key is None:
            return ""
        if isinstance(param.param_value, dict):
            ret = ""
            char = '?'
            for key, val in param.param_value.iteritems():
                ret += '%s%s=%s' % (char, key, val)
                char = '&'
                
            return ret
        
        return "?name=" + param.param_value 
    except:
        return ""

def get_json(command_executor):
    '''
    This function is called to generate FSMC API request JSON.

    @param command_executor: Holds command and dispatcher information.
    @return: JSON
    '''
    json_param = ""
    try:
        if command_executor.command_holder.does_id_exist():
            json_param = command_executor.command_holder.dataParam.param_uuid
        else:
            json_param = command_executor.command_holder.defaultData.param_uuid
            
        # Do deletion first before addition
        if len(command_executor.command_holder.removable) > 0:#do we have anything that is supposed to be removed?
            for json_key, json_value in json_param.iteritems():
                command_executor.command_holder.remove_value_from_dict(json_key, json_value, json_param)

        for paramKey, paramValue in command_executor.command_holder.params.iteritems():
            if paramValue.param_uuid == "<deviceUUID>":
                paramValue.param_uuid = command_executor.dispatch.device_uuid
            elif paramValue.param_uuid == "<deviceNAME>":
                paramValue.param_uuid = command_executor.dispatch.device_ip
            #We can cut a lot of the if statements that are here by just using our own known system
            
            if paramKey == command_executor.command_holder.command:
                continue
            if not paramValue.should_add_to_json():
                continue
            try:
                #inlinepairs and accessrules have special cases where we cannot remove data from them. we just append the data instead
                if paramKey == "sourceZones" or paramKey == "destinationZones":
                    for val in paramValue.param_uuid['objects']:#issue
                        if not json_param.has_key(paramKey):
                            val = update_id(val)
                            json_param[paramKey] = {'objects': [val]}
                        else:
                            found = False
                            for n in json_param[paramKey]['objects']:
                                if n['name'] == val['name']:
                                    found = True
                                    break
                            if not found:
                                val = update_id(val)
                                json_param[paramKey]['objects'].append(val)
                    continue
                if json_param.has_key(paramValue.param_formatter) and isinstance(json_param[paramValue.param_formatter], list) and len(paramValue.param_uuid) > 0:
                    if paramKey == "inlinepairs":
                        specialArg = "first"
                    else:
                        specialArg = None
                    if isinstance(paramValue.param_uuid, list):
                        for stuff in paramValue.param_uuid:
                            if not is_dict_in_list(stuff, json_param[paramValue.param_formatter], specialArg):
                                json_param[paramValue.param_formatter].append(stuff) 
                    else: 
                        if not is_dict_in_list(paramValue.param_uuid, json_param[paramValue.param_formatter], specialArg):
                            json_param[paramValue.param_formatter].append(paramValue.param_uuid)
                    continue
            except:
                env.debug("========== continue")
                pass
            json_param[paramValue.param_formatter] = paramValue.param_uuid #add to the json_param
        
        # TODO: check why sourceZones and destinationZones are removable here, skip the removal for the time being as this shouldn't happen
        '''
        if len(command_executor.command_holder.removable) > 0:#do we have anything that is supposed to be removed?
            for json_key, json_value in json_param.iteritems():
                command_executor.command_holder.remove_value_from_dict(json_key, json_value, json_param)
        '''
        
        json_param = remove_unneeded_keys_from_json(json_param)
        
        json_dumps = json.dumps(json_param, cls=CommandInteraction.CommandInteractionEncoder)

        if '""' in json_dumps: #there is an empty area
            # Couldn't find id? No need to issue another GET request to FMC, we can find it in probe data, e.g. json_param={"interfaces":[{"id":<CommandParam>}]}
            json_param = replace_id_from_probe(json_param, command_executor.probe)
            json_dumps = json.dumps(json_param, cls=CommandInteraction.CommandInteractionEncoder)
            if '""' in json_dumps:
                env.debug("json_dumps=" + asciistr(json_dumps))
                raise Exception("Unexpected Json Validation, please try to push configuration again.")
            else:
                return json_dumps
        return json_dumps
    except Exception as e:
        env.debug("Unexpected exception: " + asciistr(e) + '\n' + traceback.format_exc() + '\n' + asciistr(json_param))
        raise DeviceConfigError([(command_executor.command_holder.model_key, FaultCode.CONFIGURATION_ERROR, "Unexpected exception: " + asciistr(e) + asciistr(json_param))])

def update_id(val, probe):
    if not isinstance(val['id'], CommandParam) and asciistr(val['id']) != '':
        # Already a none-empty string, no need to update the id, returning
        return val
    sz_id = asciistr(val['id'].toDict()['param_uuid'])
    if sz_id != '':
        val['id'] = sz_id
    else:
        val['id'] = probe.config_keeper.get_id_from_securityzones(val['name'])
    return val

    
def replace_id_from_probe(json_param, probe):
    param_type = asciistr(json_param['type']) if json_param.has_key('type') else 'unknown'
    if param_type == 'SecurityZone':
        if json_param.has_key('interfaces'):
            updated_intf_list = []
            for each_intf in json_param['interfaces']:
                if not isinstance(each_intf['id'], dict) and not isinstance(each_intf['id'], CommandParam):
                    each_intf['id'] = asciistr(each_intf['id'])
                else:
                    intf = each_intf['id'].toDict()
                    if intf.has_key('param_value') and intf['param_value'].has_key('name'):
                        intf_name = intf['param_value']['name']
                        intf_id = probe.config_keeper.get_id_from_probe(intf_name, 'physicalinterfaces')
                        each_intf['id'] = intf_id
                if each_intf not in updated_intf_list:
                    updated_intf_list.append(each_intf)
            json_param['interfaces'] = updated_intf_list
    elif param_type == "BridgeGroupInterface": # e.g. {"selectedInterfaces": [{"type": "PhysicalInterface", "name": "GigabitEthernet0/0", "id": <CommandParamValue>}, {"type": "PhysicalInterface", "name": "GigabitEthernet0/1", "id": <CommandParamValue>}]}
        if json_param.has_key('selectedInterfaces'):
            updated_intf_list = []
            for each_intf in json_param['selectedInterfaces']:
                if each_intf.has_key('name'):
                    intf_type = 'etherchannelinterfaces' if asciistr(each_intf['name']).startswith('Port-channel') else 'physicalinterfaces'
                    each_intf['id'] = probe.config_keeper.get_id_from_probe(each_intf['name'], intf_type)
                    updated_intf_list.append(each_intf)
            json_param['selectedInterfaces'] = updated_intf_list
    elif param_type == "InlineSet": # e.g. {"inlinepairs":[{"first":{"id":<CommandParam>,"name":<CommandParamValue>}, {"second":....}}]
        if json_param.has_key('inlinepairs'):
            updated_pair_list = []
            for each_pair in json_param['inlinepairs']:
                name = each_pair['first']['name']
                first_name = name if isinstance(name, str) or isinstance(name, unicode) else name.get_value()
                name = each_pair['second']['name']
                second_name = name if isinstance(name, str) or isinstance(name, unicode) else name.get_value()
                intf_type = 'etherchannelinterfaces' if asciistr(first_name).startswith('Port-channel') else 'physicalinterfaces'
                each_pair['first']['id'] = probe.config_keeper.get_id_from_probe(first_name, intf_type)
                intf_type = 'etherchannelinterfaces' if asciistr(second_name).startswith('Port-channel') else 'physicalinterfaces'
                each_pair['second']['id'] = probe.config_keeper.get_id_from_probe(second_name, intf_type)
                updated_pair_list.append(each_pair)
            json_param['inlinepairs'] = updated_pair_list
    elif param_type == "AccessRule": # e.g. {"sourceZones":{"objects":[{"id":<CommandParam>,"name":'pro0SZRT5+TenantScaleTest2+FTDtest'}, {...}]}, "destinationZones": {...}}
        if json_param.has_key('sourceZones'):
            update_zone_id(json_param['sourceZones']['objects'], probe)
        if json_param.has_key('destinationZones'):
            update_zone_id(json_param['destinationZones']['objects'], probe)
    elif param_type == "PolicyAssignment": # e.g. {'policy':{'type':'AccessPolicy', 'name':'ACIAccPolicyRouted', 'id':<CommandParam>...}, 'type':'PolicyAssignment', 'name': 'ACIAccPolicyRouted'}
        if json_param.has_key('policy'):
            updated_policy = json_param['policy']
            updated_policy['id'] = probe.config_keeper.get_id_from_probe(updated_policy['name'], 'AccessPolicy')
            json_param['policy'] = updated_policy

    return json_param

def update_zone_id(zone_list, probe):
    updated_sz_list = []
    for each_sz in zone_list:
        if not isinstance(each_sz['id'], CommandParam) and asciistr(each_sz['id']) != '':
            # id is already resolved
            continue
        else:
            sz_id = probe.config_keeper.get_id_from_securityzones(each_sz['name'])
            if sz_id != '':
                each_sz['id'] = sz_id
            else:
                each_sz['id'] = probe.config_keeper.get_id_from_probe(each_sz['name'], 'securityzones')
        updated_sz_list.append(each_sz)
    zone_list = updated_sz_list

def remove_unneeded_keys_from_json(json_object):
    """
    Does the basic removal of all of the unneeded keys in every json object
    """
    if 'metadata' in json_object:
        del json_object['metadata']
    if 'links' in json_object:
        del json_object['links']
    if 'rules' in json_object:
        del json_object['rules']
    if 'paging' in json_object:
        del json_object['paging']
    if 'ipv6' in json_object:
        del json_object['ipv6']
    if "commentHistoryList" in json_object:
        del json_object["commentHistoryList"]
    if "enableAntiSpoofing" in json_object:
        del json_object["enableAntiSpoofing"]
    if "fragmentReassembly" in json_object:
        del json_object["fragmentReassembly"]
    return json_object

def is_dict_in_list(dictionary, array, specialArg = None):
    """
    is a certain dictionary in a list
    @param dictionary: the dictionary that we are looking for
    @param array: the array that we are looking in
    @param specialArg: if set to something searches inside that dictionaries array
    """
    try:
        if specialArg:
            for things in array:
                if not isinstance(things, dict) or not isinstance(dictionary, dict):
                    return False
                if asciistr(things[specialArg]['id']) == asciistr(dictionary[specialArg]['id']):
                    return True
                elif specialArg in ('first', 'second'):
                    # In the special case of inlineset, we compare physical interface name directly as that won't change
                    return asciistr(things[specialArg]['name']) in (asciistr(dictionary['first']['name'].toDict()['param_value']), asciistr(dictionary['second']['name'].toDict()['param_value']))
        else:
            for things in array:
                if isinstance(things, dict) and isinstance(dictionary, dict) and things.has_key('id') and dictionary.has_key('id') and asciistr(things['id']) == asciistr(dictionary['id']):
                    return True
                elif isinstance(things, str) and isinstance(dictionary, str) and asciistr(things) == asciistr(dictionary):
                    return True
    except Exception as e:
            env.debug("Unexpected exception: " + asciistr(e) + '\n' + traceback.format_exc()) #should not get here
    return False

def get_apply_json(command_executor):
    '''
    This function is called to generate FSMC API device changes apply request JSON.

    @param command_executor: Holds command and dispatcher information.
    @return: JSON
    '''
    json_params = {}
    json_params.update({"type"          :   "DeploymentRequest"})
    json_params.update({"version"       :   int(env.get_variable(env.DEPLOYMENT_VERSION))})
    json_params.update({"forceDeploy"   :   "true"})
    json_params.update({"ignoreWarning" :   "true"})
    json_params.update({"deviceList"    :   [command_executor.dispatch.device_uuid]})
    return json.dumps(json_params)

'''
Requests executors.
'''

def dispatch_executor(command_executor):
    '''
    This function is called to execute dispatch (Login) requests.

    @param dispatcher: Holds device information.
    @param dev_response: Response JSON.
    @return: Results
    '''
    
    'Authenticate FSMC and set session token into the dispatcher to execute device commands.'
    url = command_executor.dispatch.url + "fmc_platform/v1/auth/generatetoken"
    params = "login"
    command_result = CommandResult(
                cli='Devices',
                err_type='conn_error',
                err_msg=None,
                model_key='host')
    command_result.resource = command_executor.dispatch
    command_result.resource_key = 'device_uuid'
    err = (execute_request(command_executor, url, params, command_result, POST))[0]
    if err is None or err[0] is None:
        return get_device_uuid(command_executor)
    else:
        return err
    
'ToDo: CSCvd20931 - Comment out the code block for now to improve code coverage, as the code is not executed in current implementations.'
"""       
def session_logout(command_executor):
    '''
    This function is called to execute session logout.

    @param command_executor: Holds device information.
    @return: Results
    '''
    url = command_executor.dispatch.url + 'fmc_platform/v1/auth/revokeaccess'
    command_executor.command_holder.command = "Logout"
    command_executor.command_holder.response_parser = None
    return execute_request(command_executor, url, "", None, POST)[0]
 """

def param_executor(command_executor):
    '''
    This function is called to execute parameter requests.

    @param command_executor: Holds command and dispatcher information.
    @return: Updates parameter internal i.e. UUID information.
    '''
    url = command_executor.dispatch.url + command_executor.command_holder.get_url(domainUUID=command_executor.dispatch.domain_uuid, deviceUUID=command_executor.dispatch.device_uuid)
    param_json = generate_params_uuid_json(command_executor.dispatch, command_executor.command_holder)
    command_result =  CommandResult(
                    cli=command_executor.command_holder.command,
                    err_type='error',
                    err_msg=None,
                    model_key=command_executor.command_holder.model_key)
    command_result.resource = command_executor.command_holder
    command_result.resource_key = 'param_uuid'
    return execute_request(command_executor, url, param_json, command_result)[0]

'ToDo: CSCvd20931 - Comment out the code block for now to improve code coverage, as the code is not executed in current implementations.'
"""            
def remove_all_sz_from_all_rules(command_executor):
    '''
    This function removes all sz in all of the access rules.
    
    @param command_executor: Holds command and dispatcher information.
    @return: Success or Faults results.
    '''
    #TODO: We shall see if we need this or not. Not finished
    url = command_executor.dispatch.url + command_executor.command_holder.parent.get_url(domainUUID=command_executor.dispatch.domain_uuid, deviceUUID=command_executor.dispatch.device_uuid)
    all_rules = command_executor.command_holder.parent.dataParam.param_value
    try:
        #id = all_rules['id']
        for key, value in all_rules.iteritems():
            command_executor.command_holder.parent.remove_value_from_dict(key, value, all_rules)
        #we now just add the sz to the all of the items
        
        all_rules = remove_unneeded_keys_from_json(all_rules)
        
        return execute_request(command_executor, url, json.dumps(all_rules, cls=CommandInteraction.CommandInteractionEncoder), None, PUT)[0]
    except:
        pass

def access_rules_executor_all(command_executor):
    '''
    This function sets the sz in the access rule and sets it to all of the access rules that are there
    
    @param command_executor: Holds command and dispatcher information.
    @return: Success or Faults results.
    '''
    url = command_executor.dispatch.url + command_executor.command_holder.get_url(domainUUID=command_executor.dispatch.domain_uuid, deviceUUID=command_executor.dispatch.device_uuid)
    all_rules = command_executor.command_holder.parent.params['acrule_all'].param_uuid
    try:
        all_rules = all_rules['items']
        for item in all_rules:
            #we now just add the sz to the all of the items
            id = item['id']
            try:
                if item.has_key('sourceZones'):
                    item['sourceZones']['objects'].append(command_executor.command_holder.parent.params['sourceZones'].param_uuid['objects'][0])
                    item['sourceZones']['objects'].append(command_executor.command_holder.parent.params['sourceZones'].param_uuid['objects'][1])
    
                else:    
                    item['sourceZones'] = command_executor.command_holder.parent.params['sourceZones'].param_uuid
            except:
                pass
            
            try:
                if item.has_key('destinationZones'):
                    item['destinationZones']['objects'].append(command_executor.command_holder.parent.params['destinationZones'].param_uuid['objects'][0])
                    item['destinationZones']['objects'].append(command_executor.command_holder.parent.params['destinationZones'].param_uuid['objects'][1])
    
                else:    
                    item['destinationZones'] = command_executor.command_holder.parent.params['destinationZones'].param_uuid
            except:
                pass
            
            item = remove_unneeded_keys_from_json(item)
            
            execute_request(command_executor, url+id, json.dumps(item, cls=CommandInteraction.CommandInteractionEncoder), None, PUT)
    except:
        pass
    finally:
        return [None] #this will never return an error
"""

def access_policy_assignment_executor(command_executor):
    """
    This function is called to execute the access policy assignment

    @param command_executor: Holds command and dispatcher information.
    @return: Success or Faults results.
    """
    if command_executor.command_holder.does_id_exist():
        
        data = command_executor.command_holder.dataParam.param_value
        try:
            targets = data['targets']
            for target in targets:
                if target["name"] == command_executor.dispatch.device_name:
                    return [None]
        except:
            pass
    return device_object_executor(command_executor)

def get_ipv4_static_route_id(command_executor):
    ifname = command_executor.command_holder.params['interfaceName'].param_value + '_' + command_executor.dispatch.top_uuid
    network = command_executor.command_holder.params['network'].param_value
    gateway = command_executor.command_holder.params['gateway'].param_value
    return command_executor.probe.config_keeper.get_ipv4_static_route_id(ifname, network, gateway)

def get_network_object_id(command_executor):
    network_value = command_executor.command_holder.params['network'].param_value
    (network_name, network_id) = command_executor.probe.config_keeper.get_network_from_networks_by_value(network_value)
    if network_name:
        return network_id
    return ''

def get_ipv4_network_json(command_executor):
    network_value = command_executor.command_holder.params['network'].param_value
    (network, prefix) = network_value.split('/')
    network_name = 'NET-' + network + '-' + prefix
    json_param = {"name": asciistr(network_name),
                  "value": asciistr(network_value),
                  "type": "Network"
                  }
    return json_param

def get_ipv4_gateway_id(command_executor):
    gateway_value = command_executor.command_holder.params['gateway'].param_value
    (host_name, host_id) = command_executor.probe.config_keeper.get_host_from_hosts_by_value(gateway_value)
    if host_name:
        return host_id
    return ''

def get_ipv4_gateway_json(command_executor):
    gateway = command_executor.command_holder.params['gateway'].param_value
    host_name = 'HOST-' + gateway
    json_param = {"name": asciistr(host_name),
                  "value": asciistr(gateway),
                  "type": "Host"
                  }
    return json_param

def get_ipv4_static_route_json(command_executor):
    ldev = command_executor.dispatch.top_uuid
    ifname = command_executor.command_holder.params['interfaceName'].param_value + '_' + ldev
    network_value = command_executor.command_holder.params['network'].param_value
    if '/' not in network_value:
        raise DeviceConfigError([([(0, '', None)], FaultCode.CONFIGURATION_ERROR, "Network for IPv4 static route should be in a well-formed format as network/prefix")])
    (network, prefix) = network_value.split('/')
    if prefix == '32': # This is considered a HOST on FMC, so look it up from the hosts table (see also CSCvj36576)
        (network_name, network_id) = command_executor.probe.config_keeper.get_host_from_hosts_by_value(network)
    else:
        (network_name, network_id) = command_executor.probe.config_keeper.get_network_from_networks_by_value(network_value)
    if not network_name:
        # create new network object
        network_obj = Networks('NET-' + network + '-' + prefix, network_value, State.CREATE, command_executor.probe)
        device = ConfigKeeper.get_global(ldev + ':' + 'device')
        network_obj.push_config(device, dispatch_executor)
        # reload network_id
        network_probe = selective_probe.SelectiveProbe(ldev, 'networks')
        network_probe.run(device)
        (network_name, network_id) = command_executor.probe.config_keeper.get_network_from_networks_by_value(network_value)
    gateway = command_executor.command_holder.params['gateway'].param_value
    (host_name, host_id)  = command_executor.probe.config_keeper.get_host_from_hosts_by_value(gateway)
    if not host_name:
        # create new host object
        gateway_obj = Hosts('HOST-' + gateway, gateway, State.CREATE, command_executor.probe)
        device = ConfigKeeper.get_global(ldev + ':' + 'device')
        gateway_obj.push_config(device, dispatch_executor)
        # reload host_id
        host_probe = selective_probe.SelectiveProbe(ldev, 'hosts')
        host_probe.run(device)
        (host_name, host_id)  = command_executor.probe.config_keeper.get_host_from_hosts_by_value(gateway)
    isTunneld = command_executor.command_holder.params['isTunneled'].param_value if command_executor.command_holder.params.has_key('isTunneled') else False
    json_param = {"interfaceName": asciistr(ifname),
                  "selectedNetworks":[
                    {"type":"Network",
                     "id": asciistr(network_id),
                     "name": asciistr(network_name)}],
                  "gateway":{
                    "object":{
                        "type":"Host",
                        "id": asciistr(host_id),
                        "name": asciistr(host_name)}},
                  "type":"IPv4StaticRoute",
                  "isTunneled": isTunneld
                  }
    if command_executor.command_holder.params.has_key('metric') and command_executor.command_holder.params['metric'].param_value is not None:
        json_param['metricValue'] = command_executor.command_holder.params['metric'].param_value
    return json_param
    
def ipv4_gateway_executor(command_executor):
    '''
    This function is called to execute networks requests.

    @param command_executor: Holds command and dispatcher information.
    @return: Success or Faults results.
    '''
    command_result =  CommandResult(
                        cli=command_executor.command_holder.command,
                        err_type='error',
                        err_msg=None,
                        model_key=command_executor.command_holder.model_key)
    command_result.lookup_key = command_executor.command_holder
    command_result.resource = command_executor.command_holder.idParam
    command_result.resource_key = 'param_uuid'
    gateway_id = get_ipv4_gateway_id(command_executor)
    json_param = get_ipv4_gateway_json(command_executor)
    if gateway_id: # found a matching route_id on FMC
        json_param['id'] = gateway_id
        param = json.dumps(json_param)
        url = command_executor.dispatch.url + command_executor.command_holder.get_url(domainUUID=command_executor.dispatch.domain_uuid, deviceUUID=command_executor.dispatch.device_uuid, new=True)
        url += gateway_id
        return execute_request(command_executor, url, param, command_result, PUT)[0]
    else:
        url = command_executor.dispatch.url + command_executor.command_holder.get_url(domainUUID=command_executor.dispatch.domain_uuid, deviceUUID=command_executor.dispatch.device_uuid, new=True)
        url = url[:-1]
        param = json.dumps(json_param)
        return execute_request(command_executor, url, param, command_result, POST)[0]

def ipv4_network_executor(command_executor):
    '''
    This function is called to execute networks requests.

    @param command_executor: Holds command and dispatcher information.
    @return: Success or Faults results.
    '''
    command_result =  CommandResult(
                        cli=command_executor.command_holder.command,
                        err_type='error',
                        err_msg=None,
                        model_key=command_executor.command_holder.model_key)
    command_result.lookup_key = command_executor.command_holder
    command_result.resource = command_executor.command_holder.idParam
    command_result.resource_key = 'param_uuid'
    network_id = get_network_object_id(command_executor)
    json_param = get_ipv4_network_json(command_executor)
    if network_id: # found a matching route_id on FMC
        json_param['id'] = network_id
        param = json.dumps(json_param)
        url = command_executor.dispatch.url + command_executor.command_holder.get_url(domainUUID=command_executor.dispatch.domain_uuid, deviceUUID=command_executor.dispatch.device_uuid, new=True)
        url += network_id
        return execute_request(command_executor, url, param, command_result, PUT)[0]
    else:
        url = command_executor.dispatch.url + command_executor.command_holder.get_url(domainUUID=command_executor.dispatch.domain_uuid, deviceUUID=command_executor.dispatch.device_uuid, new=True)
        url = url[:-1]
        param = json.dumps(json_param)
        return execute_request(command_executor, url, param, command_result, POST)[0]

def ipv4_static_route_executor(command_executor):
    '''
    This function is called to execute inline set requests.

    @param command_executor: Holds command and dispatcher information.
    @return: Success or Faults results.
    '''
    command_result =  CommandResult(
                        cli=command_executor.command_holder.command,
                        err_type='error',
                        err_msg=None,
                        model_key=command_executor.command_holder.model_key)
    command_result.lookup_key = command_executor.command_holder
    command_result.resource = command_executor.command_holder.idParam
    command_result.resource_key = 'param_uuid'
    route_id = get_ipv4_static_route_id(command_executor)
    json_param = get_ipv4_static_route_json(command_executor)
    if route_id: # found a matching route_id on FMC
        json_param['id'] = route_id
        param = json.dumps(json_param)
        url = command_executor.dispatch.url + command_executor.command_holder.get_url(domainUUID=command_executor.dispatch.domain_uuid, deviceUUID=command_executor.dispatch.device_uuid, new=True)
        url += route_id
        return execute_request(command_executor, url, param, command_result, PUT)[0]
    else:
        url = command_executor.dispatch.url + command_executor.command_holder.get_url(domainUUID=command_executor.dispatch.domain_uuid, deviceUUID=command_executor.dispatch.device_uuid, new=True)
        url = url[:-1]
        param = json.dumps(json_param)
        return execute_request(command_executor, url, param, command_result, POST)[0]

def device_object_executor(command_executor):
    '''
    This function is called to execute inline set requests.

    @param command_executor: Holds command and dispatcher information.
    @return: Success or Faults results.
    '''
    command_result =  CommandResult(
                        cli=command_executor.command_holder.command,
                        err_type='error',
                        err_msg=None,
                        model_key=command_executor.command_holder.model_key)
    command_result.lookup_key = command_executor.command_holder
    command_result.resource = command_executor.command_holder.idParam
    command_result.resource_key = 'param_uuid'
    param = get_json(command_executor)
    """
    if param is None:
            return [None]"""
    if command_executor.command_holder.does_id_exist():
        return execute_request(command_executor, None, param, command_result, PUT)[0]
    else:
        url = command_executor.dispatch.url + command_executor.command_holder.get_url(domainUUID=command_executor.dispatch.domain_uuid, deviceUUID=command_executor.dispatch.device_uuid, new=True)
        url = url[:-1]
        return execute_request(command_executor, url, param, command_result, POST)[0]

'ToDo: CSCvd20931 - Comment out the code block for now to improve code coverage, as the code is not executed in current implementations.'
"""
def device_executor(command_executor):
    '''
    This function performs device operation

    @param command_executor: Holds command and dispatcher information.
    @return: Success or Faults results.
    '''
    command_result =  CommandResult(
                        cli=command_executor.command_holder.command,
                        err_type='error',
                        err_msg=None,
                        model_key=command_executor.command_holder.model_key)
    command_result.lookup_key = command_executor.command_holder
    command_result.resource = command_executor.command_holder
    command_result.resource_key = 'param_uuid'
    param = get_json(command_executor)
    url = command_executor.dispatch.url + command_executor.command_holder.get_url(domainUUID=command_executor.dispatch.domain_uuid, new=True)
    url = url[:-1]
    return execute_request(command_executor, url, param, command_result, POST)
"""

def network_object_executor(command_executor):
    '''
    This function is called to execute network object group requests
    
    @param command_executor: Holds command and dispatcher information.
    @return: Success or Faults results.
    '''
    
    command_result =  CommandResult(
                    cli=command_executor.command_holder.command,
                    err_type='error',
                    err_msg=None,
                    model_key=command_executor.command_holder.model_key)
    command_result.lookup_key = command_executor.command_holder
    command_result.resource = command_executor.command_holder.idParam
    command_result.resource_key = 'param_uuid'
    param = get_json(command_executor)
    param_json = json.loads(param)
    if param_json.has_key('literals') and len(param_json['literals']) > 0:
        if command_executor.command_holder.does_id_exist():
            return execute_request(command_executor, None, param, command_result, PUT)[0]
        else:
            url = command_executor.dispatch.url + command_executor.command_holder.get_url(domainUUID=command_executor.dispatch.domain_uuid, deviceUUID=command_executor.dispatch.device_uuid, new=True)
            url = url[:-1]
            return execute_request(command_executor, url, param, command_result, POST)[0]
    else:
        return delete_device_object_executor(command_executor)
    
def remove_interface_from_inline_set_executor(command_executor):
    '''
    Indepentant call that can be made. Needs to be able to function independantly and work that way.
    param_executor will get the json and the json only

    @param command_executor: Holds command and dispatcher information.
    @return: Success or Faults results.
    '''
    command_executor.command_holder.response_parser = return_json
    if isinstance(command_executor.command_holder.param_value, dict) and len(command_executor.command_holder.param_value) > 0:
        for command_executor.command_holder.param_value in command_executor.command_holder.param_value.itervalues():
            break
    command_executor.command_holder.param_value = {"name":command_executor.command_holder.param_value, "expanded":"true"}
    errs = param_executor(command_executor)
    if errs and errs.__len__() > 0 and errs[0] is not None:
        return [None]
    j = command_executor.command_holder.param_uuid
    if j.has_key('items') and len(j['items']) == 1: #we have to find items and its length should be 1 if not then we have a problem and cannot execute further
        j = j['items'][0]
    else:
        return [None]
    
    interface = command_executor.command_holder.parent.get_name()
    if not j.has_key("inlinepairs"):
        return [None]
    for pairs in j['inlinepairs']:
        if pairs['first']['name'] == interface or pairs['second']['name'] == interface:
            j['inlinepairs'].remove(pairs)
            break
    
    j = remove_unneeded_keys_from_json(j)
    url = command_executor.dispatch.url + command_executor.command_holder.get_url(domainUUID=command_executor.dispatch.domain_uuid, deviceUUID=command_executor.dispatch.device_uuid)
    
    return execute_request(command_executor, url + "/" + j['id'], json.dumps(j, cls=CommandInteraction.CommandInteractionEncoder), None, PUT)[0]

def acrule_update_comments(command_executor):
    '''
    We will update the acrule newComments line after evaluvating the 
    '''
    parent = command_executor.command_holder.parent #we need the parent objecct that has some of the information we need
    command_executor.command_holder.param_uuid = []
    if parent.state == State.MODIFY or parent.state == State.NOCHANGE:#this is just an update so we shall just set this CommandParam and then
        return [None]
    
    if parent.does_id_exist():
        try:
            if parent.dataParam.param_value.has_key('commentHistoryList'):
                old_comments_list = parent.dataParam.param_value['commentHistoryList']
            else:
                old_comments_list = []
            aci_gen = False
            for comments in reversed(old_comments_list):
                if ':::' in comments['comment']:#this is our key:%s:
                    if (':' + asciistr(command_executor.command_holder.param_value) + ':') in comments['comment'] and parent.state == State.DESTROY: #set to delete
                        command_executor.command_holder.param_uuid.append(comments['comment'].replace(":%s:" % command_executor.command_holder.param_value, ''))
                    elif parent.state != State.DESTROY:
                        command_executor.command_holder.param_uuid.append(comments['comment'] + ":%s:" % command_executor.command_holder.param_value)
                    aci_gen = True
                    break
            if not aci_gen:
                command_executor.command_holder.param_uuid.append(":::USER_GENERATED;ACI:%s:" % command_executor.command_holder.param_value)
        except:
            pass
    else:
        #We Need to create the comments instead of updating the old ones
        command_executor.command_holder.param_uuid.append(":::ACI_GENERATED;ACI:%s:" % command_executor.command_holder.param_value)
    return [None]

def acpolicy_update_description(command_executor):
    '''
    We will update the acpolicy description field
    '''
    
    parent = command_executor.command_holder.parent #we need the parent objecct that has some of the information we need
    if parent.does_id_exist():
        try:
            comments = parent.dataParam.param_value['description']
        except:
            comments = ""
        if ':::' in comments:
            if (":%s:" % command_executor.command_holder.param_value) in comments: #GraphID already exists
                command_executor.command_holder.param_uuid =  comments
                return [None]
            command_executor.command_holder.param_uuid =  comments + ":%s:" % command_executor.command_holder.param_value
        else: #This existed before and we have not made this
            command_executor.command_holder.param_uuid = ":::USER_GENERATED;ACI:%s:" % command_executor.command_holder.param_value
        
    else:#THis is a brand new Policy
        command_executor.command_holder.param_uuid = ":::ACI_GENERATED;ACI:%s:" % command_executor.command_holder.param_value
    
    return [None]
    
def acpolicy_update_default_action(command_executor):
    '''
    We will update the defaultAction 
    '''
    parent = command_executor.command_holder.parent #we need the parent object that has some of the information we need
    if not command_executor.command_holder.param_uuid == "": #this is the second time running this. We already have the information that we need
        return [None]
    if parent.does_id_exist():
        default_action = parent.dataParam.param_value['defaultAction']
        command_executor.command_holder.param_command += default_action['id']
        command_executor.command_holder.param_value = None
        error = param_executor(command_executor)[0]
        if command_executor.command_holder.param_uuid is None or command_executor.command_holder.param_uuid == '':
            return error
        action = command_executor.command_holder.param_uuid
        command_executor.command_holder.param_uuid = default_action
        command_executor.command_holder.param_uuid['action'] = action
        
    else:
        command_executor.command_holder.param_uuid = {"action" : "BLOCK"}
        
    return [None]
        
    
def delete_device_object_executor(command_executor):
    '''
    This function is called to execute delete inline set requests.

    @param command_executor: Holds command and dispatcher information.
    @return: Success or Faults results.
    '''
    return execute_request(command_executor, None, "", None, DELETE)[0]

def delete_interface_executor(command_executor):
    '''
    This function is called to execute delete interface requests.

    @param command_executor: Holds command and dispatcher information.
    @return: Success or Faults results.
    '''
    return execute_request(command_executor, None, "", None, DELETE)[0]

def delete_acpolicy_executor(command_executor):
    '''
    Removes the appropriate description. Does not remove the access policy
    '''
    #We know that this exists otherwise we would not be able to be here
    if not command_executor.command_holder.does_id_exist():
        return [None]
    data = command_executor.command_holder.dataParam.param_value
    ac_id = command_executor.command_holder.params['top_id'].param_value

    if data.has_key('description'):
        if ':::' in data['description']:
            data['description'] = data['description'].replace(':' + asciistr(ac_id) + ':', '')
            
    return execute_request(command_executor, None, None, None, PUT)[0]

def delete_acrule_executor(command_executor):
    '''
    Will Delete the access rule if there is no other comments. Otherwise will update the comments
    '''
    re_str = None
    
    if not command_executor.command_holder.does_id_exist():
        return [None] #shouldnt get here but checking just in case
    data = command_executor.command_holder.params
    if not data.has_key("new_comments"):
        return [None] #also should not be able to hit, But can happen if there was a weird config
    
    comments = data['new_comments'].param_uuid
    if len(comments) == 0:
        re_str = "found"
    for comment in reversed(comments):
        if ':::' in comment:
            re_str = re.search(":.+?_.+?:", comment)
            if ':::USER_GENERATED;ACI' in comment or not ':::ACI_GENERATED;ACI' in comment:
                re_str = "found"
            
    if re_str is None:
        #we delete
        return execute_request(command_executor, None, "", None, DELETE)[0]
    else:
        #mark the security zones for removal in the json
        try:
            acrule_data = data['STORE'].param_value
            graph_identifier = data['new_comments'].param_value
            destination_zone = acrule_data['destinationZones']['objects']
            source_zone = acrule_data['sourceZones']['objects']
            for zone in destination_zone:
                if graph_identifier in zone['name']:
                    command_executor.command_holder.add_removable('destinationZones', zone['name'])
            for zone in source_zone:
                if graph_identifier in zone['name']:
                    command_executor.command_holder.add_removable('sourceZones', zone['name'])
        except:
            pass
        
        return execute_request(command_executor, None, None, None, PUT)[0]

def check_if_deployable(command_executor):
    '''
    This function will check if the device is deployable
    '''
    command_executor.executor = check_if_deployable

    command_result = CommandResult(
                cli='applyChanges',
                err_type='error',
                err_msg=None,
                model_key=[(0, '', None)])
    command_result.lookup_key = None
    command_executor.command_holder.command = "applyChanges"
    command_executor.command_holder.response_parser = return_json
    command_result.resource = command_executor.command_holder
    command_result.resource_key = 'param_uuid'
    
    url = command_executor.dispatch.url + command_executor.command_holder.get_url(domainUUID=command_executor.dispatch.domain_uuid, url ="fmc_config/v1/domain/<domainUUID>/deployment/deployabledevices?expanded=true")
    res = execute_request(command_executor, url, "", command_result, GET)[0]
    if res is None or res[0] is None:
        pass
    else:
        return res
    
    found = [None]
    try:
        for device in command_executor.command_holder.param_uuid['items']:
            if asciistr(device['name']) == command_executor.dispatch.device_name:
                found = None
                env.set_variable(env.STATUS_DEPLOYABLE, 'True')
                env.set_variable(env.DEPLOYMENT_VERSION, asciistr(device['version']))
                return found
            else: # In the case of HA, see if it has a member device matching dispatch.device_name
                deviceMembers = 'deviceMembers'
                if device.has_key(deviceMembers):
                    for member in device[deviceMembers]:
                        if asciistr(member['name']) == command_executor.dispatch.device_name:
                            found = None
                            env.set_variable(env.STATUS_DEPLOYABLE, 'True')
                            env.set_variable(env.DEPLOYMENT_VERSION, asciistr(device['version']))
                            command_executor.dispatch.device_uuid = device['device']['id']
                            return found
    except:
        env.set_variable(env.STATUS_DEPLOYABLE, 'False')
        return [None]
    return found

def check_if_deployed(command_executor):
    """
    this function will check to see if the device deployed or not given the task id
    """
    command_executor.executor = check_if_deployed
    command_executor.command_holder.dataParam = command_executor.command_holder.add_data_param("", "STORE", addToJson=False)
    command_result = CommandResult(
                cli='applyChanges',
                err_type='error',
                err_msg=None,
                model_key=[(0, '', None)])
    command_result.lookup_key = None
    command_executor.command_holder.command = "applyChanges"
    command_executor.command_holder.response_parser = parse_response_for_target_and_store_json
    command_executor.command_holder.response_parser_arg ={"target":"status", "store":command_executor.command_holder.dataParam}
    command_result.resource = command_executor.command_holder
    command_result.resource_key = 'param_uuid'

    url = command_executor.dispatch.url + command_executor.command_holder.get_url(domainUUID=command_executor.dispatch.domain_uuid, url ="fmc_config/v1/domain/<domainUUID>/job/taskstatuses/<task_id>")
    ret = ["something"]
    i = 0
    while i < 30: #we retry 30 times with 10 seconds interval, so 300 seconds = 5 minutes.
        ret = execute_request(command_executor, url, "", command_result, GET)[0]
        if not (ret is None or ret[0] is None):
            return ret
        if not command_executor.command_holder.param_uuid == "Deploying":
            if not command_executor.command_holder.param_uuid == "Deployed":
                return [CommandResult(
                                     cli = command_result.cli,
                                     err_type = command_result.err_type,
                                     err_msg = command_executor.command_holder.dataParam.param_value['message'],
                                     model_key = command_result.model_key)]
            else:
                return [None]
        sleep(10)#we wait in between each call
    
    return [CommandResult(
                         cli = command_result.cli,
                         err_type = command_result.err_type,
                         err_msg = "Deploy Timed Out. Took longer than 5 minutes.",
                         model_key = command_result.model_key)]


def apply_changes_executor(command_executor):
    '''
    This function is called to execute device changes apply requests.

    @param dispatcher: Holds device information.
    @return: Success or Faults results.
    '''
    def apply_changes_error_parse(response, command_executor):
        if response.status_code == 500:
            return "Unable to deploy configuration changes to device. Possible reasons could be that another deployment is in progress or APIC and FMC times are out of sync. Please ensure to sync their time to the same NTP service, setup their time zones, and retry by re-attaching the service graph."
        
    command_executor.executor = apply_changes_executor
    url = command_executor.dispatch.url + command_executor.command_holder.get_url(domainUUID=command_executor.dispatch.domain_uuid, url ="fmc_config/v1/domain/<domainUUID>/deployment/deploymentrequests")
    param_json = get_apply_json(command_executor)
    command_result = CommandResult(
                cli='applyChanges',
                err_type='error',
                err_msg=None,
                model_key=[(0, '', None)])
    command_result.lookup_key = None
    command_executor.command_holder.command = "applyChanges"
    command_executor.command_holder.error_parser = apply_changes_error_parse
    command_executor.command_holder.response_parser = return_json
    command_result.resource = command_executor.command_holder
    command_result.resource_key = 'param_uuid'
    
    return execute_request(command_executor, url, param_json, command_result, POST, deploy_request=True)[0] 

def apply_changes_executor_handler(command_executor):
    """
    Runs 3 functions in a order so that the whole apply change functionality happens crisply
    We first check if the device is deployable
    Then we deploy the device
    Then we see when the deploy is finished
    """
    sleep(10) #sleep for 10s so that any previous changes gets handled by the FMC
    res = check_if_deployable(command_executor)
    if res == [None]:
        return None
    elif res is None:
        pass
    else:
        return None
    
    ret = apply_changes_executor(command_executor)
    if not (ret is None or ret[0] is None):
        return ret
    
    # When deleting a tenant, observed that sometimes APIC send a clusterAudit immediately after a serviceModify,
    # and there we might see missing 'metadata' key, so adding a check here.
    if command_executor.command_holder.param_uuid.has_key('metadata'):
        task_id = command_executor.command_holder.param_uuid['metadata']['task']['id']
        command_executor.command_holder.add_data_param(task_id, "task_id")
        return check_if_deployed(command_executor)

def interface_health_executor(command_executor): # pragma: no cover
    '''
    This function is called to execute device changes apply requests.

    @param command_executor: Holds command and dispatcher information.
    @return: Success or Faults results.
    '''
    'This is read only FSMC API calls. So no need to apply changes in FSMC.'
    command_executor.dispatch.is_apply_changes = False
    
    results = [None] * len(command_executor.command_holder.params)
    count = 0
    url = command_executor.dispatch.url + command_executor.command_holder.get_url(domainUUID=command_executor.dispatch.domain_uuid, deviceUUID=command_executor.dispatch.device_uuid)
    for paramValue in command_executor.command_holder.params.itervalues():
        response = execute_request(command_executor, url, paramValue.param_value, None)[1]
        success = CommandResult(cli="Health", err_type="success", err_msg=response.text, model_key=paramValue.param_value)
        results[count] = success
        count = count + 1
    return results

def parse_connector_counters(show_output): # pragma: no cover
    '''
    This function is called to parse interface data to health counters.

    @param show_output: Response JSON.
    @return: Result dictionary.
    '''
    # The NGIPS doesn't have support for these counters
    result = {
        'rxerrors': 0,
        'txerrors': 0,
        'txdrops': 0
    }
    try:
        json_out = json.loads(show_output)
        json_out = json_out["items"][0]
        result['rxpackets'] = json_out['rxBytes']
        result['txpackets'] = json_out['txBytes']
        result['rxdrops'] = json_out['droppedPackets']
    except:
        pass
    return result
