'''
Created on Aug 3, 2016

@author: Akshay Ahluwalia

Copyright (c) 2016 by Cisco Systems, Inc.
All rights reserved.

Classes used for Access Rules.
'''
from devpkg.base.dmobject import DMObject
from devpkg.base.dmlist import DMList
from devpkg.utils.state_type import State

from fmc.parsers import device_object_executor, return_json
from devpkg.base.command_interaction import CommandInteraction
from devpkg.base.simpletype import SimpleType
from fmc.parsers import delete_acrule_executor, acrule_update_comments

class AccessRule(DMObject):
    """
    Access Rule Class that makes the cli
    """
    GENERAL_AC_RULE = """
    {
        "name" : "AccessRule",
        "type" : "AccessRule",
        "enabled" : true,
        "action" : "BLOCK"      
    }
    """
    
    def __init__(self, instance, probe):
        super(AccessRule, self).__init__(instance, dm_key='AccessRule', probe=probe)
        self.dm_key_command='createAccessRule'
        self.dm_key = 'AccessRule'
        self.command_executor=device_object_executor
        self.register_child(SimpleType('bidirectional', 'bidirectional', defaults='false'))
        self.register_child(SimpleType('AccDestinationZones', 'accDestinationZones', defaults={}))
        self.register_child(SimpleType('AccSourceZones', 'accSourceZones', defaults={}))
        self.dependent_on = []
        self.probe = probe

    def get_existing_sz_to_keep(self, rule_name, removable):
        src_zones = []
        dest_zones = []
        for rule in self.probe.config_keeper.get('accessrules'):
            if rule['name'] == rule_name:
                if rule.has_key('sourceZones') and len(rule['sourceZones']) > 0:
                    src_zones.extend([sz for sz in rule['sourceZones']['objects'] if not removable.has_key('sourceZones') or sz['name'] not in removable['sourceZones']])
                if rule.has_key('destinationZones') and len(rule['destinationZones']) > 0:
                    dest_zones.extend([sz for sz in rule['destinationZones']['objects'] if not removable.has_key('destinationZones') or sz['name'] not in removable['destinationZones']])
        return src_zones, dest_zones
            
    def update_sz_for_ar(self, clii, local_dm_cfg_list):
        """
        SZ needs to be set to how the access rule will be
        Called Internally
        @param clii: the clii
        @param local_dm_cfg_list: the local_dm_cfg_list
        @return: all_sz_dest, all_sz_source
        """
        def setup_sz_for_zone(zone, clii, local_dm_cfg_list):
            """
            done twice so put into a function. adds sz that need to be updated that the AR directly known/cares about.
            """
            sz_list = []
            try:
                if zone == 'AccSourceZones':
                    #we are checking to see if this is an AccSourceZones or a AccDestinationZones
                    if local_dm_cfg_list[0].params[zone].param_value.has_key('SourceZones'):#we are checking to see if there is a SoueceZones. IF not then it was removed
                        acc_source_zones = local_dm_cfg_list[0].params[zone].param_value['SourceZones'] #get child SourceZones
                    else:
                        acc_source_zones = {}
                else:
                    #same as above for the DestionationZones
                    if local_dm_cfg_list[0].params[zone].param_value.has_key('DestinationZones'):
                        acc_source_zones = local_dm_cfg_list[0].params[zone].param_value['DestinationZones'] #get child DestinationZones
                    else:
                        acc_source_zones = {}
                if not isinstance(acc_source_zones, list): #we only deal with lists
                    acc_source_zones = [acc_source_zones]
                for sz in acc_source_zones:
                    try:
                        source_sz = self.find("/InterfaceConfig/%s" % sz) #Gets the SecurityZone
                        source_sz_target = source_sz.get_value() #get the name
                        source_sz = self.find("/SecurityZone/"  + source_sz_target['security_zone']).cli #get the cli of that SZ
                        source_sz_object = {"name" : source_sz.params['GET'].param_value['name'], 'id' : source_sz.idParam, "type": "SecurityZone"} #setup the SZ dictionary for insertion
                        if self.find("/SecurityZone/"  + source_sz_target['security_zone']).get_state() != State.DESTROY: #if not destroy
                            sz_list.append(source_sz_object)
                        else:
                            clii.add_removable('sourceZones', source_sz.params['GET'].param_value['name']) #we set to delete from the parameter
                            clii.add_removable('destinationZones', source_sz.params['GET'].param_value['name'])
                    except:
                        pass
            except:
                pass
            
            return sz_list
            
        all_sz_source = setup_sz_for_zone("AccSourceZones", clii, local_dm_cfg_list)
        all_sz_dest = setup_sz_for_zone("AccDestinationZones", clii, local_dm_cfg_list)
       
        #add all SZ that are set to delete as removable from this AR
        try:
            all_known_sz = self.find("/SecurityZone")
            for szn in all_known_sz.children.iterkeys():
                temp = self.find("/SecurityZone/%s" % szn)
                if temp.get_state() == State.DESTROY:
                    if not self.does_string_have_unique_graph_string(szn):
                        szn += self.get_unique_graph_string_append()
                    clii.add_removable('sourceZones', szn)
                    clii.add_removable('destinationZones', szn)
                    
        except:
            pass
        
        if len(all_sz_source) == 0: #empty list meaning we will remove all SZ from this
            for szn in all_known_sz.children.iterkeys():
                if not self.does_string_have_unique_graph_string(szn):
                        szn += self.get_unique_graph_string_append()
                        clii.add_removable('sourceZones', szn)
                        
        if len(all_sz_dest) == 0: #same as above
            for szn in all_known_sz.children.iterkeys():
                if not self.does_string_have_unique_graph_string(szn):
                        szn += self.get_unique_graph_string_append()
                        clii.add_removable('destinationZones', szn)
        return all_sz_dest, all_sz_source
                 
    def ifc2dm(self, no_dm_cfg_stack, dm_cfg_list):
        """
        construct the cli
        """
        try:
            source = self.find('AccSourceZones').get_value()['SourceZones']
            security_zone = []
            if not isinstance(source, list):
                source = [source]
            for sr in source:
                self.dependent_on.append('/InterfaceConfig/%s' % sr)
                security_zone.append(self.find('/InterfaceConfig/%s' % sr).get_value()['security_zone'])
            
            destination = self.find('AccDestinationZones').get_value()['DestinationZones']
            if not isinstance(destination, list):
                destination = [destination]
            for dt in destination:
                self.dependent_on.append('/InterfaceConfig/%s' % dt)
                security_zone.append(self.find('/InterfaceConfig/%s' % dt).get_value()['security_zone'])
            
            for sz in security_zone:
                self.dependent_on.append('/SecurityZone/%s' % sz)
            
        except:
            pass
        
        all_security_zones = self.find("/SecurityZone")
        self.recursive_state = self.get_state_recursive()
        
        
        local_dm_cfg_list = []
        if not self.has_ifc_delta_cfg():
            return
        super(AccessRule, self).ifc2dm(no_dm_cfg_stack, local_dm_cfg_list)
        num, pol, name = self.delta_ifc_key
        clii = None      
        if self.get_state() == State.CREATE or self.get_state() == State.MODIFY:
            clii = CommandInteraction(self.dm_key, model_key=self.get_config_path(), probe=self.probe)
            clii.add_data_param(self.parent.parent.cli.idParam, 'acUUID', False)
            clii.add_basic_interaction('fmc_config/v1/domain/<domainUUID>/policy/accesspolicies/<acUUID>/accessrules', name, device_object_executor, AccessRule.GENERAL_AC_RULE)
            try:
                bidirec = local_dm_cfg_list[0].params['bidirectional'].param_value
            except:
                bidirec = "false"
                
            if bidirec == "true":
                bidirec = True
            else:
                bidirec = False
            all_sz_dest, all_sz_source = self.update_sz_for_ar(clii, local_dm_cfg_list)
            
            clii.add_data_param(name, "name")
            clii.add_data_param("true", "enabled")

            clii.add_param("new_comments", self.get_top_uuid(), acrule_update_comments, "", "newComments")
            if bidirec:
                temp = all_sz_dest + all_sz_source 
                all_sz_dest = temp
                all_sz_source = temp

            #we have to mark for removal of all SZ that we might have attached in the FMC to this AccessRule that we control that are not explicitly specified now
            for sz in all_security_zones.children:
                sz += self.get_unique_graph_string_append()
                if not sz in [s['name'] for s in all_sz_source if (not clii.removable.has_key('sourceZones') or s['name'] not in clii.removable['sourceZones'])]: 
                    clii.add_removable("sourceZones", sz)
                if not sz in [s['name'] for s in all_sz_dest if (not clii.removable.has_key('destinationZones') or s['name'] not in clii.removable['destinationZones'])]:
                    clii.add_removable("destinationZones", sz)
                    
            # remove removable sz from all_sz_source and all_sz_dest
            all_sz_dest_updated = [sz for sz in all_sz_dest if sz['name'] not in clii.removable]
            all_sz_source_updated = [sz for sz in all_sz_source if sz['name'] not in clii.removable]

            # get existing zones to keep, and add them to the new updated zones, that'd be the final zone list we'd like to put in the rule
            # for example, if existing_src_zones=['zone1'], and all_sz_source_updated=['zone2'], then the final all_sz_source_updated=['zone1', 'zone2']
            existing_src_zones, existing_dest_zones = self.get_existing_sz_to_keep(name, clii.removable)
            src_sz_names = [sz['name'] for sz in all_sz_source_updated]
            dest_sz_names = [sz['name'] for sz in all_sz_dest_updated]
            for sz in existing_src_zones:
                if sz['name'] not in src_sz_names:
                    # add to the final list if it's not there yet
                    all_sz_source_updated.append(sz)
            for sz in existing_dest_zones:
                if sz['name'] not in dest_sz_names:
                    # add to the final list if it's not there yet
                    all_sz_dest_updated.append(sz)
            if len(all_sz_dest_updated) > 0:
                clii.add_data_param({"objects" : all_sz_dest_updated}, "destinationZones")
            
            if len(all_sz_source_updated) > 0:
                clii.add_data_param({"objects" : all_sz_source_updated}, "sourceZones")
                
                
        elif self.get_state() == State.DESTROY: 
            clii = CommandInteraction(self.dm_key, model_key=self.get_config_path(), probe=self.probe)
            clii.add_data_param(self.parent.parent.cli.idParam, 'acUUID', False)
            clii.add_basic_interaction('fmc_config/v1/domain/<domainUUID>/policy/accesspolicies/<acUUID>/accessrules', name, delete_acrule_executor, '')
            
            clii.add_param("new_comments", self.get_top_uuid(), acrule_update_comments, "", "newComments")

            all_sz_dest, all_sz_source = self.update_sz_for_ar(clii, local_dm_cfg_list)

            if len(all_sz_dest) > 0:
                clii.add_data_param({"objects" : all_sz_dest}, "destinationZones")
            
            if len(all_sz_source) > 0:
                clii.add_data_param({"objects" : all_sz_source}, "sourceZones")
                    
        if clii is not None:
            self.add_cli_to_array(clii, dm_cfg_list)

class AccessRules(DMList):
    'A list of Interfaces'

    def __init__(self, name = AccessRule.__name__, child_class = AccessRule, probe=None):
        super(AccessRules, self).__init__(name, child_class, probe=probe)

    def ifc2dm(self, no_dm_cfg_stack,  dm_cfg_list):
        super(AccessRules, self).ifc2dm(no_dm_cfg_stack,  dm_cfg_list)