'''
Copyright (c) 2013 by Cisco Systems, Inc.

@author: emilymw
'''

import devpkg.base.command_interaction as command_interaction
from devpkg.base.dmlist import DMList
from devpkg.base.dmobject import DMObject
import devpkg.utils.util as util
from devpkg.utils.state_type import State, Type
from utils.util import normalize_network_address
from utils.util import netmask_from_prefix_length

from devpkg.utils.errors import DeviceConfigError, FaultCode

class Connector(DMObject):
    '''
    A network service can contains two connectors of type external and internal.  A
    Connector in terms of device is expressed by a VLAN interface.

    There are several ConnObj (VIF, VLAN, ENCAPASS) objects defined in the global
    configuration under Device Config.  These objects binds the vlan tag with the interface.
    '''

    def __init__(self, instance):
        super(Connector, self).__init__(instance)
        self._interface = None

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        '''
        Populate the Connector configuration
        '''
        self.delta_ifc_key = delta_ifc_key
        self.delta_ifc_cfg_value = delta_ifc_cfg_value
        self.state = delta_ifc_cfg_value['state']
        self.config = util.normalize_param_dict(delta_ifc_cfg_value['value'])
        self.conn_type = self.delta_ifc_key[1]

    @property
    def interface(self):
        if not self._interface and self.has_ifc_delta_cfg():
            intf_config_rel = self.get_intf_config_rel()
            if hasattr(intf_config_rel, 'value'):
                self._interface = (self.get_top().get_child('InterfaceConfig').
                                   get_child(intf_config_rel.value))
        return self._interface
        
    def get_nameif(self):
        return self.interface.nameif if self.interface else ''

    def get_intf_config_rel(self):
        if self.conn_type == 'external':
            return self.parent.parent.get_child('ExIntfConfigRelFolder').get_child('ExIntfConfigRel')
        else:
            return self.parent.parent.get_child('InIntfConfigRelFolder').get_child('InIntfConfigRel')

    def ifc2dm(self, no_dm_cfg_stack, dm_cfg_list):
        'Update the corresponding interface with VIF and VLAN information'
        if self.interface:
            top = self.get_top()
            encaprel = self.config.get('ENCAPREL')
            encapstr = encaprel
            if isinstance(encaprel, list):
                for item1 in encaprel:
                    if 'any' not in item1:
                        encapstr = item1
            encapass = top.get_child('ENCAPASS').get_child(encapstr)
            if encapass:
                self.interface.vif = top.get_child('VIF').get_child(encapass.vif).value
                self.interface.vlan = top.get_child('VLAN').get_child(encapass.encap).tag
                if self.state == State.DESTROY:
                    self.interface.set_state(self.state)
                
                ''' In case of wrong interface corrected. Remove the comment, it will make it work.
                if top.get_child('VIF').get_child(encapass.vif).get_state() != State.NOCHANGE and top.get_child('VIF').get_child(encapass.vif).get_state() != None:
                    self.interface.recursive_state = top.get_child('VIF').get_child(encapass.vif).get_state()
                if top.get_child('VLAN').get_child(encapass.encap).get_state() != State.NOCHANGE and top.get_child('VLAN').get_child(encapass.encap).get_state() != None:
                    self.interface.recursive_state = top.get_child('VLAN').get_child(encapass.encap).get_state()
                '''
        """
        The reason of calling populate_epg_data here instead of populate_model is because
        populate_epg_data can change the configuration dictionary that populate_model iterates through.
        One cannot iterate through a dictionary whose entries are changing.
        """
        if self.get_state() != State.NOCHANGE:
            self.populate_epg_data()
    
    def populate_epg_data(self):
        'Populate EPG membership updates to the corresponding NetworkObjectGroup'
        """
        For audit operation, this method will be called twice for the same connector.
        Use 'epg_processed' in the configuration dictionary as a flag, to avoid
        doing it twice to the configuration data.
        """
        if self.delta_ifc_cfg_value.get('epg_processed'):
            return
        for epg, members in self.get_epgs().iteritems():
            self.populate_network_object_group(epg, members)
        self.delta_ifc_cfg_value['epg_processed'] = True

    def get_epgs(self):
        """
        Get all the EPGs concerned by the connector.
        @return a dictionary, keyed by EPG name. The value of each entry is list of members.
        Each member is a tuple of (state, type, value)
        """
        epgs = {}
        for key, value in self.delta_ifc_cfg_value['value'].iteritems():
            if key[0] not in (Type.ENDPOINT, Type.SUBNET):
                continue
            state = value['state']
            network_value = key[2]
            network_type = key[0]
            'epg value will be tenant-profile-epg'
            epg = '-'.join(map(value.get, filter(value.has_key, ('tenant', 'profile', 'epg'))))
            if not epgs.has_key(epg):
                epgs[epg] = []
            epgs[epg].append((state, network_type, network_value))
        return epgs

    def populate_network_object_group(self, epg, members):
        'convert EPG membership update information to the corresponding NetworkObjectGroup'
        def get_entry_key_value(state, network_type, network_value):
            if network_type == Type.ENDPOINT:
                key = 'host_ip_address'
                value = network_value
            else:
                (network_address, prefix_length) = (network_value.split('/'))
                key = 'network_ip_address'
                if '.' in network_address: #IPv4 Address
                    mask = netmask_from_prefix_length(prefix_length)
                    value = normalize_network_address(network_address, mask) + '/' + mask
                else: #IPv6 address
                    value = normalize_network_address(network_address, prefix_length) + '/' + prefix_length
            return ((Type.PARAM, key, value), {'state': state, 'value': value})

        name = self.epg_name2nog_name(epg)
        top = self.get_top()
        config = top.delta_ifc_cfg_value
        nog_key = (Type.FOLDER, 'NetworkObjectGroup', name)
        if not config['value'].has_key(nog_key):
            nog_config = {'state': self.state, 'value': {}}
            top.delta_ifc_cfg_value['value'][nog_key] = nog_config
        else:
            nog_config = config['value'][nog_key]
        for state, network_type, network_value in members:
            entry_key, entry_value = get_entry_key_value(state, network_type, network_value)
            old_value = nog_config['value'].get(entry_key)
            if not old_value or state != State.NOCHANGE:
                'Make sure we do not override the state with 0 to workaround CSCuv87932.'
                nog_config['value'][entry_key] = entry_value
        nework_object_groups = top.get_child('NetworkObjectGroup')
        nework_object_groups.populate_model(nog_key, nog_config)
        
    @staticmethod
    def epg_name2nog_name(epg):
        'Make an network object group name from an EPG name'
        return '__EPG_%s' % epg
    
    'TODO: Check if I am required!!'            
    def get_vif(self):
        self.ifc2dm(None, None)
        return (self.interface.vif, self.interface.vlan)

class ConnObj(DMObject):
    '''
    This is a base class for all connector objects that are defined under
    Device Config (VIF, VLAN, ENCAPASS, InterfaceConfig).  It overwrite the populate_model()
    and ifc2dm() method. The Connector will handle the generation of call
    instead of handling it in these objects.

    Note:  ConnObj are not specified in device specification, they are
    generated by the IFC, except InterfaceConfig which is a relationship binding
    interface to ip address/mask.
    '''
    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        self.delta_ifc_key = delta_ifc_key
        self.delta_ifc_cfg_value = delta_ifc_cfg_value
        self.state = delta_ifc_cfg_value['state']
        self.response_parser = command_interaction.ignore_info_response_parser

    def ifc2dm(self, no_dm_cfg_stack, dm_cfg_list):
        'Handled by Connector'
        pass

class Vifs(DMList):
    def __init__(self):
        super(Vifs, self).__init__('VIF', Vif, 'interfaceconfig')

class Vif(DMObject):
    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        super(Vif, self).populate_model(delta_ifc_key, delta_ifc_cfg_value)
        if delta_ifc_cfg_value['cifs']:
            cif = delta_ifc_cfg_value['cifs'].iteritems().next()[1]
        else:
            raise DeviceConfigError([([(0, '', None)], FaultCode.CONFIGURATION_ERROR, "Device interface configuration missing.")])
        self.value = util.normalize_interface_name(cif)

class Vlan(ConnObj):
    '''
    Describe the VLAN tag
    '''
    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        '''
        Populate model
        '''
        super(Vlan, self).populate_model(delta_ifc_key, delta_ifc_cfg_value)
        self.type = str(delta_ifc_cfg_value.get('type'))
        '''
        hardcode to avoid triggering vxlan code in brownfield because an APIC bug
        which is still generating 1 instead of 0
        '''
        self.type = '0'
        self.tag = delta_ifc_cfg_value.get('tag')

class EncapAss(ConnObj):
    '''
    Present the association of VLAN and VIF
    '''
    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        super(EncapAss, self).populate_model(delta_ifc_key, delta_ifc_cfg_value)
        self.vif = delta_ifc_cfg_value['vif']
        self.encap = delta_ifc_cfg_value['encap']

class IntfConfigRel(DMObject):
    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        self.delta_ifc_key = delta_ifc_key
        self.delta_ifc_cfg_value = delta_ifc_cfg_value
        self.state = delta_ifc_cfg_value['state']
        'Will deprecate check for target once the target is replaced with value in the initialization'
        if 'target' in delta_ifc_cfg_value:
            self.value = delta_ifc_cfg_value['target']
        else:
            self.value = delta_ifc_cfg_value['value']

class ExIntfConfigRel(IntfConfigRel):
    def __init__(self):
        DMObject.__init__(self, ExIntfConfigRel.__name__)

class InIntfConfigRel(IntfConfigRel):
    def __init__(self):
        DMObject.__init__(self, InIntfConfigRel.__name__)

class ExIntfConfigRelFolder(DMObject):
    'A list of additional interface parameters for external Connectors'

    def __init__(self):
        DMObject.__init__(self, ExIntfConfigRelFolder.__name__)
        self.register_child(ExIntfConfigRel())

class InIntfConfigRelFolder(DMObject):
    'A list of additional interface parameters for internal Connectors'

    def __init__(self):
        DMObject.__init__(self, InIntfConfigRelFolder.__name__)
        self.register_child(InIntfConfigRel())

