'''
Created on Sep 1, 2015

@author: jeffryp

Copyright (c) 2015 by Cisco Systems, Inc.
All rights reserved.

Classes used for security object group configuration.
'''

import re

from asaio.cli_interaction import object_response_parser
from base.compositetype import CompositeType, Description
from base.simpletype import SimpleType
from object_group import ObjectGroupList, ObjectList
from state_type import State, Type
from structured_cli import StructuredCommand

class CommonObject(SimpleType):
    '''
    Common object configuration
    '''

    def __init__(self, instance, asa_key):
        super(CommonObject, self).__init__(instance, asa_key)
        self.response_parser = object_response_parser

    def create_asa_key(self):
        return self.get_cli()

class GroupObject(CommonObject):
    '''
    Group object configuration
    '''

    ASA_KEY = 'group-object'

    def __init__(self, instance):
        super(GroupObject, self).__init__(instance, self.ASA_KEY)

class Name(CommonObject):
    '''
    Security group name configuration
    '''

    ASA_KEY = 'security-group name'

    def __init__(self, instance):
        super(Name, self).__init__(instance, self.ASA_KEY)

class Tag(CommonObject):
    '''
    Security group tag configuration
    '''

    ASA_KEY = 'security-group tag'

    def __init__(self, instance):
        super(Tag, self).__init__(instance, self.ASA_KEY)

class SecurityObjectGroup(CompositeType):
    '''
    Security object group configuration
    '''

    ASA_KEY = 'object-group security'
    CLI_PATTERN = None

    def __init__(self, instance):
        super(SecurityObjectGroup, self).__init__(instance, self.ASA_KEY)
        self.response_parser = object_response_parser
        self.register_child(Description())
        self.register_child(SecurityObjectList('security_group_name', Name))
        self.register_child(SecurityObjectList('security_group_tag', Tag))
        self.register_child(SecurityObjectList('security_object_group',
                                               GroupObject))

    def create_asa_key(self):
        return self.get_cli()

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.delta_ifc_key = (Type.FOLDER, self.parent.ifc_key, self.ifc_key)
            self.delta_ifc_cfg_value = {'state': State.DESTROY, 'value': {}}
            ancestor = self.get_ifc_delta_cfg_ancestor()
            if ancestor:
                ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] = self.delta_ifc_cfg_value
        elif self.is_the_same_cli(cli):
            self.set_action(State.NOCHANGE)
        else:
            self.set_action(State.MODIFY)

        if isinstance(cli, StructuredCommand):
            for cmd in cli.sub_commands:
                translator = self.get_child_translator(cmd)
                if translator:
                    translator.diff_ifc_asa(cmd)

    def get_cli(self):
        return self.asa_gen_template % self.ifc_key

    @classmethod
    def get_cli_pattern(cls):
        if not cls.CLI_PATTERN:
            cls.CLI_PATTERN = re.compile(cls.ASA_KEY + ' (\S+)')
        return cls.CLI_PATTERN

    def get_name(self):
        'Used by ObjectGroupList to access the name of the object group'
        return self.ifc_key

class SecurityObjectGroups(ObjectGroupList):
    '''
    A list of security object groups
    '''

    def __init__(self):
        super(SecurityObjectGroups, self).__init__(
            SecurityObjectGroup.__name__, SecurityObjectGroup,
            object_group_key='security_object_group')

    def get_translator(self, cli):
        if isinstance(cli, StructuredCommand):
            cli = cli.command
        m = SecurityObjectGroup.get_cli_pattern().match(cli.strip())
        if m:
            name = m.group(1)
            group = self.get_child(name)
            if not group:
                group = self.child_class(name)
                self.register_child(group)
            return group

class SecurityObjectList(ObjectList):
    '''
    A list of security objects of the same type
    '''

    def __init__(self, name, child_class):
        super(SecurityObjectList, self).__init__(
            name, child_class, child_class.ASA_KEY,
            'show run object-group security')
