'''
Created on June 5, 2014

@author: emilymw
Copyright (c) 2014 by Cisco Systems
'''
from base.dmlist import DMList
from base.simpletype import SimpleType
from base.compositetype import CompositeType
import asaio.cli_interaction as cli_interaction
from structured_cli import StructuredCommand
from state_type import State
from validators import AdminContextValidator
from asaio.cli_interaction import ignore_info_warning_response_parser

class LACPMaxBundles(DMList):
    'Container of LACPMaxBundle'
    def __init__(self, is_multi_mode_asa):
        DMList.__init__(self, name='LACPMaxBundle',
                        asa_key='interface Port-channel',
                        child_class=LACPMaxBundle)
        self.is_system_context = is_multi_mode_asa

    def is_my_cli(self, cli):
        """
        Only handle port-channel main interface command, not the sub-interface command.
        """
        if not isinstance(cli, StructuredCommand):
            return False
        if not cli.command.startswith(self.asa_key):
            return False
        return cli.command.find('.') == -1

    def register_child(self, dmobj):
        DMList.register_child(self, dmobj)
        for child in dmobj.children.values():
            child.is_system_context =  dmobj.is_system_context

class LACPMaxBundle(CompositeType, AdminContextValidator):
    'Model for "interface Port-channel" CLI'
    def __init__(self, name):
        CompositeType.__init__(self, name,
                               asa_gen_template = 'interface Port-channel%(pc_id)s')
        self.register_child(LACPMax())
        self.register_child(SpanCluster())
        self.response_parser = cli_interaction.ignore_info_response_parser

    def port_channel_not_empty(self):
        pc_id = self.get_value()['pc_id']
        port_channels = self.get_top().get_child('PortChannelMember')
        return port_channels.port_channel_not_empty(pc_id)

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        'Override the default so that we will not delete a port-channel if it has member interface'
        if self.get_top().is_fpr_device():
            'For FPR device, the port-channel configuration is done in the MIO'
            return
        if self.get_state() == State.DESTROY:
            if self.port_channel_not_empty():
                self.set_state(State.NOCHANGE)
            else:
                """
                Removal of 'interface port-channel' can fail if a graph is using it.
                Once the graph is removed, the deletion will succeed. See CSCuv20304
                """
                self.response_parser = cli_interaction.ignore_response_parser
        temp_asa_cfg_list = []
        CompositeType.ifc2asa(self, no_asa_cfg_stack, temp_asa_cfg_list)
        if temp_asa_cfg_list:
            no_shut_cmd = []
            self.generate_cli(no_shut_cmd, 'no shutdown', mode_command = self.get_cli())
            asa_cfg_list.extend(no_shut_cmd)
            asa_cfg_list.extend(temp_asa_cfg_list)

    def create_asa_key(self):
        return self.get_cli()

    def diff_ifc_asa(self, cli):
        'Override the default so that we will not delete a port-channel if it has member interface'
        SimpleType.diff_ifc_asa(self, cli)
        if self.get_action() == State.DESTROY:
            'we only delete this port-channel if it does not have any member'
            if not self.port_channel_not_empty():
                return
        if isinstance(cli, StructuredCommand):
            for cmd in cli.sub_commands:
                translator = self.get_child_translator(cmd)
                if translator:
                    translator.diff_ifc_asa(cmd)

    def validate_configuration(self):
        result = CompositeType.validate_configuration(self)
        if not self.has_ifc_delta_cfg():
            return result
        if self.get_state() in (State.CREATE, State.MODIFY) and self.get_top().is_fpr_device():
            err = 'You configure port-channel in the chassis supervisor of the FirePower system manually, not from ASA device package.'
            self.report_fault(err, result)
        return result

class LACPMax(SimpleType):
    def __init__(self):
        SimpleType.__init__(self, ifc_key = 'lacp_max_bundle', asa_key = 'lacp max-bundle')

    def get_default_value(self):
        '''
            (9.1 and earlier) The default is 8.
            (9.2(1) and later) The default is 16.
        '''
        if self.get_top().compare_asa_version(9, 2, 1) < 0:
            return '8'
        return '16'

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if self.get_state() == State.DESTROY:
            '''
            Just issue the default value instead of 'no lacp max-bundle <n>'.
            This is for overcome a problem with ASA 9.2(1) and onwards, where
            the default value is 16, but issusing 'no lacp max-bundel' will result in
            'lacp max-bundle 8' stored in running config.
            '''
            self.generate_cli(asa_cfg_list, self.asa_key + ' ' + self.get_default_value())
        else:
            return SimpleType.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)

    def diff_ifc_asa(self, cli):
        'Override the default implementation to take care of the default value.'
        SimpleType.diff_ifc_asa(self, cli)
        if self.get_state() == State.DESTROY and cli.split(' ')[-1] == self.get_default_value():
            self.set_action(State.NOCHANGE)

class SpanCluster(SimpleType):
    '''Model 'port-channel span-cluster [vss-load-balance]' sub-command under 'interface port-channel'.
    '''
    def __init__(self):
        SimpleType.__init__(self, SpanCluster.__name__, 'port-channel span-cluster',
                            response_parser=ignore_info_warning_response_parser)

    def get_cli(self):
        result = self.asa_key
        value = self.get_value()
        if self.get_state() != State.DESTROY and value.get('vss_load_balance'):
            result += ' vss-load-balance'
        return result
