'''
Created on Oct 19, 2013

@author: feliu
@author: dli
'''

from copy import deepcopy, copy
import re

from translator.base.dmobject import DMObject
from translator.base.simpletype import SimpleType
from translator.base.dmboolean import DMBoolean
from translator.base.compositetype import CompositeType
from translator.state_type import State, Type
from utils.util import filter_first, netmask_from_prefix_length,\
    ifcize_param_dict, set_cfg_state, query_asa, read_clis
from translator.address_pool import IPv4AddressPool, MANAGEMENT_ADDRESS_POOL_NAME
from translator.validators import AdminContextValidator

class ClusterConfig(DMObject, AdminContextValidator):
    '''
    This class represents the holder of all cluster configuration.
    '''

    def __init__(self, device=None, is_multi_mode_asa = False):
        '''
        @param device: device dictionary. It is None for the master unit, and slave unit otherwise
        '''
        DMObject.__init__(self, ClusterConfig.__name__)
        self.register_child(InterfaceMode(device))
        self.register_child(ManAddressPool(device))
        self.register_child(ManInterface(device))
        self.register_child(Bootstrap(device))
        if not device:#only need Advanced for master unit
            self.register_child(Advanced(device))
        self.device = device
        self.response_parser = cluster_response_parser
        for child in self.children.values():
            if not child.ifc_key in [ManAddressPool.__name__, ManInterface.__name__]:
                child.is_system_context = is_multi_mode_asa
        self.is_system_context = is_multi_mode_asa

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        '''Override the default implementation to generate CLIs for both the master unit
        and slave unit(s)
        '''
        if not self.is_apic_managed() or not self.has_ifc_delta_cfg():
            return
        if not self.get_child(Bootstrap.__name__).has_ifc_delta_cfg():
            return
        if self.get_child(Bootstrap.__name__).is_ctrl_intf_changed():
            self.turn_on_ctrl_intf(asa_cfg_list)
        if self.device:#generate configure for a slave unit
            self.ifc2asa_plus(no_asa_cfg_stack, asa_cfg_list)
            return
        if self.get_child(Bootstrap.__name__).get_state() == State.DESTROY and self.get_state() != State.DESTROY:
            set_cfg_state(self.delta_ifc_cfg_value, State.DESTROY)
        #generate CLI for the master unit, and the slave units
        if self.need2disable_cluster():
            '''Two steps process:
            1. On the master unit: disable all cluster units; make desired configuration change; enable cluster
            2. On the slave units: make the desired configuration change; enable cluster
            We do not expose the 'enable' cluster group sub-command to user from ASA-DP.
            '''
            DMObject.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)
            self.enable_cluster(asa_cfg_list)
            self.disable_cluster(no_asa_cfg_stack) #disable CLI will be sent first to the master unit
            self.generate_clis_4_slave_units(no_asa_cfg_stack,asa_cfg_list)
        elif self.get_state() == State.DESTROY or self.is_creating_cluster():
            self.ifc2asa_plus(no_asa_cfg_stack, asa_cfg_list)
            self.generate_clis_4_slave_units(no_asa_cfg_stack,asa_cfg_list)
        else:
            DMObject.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)

    def is_creating_cluster(self):
        '''
        @return True if we are creating cluster configuration (rather than modifying) on ASA.
        '''
        state =  self.get_state()
        if state != State.CREATE:
            return
        if not self.get_top().is_audit:
            return state == State.CREATE
        def is_all_state_create(config):
            '''
            @return True if all the entries are of State.CREATE
            '''
            if config['state'] != State.CREATE:
                return
            if not isinstance(config['value'], dict):
                return True
            for child in config['value'].values():
                if not is_all_state_create(child):
                    return
            return True
        return is_all_state_create(self.delta_ifc_cfg_value)

    def ifc2asa_plus(self, no_asa_cfg_stack, asa_cfg_list):
        '''
        On removing cluster setup on ASA, we need to place the command
        'no cluster interface mode' right after 'clear configure cluster'
        '''
        if self.is_system_context and self.get_state() == State.DESTROY:
            '''
            for multi-context mode, generate the 'no shutdown' of management interface in system context at the end.
            '''
            self.generate_cli(no_asa_cfg_stack,
                              'no shutdown',
                              mode_command='interface ' + self.get_system_context_man_intf_name())
        tmp_no_asa_cfg_stack = []
        DMObject.ifc2asa(self, tmp_no_asa_cfg_stack, asa_cfg_list)
        if self.get_state() == State.DESTROY:
            self.move_no_cluster_interface_mode_cmd(tmp_no_asa_cfg_stack)
        no_asa_cfg_stack.extend(tmp_no_asa_cfg_stack)

    def is_apic_managed(self):
        if self.has_ifc_delta_cfg():
            return self.get_value().get('apic_managed')

    def generate_clis_4_slave_units(self, no_asa_cfg_stack,asa_cfg_list):
        '''
        For each slave unit, generate the CLIs and tag them with the cdev. and also enable cluster as well.
        '''
        slave_cdev_list = self.get_slave_dev_list()
        for dev in slave_cdev_list:
            cluster_config = ClusterConfig(dev, self.is_system_context)
            cluster_config.parent = self.parent
            cluster_config.populate_model(deepcopy(self.delta_ifc_key), deepcopy(self.delta_ifc_cfg_value))
            cluster_config.update_references()
            cluster_config.ifc2asa(no_asa_cfg_stack, asa_cfg_list)
            if self.need2disable_cluster():
                mode = 'cluster group ' + self.get_cluster_group_name()
                cluster_config.generate_cli(asa_cfg_list, 'enable noconfirm', mode_command=mode)

    def get_slave_dev_list(self):
        """
        @return the list of CDevs for the slave unit
        Take care of the case where master is switched from the original unit, which is not registered in CDev list.
        We need to guess it from the address range.
        """
        master = self.get_top().device
        devs = self.get_top().device['devs'].values()
        dev_with_virtual_address = filter_first(lambda dev: dev['host'] == master['host'], devs)
        if dev_with_virtual_address:
            '''
            Change virtual address to local address. It should be done by the user.
            Code here is just for robustness.
            '''
            missing_local_ip = self.find_missing_local_ip()
            if missing_local_ip:
                dev_with_virtual_address['host'] = missing_local_ip
        result = filter(lambda dev: dev['host'] not in (master['host'], self.get_man_intf_local_ip()), devs)
        return result

    def find_missing_local_ip(self):
        '''
        @return the local IP address of a unit that is not registered to APIC when cluster is enabled.
        This is most likely that of the original master unit, the first one in the management interface address-pool.
        '''
        if not self.is_cluster_formed_on_device():
            return
        address_pool = self.get_man_addresses()
        devs = self.get_top().device['devs'].values()
        result = None
        for address in address_pool:
            if not filter_first(lambda dev: dev['host'] == address, devs):
                result = address
                break
        if not result:
            return
        tmp_dev = copy(self.get_top().device)
        tmp_dev['host'] = address
        response, error = query_asa(tmp_dev, 'show cluster info | grep ^Cluster .*: On')
        if response:
            return result

    def get_man_addresses(self):
        '''
        @return list of management local IP addresses of the cluster units
        '''
        man_address_pool = self.get_child(ManAddressPool.__name__).make_address_range()
        token = man_address_pool.split('-')
        start = int(token[0].split('.')[-1])
        end = int(token[1].split('.')[-1])
        prefix = '.'.join(token[0].split('.')[: -1]) + '.'
        result = []
        for oct in range(start, end+1):
            result.append(prefix + str(oct))
        return result

    def get_master_dev(self):
        """
        @return the CDev of the master unit
        """
        master = self.get_top().device
        devs = self.get_top().device['devs'].values()
        return filter_first(lambda dev: dev['host'] in (master['host'], self.get_man_intf_local_ip()), devs)

    def turn_on_ctrl_intf(self, asa_cfg_list):
        '''Generate CLI to turn on cluster control interface
        '''
        ctrl_intf = self.get_child(Bootstrap.__name__).get_value()['ctrl_intf']
        mode = 'interface ' + ctrl_intf
        result = self.query_asa('show run %s | grep shutdown' % mode, 'system' if self.is_system_context else None)
        if result and result.strip() == 'shutdown':
            self.generate_cli(asa_cfg_list, 'no shutdown', mode_command=mode)

    def disable_cluster(self, no_asa_cfg_stack):
        '''
        We can disable cluster units from the master, using 'cluster remove unit <unit-name>' exec command
        '''
        mode = 'cluster group ' + self.get_cluster_group_name()
        self.generate_cli(no_asa_cfg_stack, 'no enable', mode_command=mode)
        slave_units = self.get_slave_dev_list()
        for dev in slave_units:
            unit_name =  str(dev['dn']).split('/')[-1].split('-')[-1]
            cli = 'cluster remove unit ' + unit_name
            self.generate_cli(no_asa_cfg_stack, cli)

    def enable_cluster(self, asa_cfg_list):
        '''
        Enable cluster on the master unit.
        '''
        mode = 'cluster group ' + self.get_cluster_group_name()
        self.generate_cli(asa_cfg_list, 'enable noconfirm', mode_command=mode)

    def get_netmask(self):
        '''
        @return the network mask of the management interface IP
        '''
        master = self.get_top().device
        netmask = master.get('man_intf_netmask')
        if not netmask:
            self.get_man_intf_info()
        netmask = master.get('man_intf_netmask', '255.255.255.0')
        return netmask

    def get_man_intf_name(self):
        '''
        @return the name of management interface, such as 'Management0/0'
        '''
        master = self.get_top().device
        intf = master.get('man_intf_name')
        if not intf:
            self.get_man_intf_info()
        intf = master.get('man_intf_name', 'Management0/0')
        return intf

    def get_system_context_man_intf_name(self):
        master = self.get_top().device
        intf = master.get('system_context_man_intf_name')
        if not intf:
            self.get_man_intf_info()
        intf = master.get('system_context_man_intf_name', 'Management0/0')
        return intf

    def get_man_intf_local_ip(self):
        '''
        @return the local IP address on the master unit
        '''
        master = self.get_top().device
        ip = master.get('man_intf_local_ip')
        if not ip:
            self.get_man_intf_info()
        ip = master.get('man_intf_local_ip')
        return ip

    def get_man_intf_info(self):
        """
        It probes the ASA and saves the following information in the device dictionary:
          'man_intf_name': the name of the management interface , such as 'Management0/0'
          'man_intf_netmask':       the netmask of the management interface IP address
          'man_intf_local_ip':      the local IP address (from IP-POOL), in case of a formed cluster

        The format of config looks like, for ASA which is not a cluster master:
            Current IP Address:
            Interface                Name                   IP address      Subnet mask     Method
            Management0/0            management             172.23.204.243  255.255.255.0   CONFIG

        For cluster master, there is an addtional entry for cluster address.

        For ASA 9.1(3), the format of config for a cluster master looks like :
            Current IP Address:
            Interface                Name                   IP address      Subnet mask     Method
            Management0/0            management             172.23.204.243  255.255.255.0   CONFIG
                                                            172.23.204.241  255.255.255.0
         For ASA 9.5(1), the format of config for a cluster master looks like:
            Current IP Address:
            Interface                Name                   IP address      Subnet mask     Method
            Management0/0            management             172.23.204.243  255.255.255.0   IP-POOL
                                     management             172.23.204.241  255.255.255.0   VIRTUAL

        The line to look at is line 3
        """
        config = self.query_asa("show ip address management | begin Current IP Address:")
        if not config:
            return
        config =  config.strip().split('\n')
        if len(config) < 3:
            return
        token = config[2].split() # the device address
        if len(token) < 4:
            return
        device = self.get_top().device
        device['man_intf_name'] = token[0]
        device['man_intf_local_ip'] = token[2]
        device['man_intf_netmask'] = token[3]
        if self.is_system_context:
            '''Get the name of the management interface in the system context.
            It is most likely to be the same as it is in the user context.
            '''
            device['system_context_man_intf_name'] = device['man_intf_name']
            context = self.get_top().get_asa_context_name()
            allocate_man_intf = self.query_asa('show run context %s | grep %s' % (context, device['man_intf_name']), 'system').strip()
            '''
            The value of allocate_man_intf, as the output of 'show run context <ctxt_name> | grep <intf_name>',  is of the format:
              allocate-interface <system_context_intf_name> [<user_context_intf_name>|visible|invisible]
            we take care of the case where the mapping is specified in which case the name of the interface in the system context differs from
            that in the user context.
            '''
            if allocate_man_intf:
                tokens = allocate_man_intf.split(' ')
                if len(tokens) > 2 and tokens[2] == device['man_intf_name']:
                    device['system_context_man_intf_name'] = tokens[1]

    def move_no_cluster_interface_mode_cmd(self, no_asa_cfg_stack):
        '''
        Make sure 'no cluster interface-mode command is issued right after 'clear configure cluster'
        '''
        clear_cluster_config_cmd = filter_first(lambda cmd: cmd.command == 'clear configure cluster',  no_asa_cfg_stack)
        if not clear_cluster_config_cmd:
            return
        index = no_asa_cfg_stack.index(clear_cluster_config_cmd)
        no_cluster_interface_mode_cmd = no_asa_cfg_stack[0]
        no_asa_cfg_stack.insert(index, no_cluster_interface_mode_cmd)
        no_asa_cfg_stack.pop(0)

    def update_references(self):
        '''
        For slave unit, only generate management address configuration on deleting the cluster
        '''
        '''
        @todo taking care of audit
        '''
        if not self.is_apic_managed():
            return
        DMObject.update_references(self)
        if self.device and self.get_state() != State.DESTROY:
            self.unregister_child(self.get_child(ManInterface.__name__))
            self.unregister_child(self.get_child(ManAddressPool.__name__))
        for child in self.children.values():
            child.response_parser = cluster_response_parser

    def get_translator(self, cli):
        if not self.is_apic_managed():
            return
        return DMObject.get_translator(self, cli)

    def need2disable_cluster(self):
        '''
        We need to disable clustering if one of the following cluser group sub-commands are modified:
        - clacp
        - cluster-interface
        - director-localization
        - local-unit
        - priority
        - site-id
        - site-redundancy
        or the global command 'cluster inteface-mode'
        '''
        for child in (Bootstrap.__name__, 'interface_mode'):
            if self.get_child(child).need2disable_cluster():
                return True
        #@todo need to take care of the advanced parameter as well.

    def get_cluster_group_name(self):
        'Use the instance string of the ClusterConfig folder as the cluster group name on ASA'
        return  self.delta_ifc_key[2]

    def set_cluster_group_name(self, name):
        self.parent.delta_ifc_cfg_value['value'].pop(self.delta_ifc_key)
        self.delta_ifc_key = (self.delta_ifc_key[0], self.delta_ifc_key[1], name)
        self.parent.delta_ifc_cfg_value['value'][self.delta_ifc_key] = self.delta_ifc_cfg_value

    def set_cluster_state(self, state):
        set_cfg_state(self.delta_ifc_cfg_value, state)

    def validate_configuration(self):
        '''
        Following are not allowed:
        1. In multi-context mode, the target context is not admin context
        2. Less than 2 CDevs
        3. Management IP addresses not contiguous
        4. Data interfaces configure before setting up cluster or before removing cluster.
        5. The cluster unit not of the same model, or same OS version, or not the same mode.
        6. ASAv
        ....
        '''
        if not self.is_apic_managed() or not self.has_ifc_delta_cfg():
            return
        state = self.get_state()
        if state == None or state == State.NOCHANGE:
            return
        result = DMObject.validate_configuration(self)
        if not self.validate_asav(result):
            return result
        devs = self.get_top().device['devs'].values()
        if len(devs) < 2:
            error = ("You must have at least two CDevs to setup ASA cluster.")
            self.report_fault(error, result)
        else:
            self.validate_cdev_addresses(result)
        self.validate_data_interfaces(result)
        self.vallidate_port_channel_span_cluster(result)
        self.validate_homogenouse(result)
        return result

    def validate_cdev_addresses(self, faults):
        '''
        Check if the CDev addresses satisfy ASA cluster pre-requisite:
        If LDev's address is x, and number of cdevs is n, then each CDev address must be either
        x, or within x and x+n inclusively. Notice that x+1 will be the local IP address of the
        master unit once cluster is formed.
        For example, with LDev address being 172.1.1.1, and three CDevs, the IP address of each 
        CDev must be 172.1.1.1 to 172.1.1.4.
        Also make sure we have the CDev for the master unit, it is found in CSCvm60471, APIC could
        calls us without it.
        '''
        if not self.get_master_dev():
            error = 'CDev for the master unit is missing.'
            self.report_fault(error, faults)
            return
        virtual_address_octet = self.get_top().device['host'].split('.')
        start = int(virtual_address_octet[-1])
        n = len(self.get_top().device['devs'])
        address_range = '%s-%s.%s' % (self.get_top().device['host'], '.'.join(virtual_address_octet[:3]), str(start+n))
        for name, cdev in self.get_top().device['devs'].iteritems():
            address_octet = cdev['host'].split('.')
            if (address_octet[:3] != virtual_address_octet[:3] or
                int(address_octet[-1]) < start or
                int(address_octet[-1]) > start + n):
                error = 'IP address of any CDev must be in the range of %s to setup ASA cluster. '\
                        'The IP address, \'%s\', of \'%s\' fails to meet this requirement.' % \
                        (address_range, cdev['host'], name)
                self.report_fault(error, faults)
        if self.is_cluster_formed_on_device():
            return
        '''
        Make sure x+1 is not used by any CDev before the cluster is formed, where x is the LDev's address.
        It will be used as the local address of the master unit upon formation of cluster.
        '''
        master_local_address = '.'.join(virtual_address_octet[:3]) + '.' + str(start+1)
        if filter_first(lambda cdev: cdev['host'] == master_local_address, self.get_top().device['devs'].values()):
            error = '%s cannot be used as the management IP address of a slave unit. '\
                    'It will be used as the local management IP address of the master unit once cluster is formed.' % \
                    (master_local_address)
            self.report_fault(error, faults)

    def validate_data_interfaces(self, faults):
        '''
        Check to see if there is any data interfaces configured before configuring cluster interface-mode.
        This is not allowed on ASA. Raise a fault in this case.
        '''
        cluster_intf_mode = self.get_child('interface_mode')
        if cluster_intf_mode.get_state() == State.NOCHANGE:
            return
        intf_list = self.get_configured_data_intfs()
        if intf_list:
            intf_names = ', '.join(intf_list)
            error = 'Please remove all data interface configuration and all service graphs from each CDev before proceed to configure or remove cluster.' 
            if len(intf_list) == 1:
                error += ' The following interface is configured: %s.' % intf_names
            else:
                error += ' The following interfaces are configured: %s.' % intf_names
            self.report_fault(error, faults)

    def get_configured_data_intfs(self):
        '''
        @return a list of data interfaces that are configured.
        '''
        query_intf = 'show run interface'
        clis = self.query_asa(query_intf)
        '''If the target device is multi-context, then going through each context.
        Only sample the master unit for now.
        @todo: go through all slave units as well.
        '''
        if self.get_top().is_multi_mode_asa():
            if not clis:
                clis = ''
            contexts = self.query_asa('show run context | grep ^context','system')
            if contexts:
                contexts = contexts.strip().split('\n')
                contexts = map(lambda c: c.split()[-1], contexts)
            else:
                contexts = []
            for c in contexts:
                if self.get_top().get_asa_context_name() == c:
                    continue
                tmp_clis = self.query_asa(query_intf, c)
                if tmp_clis:
                        clis += '\n' + tmp_clis
        if not self.is_cluster_formed_on_device():
            '''If cluster is not configured on the device, check all the slave units as well
            '''
            slave_units = self.get_slave_dev_list()
            if not clis:
                clis = ''
            for dev in slave_units:
                tmp_clis = query_asa(dev, query_intf)[0]
                if tmp_clis:
                    clis += '\n' + tmp_clis
        if not clis:
            return
        def has_nameif_or_ip_address_data_intf(cmd):
            '''
            @return True if cmd is a data interface command and has nameif or ip address sub-command configured.
            '''
            if isinstance(cmd, basestring):
                return
            for sub_cmd in filter(lambda sc: isinstance(sc, basestring), cmd.sub_commands):
                if sub_cmd =='nameif management':
                    return False
                if re.match('^nameif |^ip address |^ipv6 address ', sub_cmd):
                    return True
        clis = read_clis(clis.split('\n'))
        result = filter(lambda cmd: has_nameif_or_ip_address_data_intf(cmd), clis)
        result = map(lambda cmd: cmd.command.split(' ')[-1], result)
        return result

    def vallidate_port_channel_span_cluster(self, faults):
        '''
        Raise fault if 'port-channel span-cluster' is configured when trying to change cluster
        interface-mode to individual, or even remove cluster interface-moode
        '''
        cluster_intf_mode = self.get_child('interface_mode')
        if cluster_intf_mode.get_state() == State.NOCHANGE:
            return
        response = self.query_asa('show run interface | grep port-channel span-cluster',
                                  'system' if self.get_top().is_multi_mode_asa() else None)
        if not response:
            return
        if cluster_intf_mode.get_state() == State.DESTROY:
            if self.is_port_channel_span_cluster_not_all_destroyed():
                error = 'You must remove port-channel span-cluster option before removing the cluster configuration.'
                self.report_fault(error, faults)
        elif cluster_intf_mode.get_state() in (State.MODIFY, State.CREATE):
            error = 'You must remove port-channel span-cluster option before changing the cluster interface-mode.'
            self.report_fault(error, faults)

    def is_port_channel_span_cluster_not_all_destroyed(self):
        '''
        @return True if this callout will remove all 'port-channel span-cluster' options
        '''
        port_channel_list = self.get_top().get_child('LACPMaxBundle')
        for pc in port_channel_list:
            span_cluster = pc.get_child('SpanCluster')
            if span_cluster and span_cluster.get_state() != State.DESTROY:
                return True

    def is_cluster_formed_on_device(self):
        '''
        @return True if cluster is configure on the device
        '''
        if not hasattr(self, 'cluster_on'):
            response = self.query_asa('show cluster info | grep ^Cluster .*: On', 'system'
                                      if self.get_top().is_multi_mode_asa() else None)
            self.cluster_on = response != None and response.strip() != ''
        return self.cluster_on

    def validate_asav(self, faults):
        '''
        @return True if targeted device is physical ASA, False if ASAv
        '''
        if self.get_top().is_virtual():
            error = 'Clustering with ASAv is not supported.'
            self.report_fault(error, faults)
            return
        return True

    def validate_homogenouse(self, faults):
        '''
        @return True if all cluster units are of the same type
        '''
        def get_dev_info(dev):
            '''
            @return a dictionary with the  following information of the ASA device
            model, version, firewall_mode, context_mode
            '''
            result = {}
            response, error = query_asa(dev, 'show version | grep ^Hardware:')
            if response:
                result['model'] = response.split(':')[1].split(',')[0].strip()
            response, error = query_asa(dev, 'show version | grep ^Cisco Adaptive Security Appliance Software Version')
            if response:
                result['version'] = response.split()[6]
            response, error = query_asa(dev, 'show mode')
            if response:
                result['context_mode'] = response.split(': ')[1].strip()
            response, error = query_asa(dev, 'show firewall')
            if response:
                result['firewall_mode'] = response.split(': ')[1].strip()
            return result
        ldev = self.get_top().device
        ldev_info= get_dev_info(ldev)
        n = len(faults)
        for dev, access_info in self.get_top().device['devs'].items():
            if access_info['host'] == ldev['host']:
                continue
            info = get_dev_info(access_info)
            for t, d in (('model', 'model'), ('version', 'OS version'), ('context_mode', 'context mode'), ('firewall_mode', 'firewall mode'),):
                if info.get(t) and info[t] != ldev_info[t]:
                    error = 'Cluster units must be of the same %s for clustering. '\
                    'The %s of master unit is %s, and the %s of "%s" is %s.' % (d, d, ldev_info[t], d, dev, info[t])
                    self.report_fault(error, faults)
        return len(faults) == n

class InterfaceMode(SimpleType):
    '''
    Model after 'cluster interface-mode'
    '''
    def __init__(self, device):
        SimpleType.__init__(self, ifc_key = 'interface_mode',  asa_key = 'cluster interface-mode')
        self.device = device

    def update_references(self):
        '''
        The APIC configuration is under Bootstrap, copy it over.
        '''
        if self.has_ifc_delta_cfg():
            return
        bootstrap = self.parent.get_child(Bootstrap.__name__)
        if not bootstrap.has_ifc_delta_cfg():
            return
        bootstrap_config = bootstrap.delta_ifc_cfg_value
        key = filter_first(lambda key: key[1] == self.ifc_key, bootstrap_config['value'].keys())
        if not key:
            return
        value = bootstrap_config['value'][key]
        self.populate_model(key, value)
        self.parent.delta_ifc_cfg_value['value'][key] = value

    def get_cli(self):
        if self.get_state() in (State.CREATE, State.MODIFY):
            return SimpleType.get_cli(self) + ' force'
        if self.get_state() == State.DESTROY:
            return self.asa_key
        return SimpleType.get_cli(self)

    def need2disable_cluster(self):
        return self.get_state() == State.MODIFY

class ManAddressPool(IPv4AddressPool):
    '''
    Model after the address pool for management interface
      ip local pool __$MAN_ADDRESS_POOL_IPV4$_ ....
    '''
    def __init__(self, device):
        IPv4AddressPool.__init__(self, ManAddressPool.__name__)
        self.device = device

    def update_references(self):
        '''Override the default implementation to populate data implicitly
        '''
        if self.has_ifc_delta_cfg():
            return
        state = self.parent.get_child(Bootstrap.__name__).get_state()
        if state in (State.NOCHANGE, State.MODIFY):
            return
        key = (Type.PARAM, ManAddressPool.__name__, MANAGEMENT_ADDRESS_POOL_NAME)
        value = {'address_range': self.make_address_range(), 'mask': self.parent.get_netmask()}
        value = {'state': state, 'value': ifcize_param_dict(value)}
        set_cfg_state(value, state)
        self.populate_model(key, value)
        self.parent.delta_ifc_cfg_value['value'][key] = value

    def make_address_range(self):
        '''
        @return the address range in the form of x.x.x.a-x.x.x.b,
        where a = the last octet of the master address + 1,
        and b = a + the number of cluster units
        '''
        master_address = self.get_top().device['host']
        unit_count = len(self.get_top().device['devs'])
        octet = master_address.split('.')
        last_oct = int(octet[-1])
        start_address = deepcopy(octet)
        start_address[-1] = str(last_oct + 1)
        end_address = deepcopy(octet)
        end_address[-1] = str(last_oct + unit_count)
        return '.'.join(start_address) + '-' + '.'.join(end_address)

    def is_my_cli(self, cli):
        if isinstance(cli, basestring) and str(cli).startswith(self.asa_key):
            return IPv4AddressPool.is_my_cli(self, cli)

class ManInterface(SimpleType):
    def __init__(self, device):
        SimpleType.__init__(self, ManInterface.__name__,
                            asa_gen_template='ip address %(address)s %(mask)s cluster-pool ' + MANAGEMENT_ADDRESS_POOL_NAME )
        self.device = device

    def update_references(self):
        '''Override the default implementation to populate data implicitly
        '''
        state = self.parent.get_child(Bootstrap.__name__).get_state()
        if state in (State.NOCHANGE, State.MODIFY):
            return
        device = self.device
        if not device:
            device = self.get_top().device
        key = (Type.PARAM, ManInterface.__name__, '')
        value = {'address': device['host'], 'mask': self.parent.get_netmask()}
        value = {'state': state, 'value': ifcize_param_dict(value)}
        set_cfg_state(value, state)
        self.delta_ifc_key = key
        self.delta_ifc_cfg_value = value
        self.parent.delta_ifc_cfg_value['value'][key] = value
        self.mode_command = 'interface ' + self.parent.get_man_intf_name()

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        '''Override the default so on destroy, on the removal of the cluster,
        we will restore the IP address without the cluster-pool option. And the commands issued
        are like:
             interface Management0/0
                no ip address
                ip address 1.2.3.4 255.255.255.0
                no shutdown
        We have to issue 'no ip address' and 'no shutdown' for it to work on slave unit.
        '''
        if not self.has_ifc_delta_cfg():
            return
        if self.get_state() != State.DESTROY:
            return SimpleType.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)
        for cli in ['no shutdown', self.get_cli(), 'no ip address']:
            self.generate_cli(no_asa_cfg_stack, cli)

    def get_cli(self):
        if self.get_state() == State.DESTROY:
            return 'ip address %(address)s %(mask)s' % self.get_value()
        return SimpleType.get_cli(self)

class Bootstrap(CompositeType):
    '''
    This class represents the cluster bootstrap configuration.
    '''

    def __init__(self, device):
        CompositeType.__init__(self, ifc_key = Bootstrap.__name__,
                               asa_key = 'cluster group',
                               asa_gen_template = "cluster group %(group_name)s",
                               response_parser = cluster_response_parser)
        #hidden fields
        self.register_child(Priority())
        self.register_child(LocalUnit())
        #public fields
        self.register_child(SecretKey())
        self.register_child(ClusterInterface())
        self.register_child(EnableCluster()) #this is hidden too
        self.device = device
        for child in self.children.values():
            child.device = device

    def get_state(self):
        return self.delta_ifc_cfg_value['state'] if hasattr(self, 'delta_ifc_cfg_value') else None

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        '''Override the default implementation to parameters implicitly setup by the device pacakge
        - use the instance name of the 'ClusterConfig' folder as ASA cluster group name
        - use priority 1 for the master unit on ASA
        - use priority 2 for any slave unit on ASA
        - use the CDev name as the value of local-unit on ASA
        - 'enable' command is not exposed to user
        '''
        self.delta_ifc_key = delta_ifc_key
        self.delta_ifc_cfg_value = delta_ifc_cfg_value
        if not self.parent.is_apic_managed():
            return
        if not delta_ifc_cfg_value or not delta_ifc_cfg_value['value']:
            return
        self.set_internal_param('group_name', self.get_cluster_group_name())
        if self.get_state() != State.DESTROY:
            self.set_internal_param('priority',  2 if self.device else 1)
            self.set_internal_param('local_unit', self.get_cdev_name())
            self.set_internal_param('enable', 'enable')
            self.set_control_intf_folder(delta_ifc_cfg_value['value'])
        CompositeType.populate_model(self, delta_ifc_key, delta_ifc_cfg_value)

    def set_internal_param(self, name, value):
        state = self.get_state()
        if state == State.MODIFY:
            state = State.NOCHANGE #we only create and destroy internal parameters, never changes them.
        self.delta_ifc_cfg_value['value'][(Type.PARAM, name, '')] = {'state': state, 'value': value} 

    def set_control_intf_folder(self, value):
        '''
        This sub-folder will contain only parameters 'ctrl_intf' and 'ctrl_intf_address'
        '''
        if self.delta_ifc_cfg_value['value'].get((Type.FOLDER, ClusterInterface.__name__, '')):
            return
        intf_key = filter_first(lambda key: key[1] == 'ctrl_intf', value.keys())
        address_key = filter_first(lambda key: key[1] == 'ctrl_intf_address', value.keys())
        state = value[address_key]['state']
        if state == State.NOCHANGE:
            state = value[intf_key]['state']
        value = {intf_key: value[intf_key], address_key: value[address_key]}
        self.delta_ifc_cfg_value['value'][(Type.FOLDER, ClusterInterface.__name__, '')] = {'state': state, 'value': value}

    def get_cluster_group_name(self):
        return  self.parent.get_cluster_group_name()

    def set_cluster_group_name(self, name):
        return  self.parent.set_cluster_group_name(name)

    def set_cluster_state(self, state):
        self.parent.set_cluster_state(state)

    def get_cdev_name(self):
        dev = self.device
        if not dev: #master device
            dev = self.parent.get_master_dev()
            if not dev:
                '''Due to APIC mis-behavior, the master CDev can be missing from device dictionary in clusterAudit
                when first registering the device. See CSCvm60471.
                '''
                return ''
        ''' Figure it out from the 'dn'  attribute
            'dn': u'uni/tn-dahai/lDevVip-Firewall/cDev-unit1'
        '''
        return str(dev['dn']).split('/')[-1].split('-')[-1]

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        '''
        Override the default implementation to make sure cluster is disabled before removing it,
        otherwise the cluster cannot be removed on slave unit.
        '''
        if self.has_ifc_delta_cfg() and self.get_state() == State.DESTROY:
            self.generate_cli(no_asa_cfg_stack, 'clear configure cluster')
            self.generate_cli(no_asa_cfg_stack, 'no enable', mode_command=self.get_cli())
        else:
            return CompositeType.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)

    def need2disable_cluster(self):
        '''
        We need to disable clustering if one of the following parameter is modified:
        - cluster-interface
        '''
        if self.get_child(ClusterInterface.__name__).need2disable_cluster():
            return True

    def diff_ifc_asa(self, cli):
        '''
        Override the default so that we can get the cluster group name during audit
        '''
        result = CompositeType.diff_ifc_asa(self, cli)
        if self.get_state() == State.DESTROY:
            group_name = self.get_value().get('group_name')
            self.set_cluster_group_name(group_name)
            self.set_cluster_state(State.DESTROY)
        return result

    def has_ifc_delta_cfg(self):
        if not CompositeType.has_ifc_delta_cfg(self):
            return
        return self.delta_ifc_cfg_value.get('value')

    def is_ctrl_intf_changed(self):
        ctrl_intf_key = filter_first(lambda key: key[1] == 'ctrl_intf', self.delta_ifc_cfg_value['value'].keys())
        if ctrl_intf_key:
            return self.delta_ifc_cfg_value['value'][ctrl_intf_key]['state'] in (State.CREATE, State.MODIFY)

class Advanced(CompositeType):
    '''
    @todo to be implemented
    '''
    pass

class SecretKey(SimpleType):
    'Model after cluster secret key to authenticate units in cluster'
    def __init__(self):
        SimpleType.__init__(self, ifc_key = 'key',  asa_key = 'key')

class LocalUnit(SimpleType):
    'Model after cluster local-unit name'
    def __init__(self):
        SimpleType.__init__(self, ifc_key = 'local_unit',
                            asa_key = 'local-unit',
                            asa_gen_template = 'local-unit %s')


class Priority(SimpleType):
    'Model after cluster priority level to a cluser unit for master election'
    def __init__(self):
        SimpleType.__init__(self, ifc_key = 'priority',
                            asa_key = 'priority',
                            asa_gen_template = 'priority %s')

class ClusterInterface(SimpleType):
    '''
    This class represents the cluster control link interface configuration.
    '''
    def __init__(self):
        SimpleType.__init__(self, ifc_key = ClusterInterface.__name__,
                            asa_key = 'cluster-interface',
                            asa_gen_template='cluster-interface %(ctrl_intf)s ip %(address)s',
                            is_removable = False,
                            response_parser = cluster_response_parser)

    def get_value(self):
        '''Override the default to provide normalize address for ASA from ctrl_intf_address.
           Also adjust the address to make sure it is unique for each unit
        '''
        device = self.device
        if not device:
            device = self.get_top().device
        man_address = device['host'].split('.')
        result = SimpleType.get_value(self)
        address_mask = result.get('ctrl_intf_address').split('/')
        address = address_mask[0].split('.')
        netmask =  address_mask[1]
        address[-1] = man_address[-1]
        address = '.'.join(address)
        if '.' not in netmask: # Prefix length
            netmask = netmask_from_prefix_length(netmask)
        result['address'] = address + ' ' + netmask
        return result

    def is_the_same_cli(self, cli):
        '''only check the first three octets of the ip address to avoid un-necessary sync operation.
        This helps to avoid the device package from sending CLI during clusterAudit when there is
        a switch of master unit.
        The cli is is of the form:
           cluster-interface GigabitEthernet0/0 ip 2.1.1.201 255.255.255.0
        '''
        def get_intf_number(full_name):
            '''
            @param full_name: str. Interface name of the form: <name><number>
            @return the number part of the interface name
            '''
            m = re.match('([^\d]+)(.+)', full_name)
            if not m:
                return full_name
            return m.group(2)
        cli_on_asa = str(cli).split()
        cli_on_apic = self.get_cli().split()
        if get_intf_number(cli_on_asa[1]) != get_intf_number(cli_on_apic[1]):
            return
        return cli_on_asa[3].split('.')[:3] == cli_on_apic[3].split('.')[:3]

    def set_state(self, state):
        '''
        Override the default to use the state for all the parameters in the folder
        '''
        set_cfg_state(self.delta_ifc_cfg_value, state)

    def need2disable_cluster(self):
        '''
        @return True if 'ctrl_intf_address' or 'ctrl_intf' is modified.
        '''
        if not self.has_ifc_delta_cfg():
            return
        name_key = filter_first(lambda key: key[1] == 'ctrl_intf', self.delta_ifc_cfg_value['value'].keys())
        address_key = filter_first(lambda key: key[1] == 'ctrl_intf_address', self.delta_ifc_cfg_value['value'].keys())
        return ((self.delta_ifc_cfg_value['value'][name_key]['state'] == State.MODIFY) or
                (self.delta_ifc_cfg_value['value'][address_key]['state'] == State.MODIFY))

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        return SimpleType.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)

class EnableCluster(DMBoolean):
    'Model after enable or disable cluster'
    def __init__(self):
        DMBoolean.__init__(self, ifc_key = 'enable',
                            asa_key = 'enable',
                            on_value = 'enable',
                            response_parser=cluster_response_parser)

    def get_cli(self):
        if self.get_state() == State.DESTROY:
            return self.asa_key
        return 'enable ' + ('as-slave noconfirm' if self.device else 'noconfirm')

def cluster_response_parser(response):
    '''
    Ignores INFO, WARNING, and some expected errors in the response, otherwise returns original.
    '''

    if response:
        msgs_to_ignore = ('INFO:',
                          'WARNING:',
#                           'Interface does not have virtual MAC',
                          'No change to the stateful interface',
                          'Waiting for the earlier webvpn instance to terminate',
                          'Previous instance shut down',
                          'This unit is in syncing state',
                          'Configuration syncing is in progress',
                          'The requested mode is the SAME as the current mode',
#                           'Cluster interface-mode cannot be changed when cluster is enabled',
                          'Detected Cluster Master')
        found_msg_to_ignore = False
        for msg in msgs_to_ignore:
            if msg in response:
                found_msg_to_ignore = True
    return None if response and found_msg_to_ignore else response
