'''
Created on Mar 9, 2018

@author: dli

Copyright (c) 2018 by Cisco Systems

'''
import re

from translator.base.simpletype import SimpleType
from translator.base.dmlist import DMList
from translator.state_type import State
from asaio.cli_interaction import ignore_response_parser

IPV4_NAME_PREFIX = '__$ADDRESS_POOL_IPV4$_'
IPV6_NAME_PREFIX = '__$ADDRESS_POOL_IPV6$_'

MANAGEMENT_ADDRESS_POOL_NAME = '__$MAN_ADDRESS_POOL_IPV4$_'

class IPAddressPools(DMList):
    '''
    This object is used internally by interface ip address for 'individual' cluster interface mode., it is not exposed to the user.
    '''
    def __init__(self, name, child_class, asa_key):
        DMList.__init__(self, name, child_class, asa_key)

    def register_pool(self, delta_ifc_key, delta_ifc_cfg_value):
        '''Make sure the parent has this data hooked up as well. This is import for audit operation.
        '''
        if self.parent.delta_ifc_cfg_value['value'].get(delta_ifc_key):
            return
        DMList.populate_model(self, delta_ifc_key, delta_ifc_cfg_value)
        self.parent.delta_ifc_cfg_value['value'][delta_ifc_key] = delta_ifc_cfg_value

class IPv4AddressPools(IPAddressPools):
    '''
    This object is used internally by interface ip address for 'individual' cluster interface mode., it is not exposed to the user.
    '''
    def __init__(self):
        IPAddressPools.__init__(self, IPv4AddressPool.__name__, IPv4AddressPool,
                                re.compile('^ip local pool ' + IPV4_NAME_PREFIX.replace('$', '\$') + '\S+ '))

class IPv6AddressPools(IPAddressPools):
    '''
    This object is used internally by interface ipv6 address for 'individual' cluster interface mode., it is not exposed to the user.
    '''
    def __init__(self):
        IPAddressPools.__init__(self, IPv6AddressPool.__name__, IPv6AddressPool,
                                re.compile('^ipv6 local pool ' + IPV6_NAME_PREFIX.replace('$', '\$') + '\S+ '))
        self.mini_audit_command = 'show run ipv6 | grep ^ipv6 local pool'

class IPAddressPool(SimpleType):
    '''
    Model CLI:
       ip local pool <name> <address_range> mask <mask>
       ipv6 local pool <name> <address_range> <number>
    '''
    def __init__(self, name, asa_key, asa_gen_template, address_type):
        SimpleType.__init__(self, name, asa_key, asa_gen_template)
        self.address_type = address_type

    def get_value(self):
        '''Override the default implemenation using the instance id of its dictionary key as the pool name.
        '''
        result = SimpleType.get_value(self)
        result['name'] = self.delta_ifc_key[2]
        return result

    def get_cli(self):
        if self.get_state() == State.DESTROY:
            return self.asa_key + ' %(name)s' % self.get_value()
        return SimpleType.get_cli(self)

    def create_delta_ifc_key(self, cli):
        '''Override the default to use address-pool name as the instance key
        '''
        result =  SimpleType.create_delta_ifc_key(self, cli)
        return (result[0], result[1], cli.split(' ')[3])

    def is_my_cli(self, cli):
        '''check the name of the address pool against ifc_key
        '''
        if isinstance(cli, basestring):
            return str(cli).split(' ')[3] == self.ifc_key


class IPv4AddressPool(IPAddressPool):
    '''
    Model CLI:
       ip local pool <name> <address_range> mask <mask>'
    '''
    def __init__(self, name):
        IPAddressPool.__init__(self, name, 'ip local pool',
                            'ip local pool %(name)s %(address_range)s mask %(mask)s','IPv4Address')

    def get_config_path(self):
        '''Redirect the config path to that of the interface IP address
        '''
        nameif = self.ifc_key[len(IPV4_NAME_PREFIX):]
        try:
            intf = self.get_top().get_child('Interface').get_child(nameif)
            return intf.get_child('InterfaceConfig').get_child(self.address_type).get_config_path()
        except AttributeError:
            return self.get_top().get_config_path()

    @staticmethod
    def make_name(nameif):
        return IPV4_NAME_PREFIX + nameif

class IPv6AddressPool(IPAddressPool):
    '''
    Model CLI:
       ipv6 local pool <name> <address_range> <number>'
    '''
    def __init__(self, name):
        IPAddressPool.__init__(self, name, 'ipv6 local pool',
                            'ipv6 local pool %(name)s %(address)s %(number)s','IPv6Address')

    def get_config_path(self):
        '''Redirect the config path to that of the interface IP address
        self.ifc_key is in the form of __$ADDRESS_POOL_IPV6$_outside#1:1
        '''
        nameif = self.ifc_key[len(IPV6_NAME_PREFIX):self.ifc_key.index('#')]
        try:
            intf = self.get_top().get_child('Interface').get_child(nameif)
            return intf.get_child('InterfaceConfig').get_child(self.address_type).get_config_path()
        except AttributeError:
            return self.get_top().get_config_path()

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        '''
        Override the default implementation so that in the case of modification, we delete the old one first.
        ASA insists on it even though for IPv4, you can simply overwrite.
        '''
        if not self.has_ifc_delta_cfg():
            return
        if self.get_state() == State.MODIFY:
            self.generate_cli(no_asa_cfg_stack,
                              'no ipv6 local pool %(name)s' % self.get_value(),
                              response_parser = ignore_response_parser)
        return IPAddressPool.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)

    @staticmethod
    def make_name(nameif, address):
        '''
        @param address is of the format '1::1/32'
        @return a unique address pool name based on the IP address, e.g. 1::1.
        Example: __$ADDRESS_POOL_IPV6$_outside#1:1
        '''
        return IPV6_NAME_PREFIX + nameif + '#' +  address.split('/')[0]
